/*
 * Decompiled with CFR 0.152.
 */
package net.algart.model3d.spherepolyhedra.drawing;

import java.awt.Color;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import net.algart.arrays.JArrays;
import net.algart.arrays.Matrix;
import net.algart.arrays.PArray;
import net.algart.math.Range;
import net.algart.math.RectangularArea;
import net.algart.math.geometry.Orthonormal3DBasis;
import net.algart.model3d.spherepolyhedra.drawing.DrawingColor;
import net.algart.model3d.spherepolyhedra.objects.GeneratrixSet;
import net.algart.model3d.spherepolyhedra.objects.SpherePolyhedra;
import net.algart.model3d.spherepolyhedra.objects.SpherePolyhedron;
import net.algart.model3d.spherepolyhedra.objects.SpherePolyhedronColorSelector;

public final class SpherePolyhedraDrawer {
    public static final int DEFAULT_OPTIMIZATION_BLOCK_SIZE_LOGARITHM = 2;
    private static final System.Logger LOG = System.getLogger(SpherePolyhedraDrawer.class.getName());
    private static final int MAX_BLOCK_LOG = 10;
    private static final double STEP_ALONG_CIRCLE_WHILE_DRAWING_CYLINDER = 0.6;
    private final int dimX;
    private final int dimY;
    private final int dimXY;
    private final byte[] r;
    private final byte[] g;
    private final byte[] b;
    private final double[] zBuffer;
    private final double[] minZ;
    private final double[] maxZ;
    private final int blockLog;
    private final int blockSize;
    private final int nBlocksX;
    private final int nBlocksY;
    private final double[] zBufferMinInBlocks;
    private final int[] polygonLeftX;
    private final int[] polygonRightX;
    private final double[] polygonLeftZ;
    private final double[] polygonRightZ;
    private DrawingColor color = DrawingColor.of(0.0, 1.0, 1.0);
    private DrawingColor alternateColor = DrawingColor.of(1.0, 1.0, 1.0);
    private DrawingColor backgroundColor;
    private SpherePolyhedronColorSelector colorSelector = null;
    private double sectionZ = Double.POSITIVE_INFINITY;
    private boolean dissectedObjectsVisible = true;
    private boolean edgesOnly = false;

    private SpherePolyhedraDrawer(int dimX, int dimY, int blockLog, DrawingColor backgroundColor) {
        Objects.requireNonNull(backgroundColor, "Null backgroundColor");
        this.blockLog = Math.min(blockLog, 10);
        this.blockSize = 1 << blockLog;
        if (dimX <= 0 || dimY <= 0) {
            throw new IllegalArgumentException("Zero or negative dimX and dimY: " + dimX + "x" + dimY);
        }
        if (((long)dimX + (long)this.blockSize) * ((long)dimY + (long)this.blockSize) > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Too large dimX, dimY arguments: " + dimX + "x" + dimY);
        }
        if (blockLog < 0) {
            throw new IllegalArgumentException("Negative binary logarithm of block size, used for optimization");
        }
        this.dimX = dimX;
        this.dimY = dimY;
        this.dimXY = dimX * dimY;
        this.nBlocksX = dimX + this.blockSize - 1 >>> blockLog;
        this.nBlocksY = dimY + this.blockSize - 1 >>> blockLog;
        this.r = new byte[dimX * dimY];
        this.g = new byte[this.r.length];
        this.b = new byte[this.r.length];
        this.zBuffer = new double[this.r.length];
        this.minZ = new double[this.r.length];
        this.maxZ = new double[this.r.length];
        this.zBufferMinInBlocks = blockLog == 0 ? null : new double[this.nBlocksX * this.nBlocksY];
        this.polygonLeftX = new int[dimY];
        this.polygonRightX = new int[dimY];
        this.polygonLeftZ = new double[dimY];
        this.polygonRightZ = new double[dimY];
        this.backgroundColor = backgroundColor;
        this.clear();
    }

    public static SpherePolyhedraDrawer newSimpleInstance(int dimX, int dimY) {
        return SpherePolyhedraDrawer.newSimpleInstance(dimX, dimY, DrawingColor.BLACK);
    }

    public static SpherePolyhedraDrawer newSimpleInstance(int dimX, int dimY, DrawingColor backgroundColor) {
        return new SpherePolyhedraDrawer(dimX, dimY, 0, backgroundColor);
    }

    public static SpherePolyhedraDrawer newOptimizedInstance(int dimX, int dimY, DrawingColor backgroundColor) {
        return SpherePolyhedraDrawer.newOptimizedInstance(dimX, dimY, 2, backgroundColor);
    }

    public static SpherePolyhedraDrawer newOptimizedInstance(int dimX, int dimY, int optimizationBlockSizeLogarithm, DrawingColor backgroundColor) {
        return new SpherePolyhedraDrawer(dimX, dimY, optimizationBlockSizeLogarithm, backgroundColor);
    }

    public static SpherePolyhedron newSpherePolyhedron(double x0, double y0, Orthonormal3DBasis basis, double scale, double centerX, double centerY, double centerZ, GeneratrixSet generatrixSet, long kindId) {
        return SpherePolyhedron.newInstance(x0 + scale * basis.x(centerX, centerY, centerZ), y0 + scale * basis.y(centerX, centerY, centerZ), scale * basis.z(centerX, centerY, centerZ), generatrixSet.rotateAndScale(basis, scale), kindId);
    }

    public static Range estimateDrawnZRange(Orthonormal3DBasis basis, double scale, SpherePolyhedra spherePolyhedra) {
        RectangularArea parallelepiped = spherePolyhedra.containingAllParallelepiped(basis, scale);
        if (parallelepiped == null) {
            return Range.valueOf((double)0.0, (double)0.0);
        }
        return parallelepiped.range(2);
    }

    public int dimX() {
        return this.dimX;
    }

    public int dimY() {
        return this.dimY;
    }

    public DrawingColor getColor() {
        return this.color;
    }

    public void setColor(DrawingColor color) {
        this.color = Objects.requireNonNull(color);
    }

    public DrawingColor getAlternateColor() {
        return this.alternateColor;
    }

    public void setAlternateColor(DrawingColor alternateColor) {
        this.alternateColor = Objects.requireNonNull(alternateColor);
    }

    public DrawingColor getBackgroundColor() {
        return this.backgroundColor;
    }

    public void setBackgroundColor(DrawingColor backgroundColor) {
        this.backgroundColor = Objects.requireNonNull(backgroundColor);
    }

    public SpherePolyhedronColorSelector getColorSelector() {
        return this.colorSelector;
    }

    public void setColorSelector(SpherePolyhedronColorSelector colorSelector) {
        this.colorSelector = colorSelector;
    }

    public double getSectionZ() {
        return this.sectionZ;
    }

    public void setSectionZ(double sectionZ) {
        this.sectionZ = sectionZ;
    }

    public boolean isDissectedObjectsVisible() {
        return this.dissectedObjectsVisible;
    }

    public void setDissectedObjectsVisible(boolean dissectedObjectsVisible) {
        this.dissectedObjectsVisible = dissectedObjectsVisible;
    }

    public boolean isEdgesOnly() {
        return this.edgesOnly;
    }

    public void setEdgesOnly(boolean edgesOnly) {
        this.edgesOnly = edgesOnly;
    }

    public byte[] r(byte[] result, boolean invertLinesYOrder) {
        return this.channelBytes(result, this.r, invertLinesYOrder);
    }

    public byte[] g(byte[] result, boolean invertLinesYOrder) {
        return this.channelBytes(result, this.g, invertLinesYOrder);
    }

    public byte[] b(byte[] result, boolean invertLinesYOrder) {
        return this.channelBytes(result, this.b, invertLinesYOrder);
    }

    public List<Matrix<? extends PArray>> rgbMatrices(boolean invertLinesYOrder) {
        return List.of(Matrix.as((Object)this.r(null, invertLinesYOrder), (long[])new long[]{this.dimX, this.dimY}), Matrix.as((Object)this.g(null, invertLinesYOrder), (long[])new long[]{this.dimX, this.dimY}), Matrix.as((Object)this.b(null, invertLinesYOrder), (long[])new long[]{this.dimX, this.dimY}));
    }

    public void clear() {
        JArrays.fill((byte[])this.r, (byte)this.backgroundColor.byteR());
        JArrays.fill((byte[])this.g, (byte)this.backgroundColor.byteG());
        JArrays.fill((byte[])this.b, (byte)this.backgroundColor.byteB());
        JArrays.fill((double[])this.zBuffer, (double)Double.NEGATIVE_INFINITY);
        JArrays.fill((double[])this.minZ, (double)Double.POSITIVE_INFINITY);
        JArrays.fill((double[])this.maxZ, (double)Double.NEGATIVE_INFINITY);
        if (this.zBufferMinInBlocks != null) {
            JArrays.fill((double[])this.zBufferMinInBlocks, (double)Double.NEGATIVE_INFINITY);
        }
    }

    public void drawLine(double x1, double y1, double z1, double x2, double y2, double z2) {
        this.drawLine(x1, y1, z1, x2, y2, z2, this.color.byteR(), this.color.byteG(), this.color.byteB(), false);
    }

    public void drawGradientLine(double x1, double y1, double z1, double x2, double y2, double z2) {
        double db;
        double dg;
        double dr;
        double b1;
        double g1;
        double r1;
        boolean needToExchange;
        boolean horizontal;
        boolean bl = horizontal = Math.abs(x2 - x1) > Math.abs(y2 - y1);
        boolean bl2 = horizontal ? x2 < x1 : (needToExchange = y2 < y1);
        if (needToExchange) {
            double temp = x1;
            x1 = x2;
            x2 = temp;
            temp = y1;
            y1 = y2;
            y2 = temp;
            temp = z1;
            z1 = z2;
            z2 = temp;
            r1 = this.alternateColor.r255();
            g1 = this.alternateColor.g255();
            b1 = this.alternateColor.b255();
            dr = this.color.r255() - r1;
            dg = this.color.g255() - g1;
            db = this.color.b255() - b1;
        } else {
            r1 = this.color.r255();
            g1 = this.color.g255();
            b1 = this.color.b255();
            dr = this.alternateColor.r255() - r1;
            dg = this.alternateColor.g255() - g1;
            db = this.alternateColor.b255() - b1;
        }
        if (horizontal) {
            long x2Long;
            long x1Long = Math.round(x1);
            if (x1Long == (x2Long = Math.round(x2))) {
                this.drawPixel(x1Long, Math.round(y1), Math.max(z1, z2));
                return;
            }
            double diffXInv = 1.0 / (x2 - x1);
            double cy = (y2 - y1) * diffXInv;
            double cz = (z2 - z1) * diffXInv;
            long k1 = Math.max(0L, -x1Long);
            long k2 = Math.min(x2Long, (long)(this.dimX - 1)) - x1Long;
            for (long k = k1; k <= k2; ++k) {
                long y = Math.round(y1 + (double)k * cy);
                if (y < 0L || y >= (long)this.dimY) continue;
                double z = z1 + (double)k * cz;
                double q = (double)k * diffXInv;
                int disp = (int)(y * (long)this.dimX + x1Long + k);
                this.drawPixel(disp, z, r1 + q * dr, g1 + q * dg, b1 + q * db);
            }
        } else {
            long y2Long;
            long y1Long = Math.round(y1);
            if (y1Long == (y2Long = Math.round(y2))) {
                this.drawPixel(Math.round(x1), y1Long, Math.max(z1, z2));
                return;
            }
            double diffYInv = 1.0 / (y2 - y1);
            double cx = (x2 - x1) * diffYInv;
            double cz = (z2 - z1) * diffYInv;
            long k1 = Math.max(0L, -y1Long);
            long k2 = Math.min(y2Long, (long)(this.dimX - 1)) - y1Long;
            for (long k = k1; k <= k2; ++k) {
                long x = Math.round(x1 + (double)k * cx);
                if (x < 0L || x >= (long)this.dimX) continue;
                double z = z1 + (double)k * cz;
                double q = (double)k * diffYInv;
                int disp = (int)((y1Long + k) * (long)this.dimX + x);
                this.drawPixel(disp, z, r1 + q * dr, g1 + q * dg, b1 + q * db);
            }
        }
    }

    public void drawSphere(double centerX, double centerY, double centerZ, double radius) {
        this.drawSphere(centerX, centerY, centerZ, radius, this.color, false);
    }

    public void drawPolygon(double[] xyz, int count) {
        this.drawPolygon(xyz, count, this.color.byteR(), this.color.byteG(), this.color.byteB(), false);
    }

    public void drawParallelogram(double vx, double vy, double vz, double ix, double iy, double iz, double jx, double jy, double jz) {
        double[] xyz = new double[]{vx, vy, vz, vx + ix, vy + iy, vz + iz, vx + ix + jx, vy + iy + jy, vz + iz + jz, vx + jx, vy + jy, vz + jz};
        this.drawPolygon(xyz, 4);
    }

    public void drawCylinder(double x1, double y1, double z1, double x2, double y2, double z2, double radius) {
        this.drawCylinderSector(x1, y1, z1, x2, y2, z2, radius, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, this.color, true, false);
    }

    public boolean canBeDrawn(double centerX, double centerY, double centerZ, double radius) {
        int y2;
        int y1;
        int x2;
        if (radius < 0.0) {
            return false;
        }
        int ix = net.algart.arrays.Arrays.round32((double)centerX);
        int iy = net.algart.arrays.Arrays.round32((double)centerY);
        int ir = net.algart.arrays.Arrays.round32((double)radius);
        int x1 = ix - ir;
        if (x1 >= this.dimX) {
            return false;
        }
        if (x1 < 0) {
            x1 = 0;
        }
        if ((x2 = ix + ir) < 0) {
            return false;
        }
        if (x2 >= this.dimX) {
            x2 = this.dimX - 1;
        }
        if ((y1 = iy - ir) >= this.dimY) {
            return false;
        }
        if (y1 < 0) {
            y1 = 0;
        }
        if ((y2 = iy + ir) < 0) {
            return false;
        }
        if (y2 >= this.dimY) {
            y2 = this.dimY - 1;
        }
        double maxZ = centerZ + radius;
        if (centerZ - radius > this.sectionZ) {
            return false;
        }
        return this.needDrawAccordingOptimizingBlocks(x1, y1, x2, y2, maxZ);
    }

    public void drawSpherePolyhedron(SpherePolyhedron spherePolyhedron) {
        Color kindColor;
        Objects.requireNonNull(spherePolyhedron, "Null sphere-polyhedron");
        DrawingColor color = this.color;
        if (this.colorSelector != null && (kindColor = this.colorSelector.getColor(spherePolyhedron)) != null) {
            if (kindColor.getAlpha() == 0) {
                return;
            }
            color = DrawingColor.of(kindColor);
        }
        this.drawSpherePolyhedron(spherePolyhedron, color);
    }

    public void drawSpherePolyhedron(double x0, double y0, Orthonormal3DBasis basis, double scale, double centerX, double centerY, double centerZ, GeneratrixSet generatrixSet, long kindId) {
        this.drawSpherePolyhedron(SpherePolyhedraDrawer.newSpherePolyhedron(x0, y0, basis, scale, centerX, centerY, centerZ, generatrixSet, kindId));
    }

    public void drawSpherePolyhedra(double x0, double y0, Orthonormal3DBasis basis, double scale, SpherePolyhedra spherePolyhedra) {
        long t1 = System.nanoTime();
        int n = spherePolyhedra.numberOfSpherePolyhedra();
        long[] packedIndexesAndZ = new long[n];
        for (int k = 0; k < n; ++k) {
            double centerX = spherePolyhedra.getCenterX(k);
            double centerY = spherePolyhedra.getCenterY(k);
            double centerZ = spherePolyhedra.getCenterZ(k);
            double drawnCenterZ = scale * basis.z(centerX, centerY, centerZ);
            packedIndexesAndZ[k] = SpherePolyhedraDrawer.packToLong(k, (float)(-drawnCenterZ));
        }
        long t2 = System.nanoTime();
        Arrays.parallelSort(packedIndexesAndZ);
        int drawnCount = 0;
        long t3 = System.nanoTime();
        DrawingColor color = new DrawingColor(this.color);
        for (long packed : packedIndexesAndZ) {
            long kindId;
            double drawnR;
            double drawnCenterZ;
            double drawnCenterY;
            double centerZ;
            double centerY;
            int i = SpherePolyhedraDrawer.unpackIndex(packed);
            double centerX = spherePolyhedra.getCenterX(i);
            double drawnCenterX = x0 + scale * basis.x(centerX, centerY = spherePolyhedra.getCenterY(i), centerZ = spherePolyhedra.getCenterZ(i));
            if (!this.canBeDrawn(drawnCenterX, drawnCenterY = y0 + scale * basis.y(centerX, centerY, centerZ), drawnCenterZ = (double)(-SpherePolyhedraDrawer.unpackValue(packed)), drawnR = scale * spherePolyhedra.containingSphereRadius(i))) continue;
            GeneratrixSet generatrixSet = spherePolyhedra.getGeneratrixSet(i);
            GeneratrixSet drawnGeneratrixSet = generatrixSet.rotateAndScale(basis, scale);
            SpherePolyhedron spherePolyhedron = SpherePolyhedron.newInstance(drawnCenterX, drawnCenterY, drawnCenterZ, drawnGeneratrixSet, kindId = spherePolyhedra.getKindId(i));
            double radius = spherePolyhedron.containingSphereRadius();
            if (Math.abs(radius - drawnR) > Math.max(0.01 * radius, 1.0E-5)) {
                throw new AssertionError((Object)("Invalid SpherePolyhedra.containingSphereRadius(): " + drawnR + " instead of " + radius));
            }
            if (this.colorSelector != null) {
                Color kindColor = this.colorSelector.getColor(spherePolyhedron);
                if (kindColor != null) {
                    if (kindColor.getAlpha() == 0) continue;
                    color.setPackedRGB(kindColor.getRGB());
                } else {
                    color.setTo(this.color);
                }
            }
            this.drawSpherePolyhedron(spherePolyhedron, color);
            ++drawnCount;
        }
        long t4 = System.nanoTime();
        if (t4 - t1 > 100000000L) {
            long finalDrawnCount = drawnCount;
            LOG.log(System.Logger.Level.DEBUG, () -> String.format(Locale.US, "Drawing %d/%d sphere-polyhedra: %.3f ms = %.3f init + %.3f sort + %.3f draw%n", finalDrawnCount, n, (double)(t4 - t1) * 1.0E-6, (double)(t2 - t1) * 1.0E-6, (double)(t3 - t2) * 1.0E-6, (double)(t4 - t3) * 1.0E-6));
        }
    }

    private void drawLine(double x1, double y1, double z1, double x2, double y2, double z2, byte r, byte g, byte b, boolean findRangeZ) {
        if (Math.abs(x2 - x1) > Math.abs(y2 - y1)) {
            long x2Long;
            long x1Long;
            if (x2 < x1) {
                double temp = x1;
                x1 = x2;
                x2 = temp;
                temp = y1;
                y1 = y2;
                y2 = temp;
                temp = z1;
                z1 = z2;
                z2 = temp;
            }
            if ((x1Long = Math.round(x1)) == (x2Long = Math.round(x2))) {
                this.drawPixel(x1Long, Math.round(y1), Math.max(z1, z2), r, g, b, findRangeZ);
                return;
            }
            double diffXInv = 1.0 / (x2 - x1);
            double cy = (y2 - y1) * diffXInv;
            double cz = (z2 - z1) * diffXInv;
            long k1 = Math.max(0L, -x1Long);
            long k2 = Math.min(x2Long, (long)(this.dimX - 1)) - x1Long;
            for (long k = k1; k <= k2; ++k) {
                long y = Math.round(y1 + (double)k * cy);
                if (y < 0L || y >= (long)this.dimY) continue;
                double z = z1 + (double)k * cz;
                int disp = (int)(y * (long)this.dimX + x1Long + k);
                this.drawPixel(disp, z, r, g, b, findRangeZ);
            }
        } else {
            long y2Long;
            long y1Long;
            if (y2 < y1) {
                double temp = x1;
                x1 = x2;
                x2 = temp;
                temp = y1;
                y1 = y2;
                y2 = temp;
                temp = z1;
                z1 = z2;
                z2 = temp;
            }
            if ((y1Long = Math.round(y1)) == (y2Long = Math.round(y2))) {
                this.drawPixel(Math.round(x1), y1Long, Math.max(z1, z2), r, g, b, findRangeZ);
                return;
            }
            double diffYInv = 1.0 / (y2 - y1);
            double cx = (x2 - x1) * diffYInv;
            double cz = (z2 - z1) * diffYInv;
            long k1 = Math.max(0L, -y1Long);
            long k2 = Math.min(y2Long, (long)(this.dimX - 1)) - y1Long;
            for (long k = k1; k <= k2; ++k) {
                long x = Math.round(x1 + (double)k * cx);
                if (x < 0L || x >= (long)this.dimX) continue;
                double z = z1 + (double)k * cz;
                int disp = (int)((y1Long + k) * (long)this.dimX + x);
                this.drawPixel(disp, z, r, g, b, findRangeZ);
            }
        }
    }

    private void drawPolygon(double[] xyz, int count, byte r, byte g, byte b, boolean findRangeZ) {
        double dz;
        if (count <= 1) {
            return;
        }
        int minY = Integer.MAX_VALUE;
        int maxY = Integer.MIN_VALUE;
        int k = 0;
        int disp = 1;
        while (k < count) {
            int y = net.algart.arrays.Arrays.round32((double)xyz[3 * k + 1]);
            minY = Math.min(minY, y);
            maxY = Math.max(maxY, y);
            ++k;
            disp += 3;
        }
        if (minY >= this.dimY || maxY < 0) {
            return;
        }
        minY = Math.max(minY, 0);
        maxY = Math.min(maxY, this.dimY - 1);
        assert (minY <= maxY);
        assert (maxY < this.dimY);
        for (int y = minY; y <= maxY; ++y) {
            this.polygonLeftX[y] = Integer.MAX_VALUE;
            this.polygonRightX[y] = Integer.MIN_VALUE;
        }
        long x = net.algart.arrays.Arrays.round32((double)xyz[3 * count - 3]);
        long y = net.algart.arrays.Arrays.round32((double)xyz[3 * count - 2]);
        double z = xyz[3 * count - 1];
        int k2 = 0;
        int disp2 = 0;
        while (k2 < count) {
            long i2;
            long dx = (long)net.algart.arrays.Arrays.round32((double)xyz[disp2]) - x;
            long dy = (long)net.algart.arrays.Arrays.round32((double)xyz[disp2 + 1]) - y;
            dz = xyz[disp2 + 2] - z;
            boolean exchanged = false;
            if (dy < 0L) {
                x += dx;
                y += dy;
                z += dz;
                dx = -dx;
                dy = -dy;
                dz = -dz;
                exchanged = true;
            }
            double diffYInv = dy == 0L ? 1.0 : 1.0 / (double)dy;
            double cx = (double)dx * diffYInv;
            double cz = dz * diffYInv;
            long i1 = Math.max(0L, -y);
            if (i1 <= (i2 = Math.min(y + dy, (long)(this.dimY - 1)) - y)) {
                int j = (int)(y + i1);
                long i = i1;
                while (i <= i2) {
                    int lineX = net.algart.arrays.Arrays.round32((double)((double)x + (double)i * cx));
                    double lineZ = z + (double)i * cz;
                    if (lineX < this.polygonLeftX[j]) {
                        this.polygonLeftX[j] = lineX;
                        this.polygonLeftZ[j] = lineZ;
                    } else if (lineX == this.polygonLeftX[j]) {
                        this.polygonLeftZ[j] = Math.max(lineZ, this.polygonLeftZ[j]);
                    }
                    if (lineX > this.polygonRightX[j]) {
                        this.polygonRightX[j] = lineX;
                        this.polygonRightZ[j] = lineZ;
                    } else if (lineX == this.polygonRightX[j]) {
                        this.polygonRightZ[j] = Math.max(lineZ, this.polygonRightZ[j]);
                    }
                    ++i;
                    ++j;
                }
            }
            if (!exchanged) {
                x += dx;
                y += dy;
                z += dz;
            }
            ++k2;
            disp2 += 3;
        }
        for (int j = minY; j <= maxY; ++j) {
            int x1 = this.polygonLeftX[j];
            int x2 = this.polygonRightX[j];
            if (x2 == Integer.MIN_VALUE) {
                throw new AssertionError((Object)("Line " + j + " was not processed!"));
            }
            double z1 = this.polygonLeftZ[j];
            int dx = x2 - x1;
            dz = this.polygonRightZ[j] - z1;
            int k1 = Math.max(0, -x1);
            int k22 = Math.min(x1 + dx, this.dimX - 1) - x1;
            double cz = dx == 0 ? 0.0 : dz / (double)dx;
            int k3 = k1;
            int disp3 = j * this.dimX + x1 + k1;
            while (k3 <= k22) {
                z = z1 + (double)k3 * cz;
                this.drawPixel(disp3, z, r, g, b, findRangeZ);
                ++k3;
                ++disp3;
            }
        }
    }

    private void drawSphere(double centerX, double centerY, double centerZ, double radius, DrawingColor color, boolean findRangeZ) {
        int y2;
        int y1;
        int x2;
        if (radius < 0.0) {
            return;
        }
        int ix = net.algart.arrays.Arrays.round32((double)centerX);
        int iy = net.algart.arrays.Arrays.round32((double)centerY);
        int ir = net.algart.arrays.Arrays.round32((double)radius);
        int x1 = ix - ir;
        if (x1 >= this.dimX) {
            return;
        }
        if (x1 < 0) {
            x1 = 0;
        }
        if ((x2 = ix + ir) < 0) {
            return;
        }
        if (x2 >= this.dimX) {
            x2 = this.dimX - 1;
        }
        if ((y1 = iy - ir) >= this.dimY) {
            return;
        }
        if (y1 < 0) {
            y1 = 0;
        }
        if ((y2 = iy + ir) < 0) {
            return;
        }
        if (y2 >= this.dimY) {
            y2 = this.dimY - 1;
        }
        double maxZ = centerZ + radius;
        if (!findRangeZ) {
            if (centerZ - radius > this.sectionZ) {
                return;
            }
            if (maxZ > this.sectionZ && !this.dissectedObjectsVisible) {
                return;
            }
        }
        if (!this.needDrawAccordingOptimizingBlocks(x1, y1, x2, y2, maxZ)) {
            return;
        }
        radius = ir;
        double rSqr = radius + 0.5;
        rSqr *= rSqr;
        double rInv = 1.0 / radius;
        int i = y1;
        int dy = y1 - iy;
        int lineDisp = y1 * this.dimX;
        while (i <= y2) {
            int j = x1;
            int dx = x1 - ix;
            int disp = lineDisp + x1;
            while (j <= x2) {
                double deltaY;
                double deltaX;
                double deltaZSqr;
                if (!(this.zBuffer[disp] >= maxZ) && !((deltaZSqr = rSqr - ((deltaX = (double)dx) * deltaX + (deltaY = (double)dy) * deltaY)) < 0.0)) {
                    double brightness;
                    double deltaZ = Math.sqrt(deltaZSqr);
                    double zTop = centerZ + deltaZ;
                    double zBottom = centerZ - deltaZ;
                    if (findRangeZ) {
                        if (zBottom < this.minZ[disp]) {
                            this.minZ[disp] = zBottom;
                        }
                        if (zTop > this.maxZ[disp]) {
                            this.maxZ[disp] = zTop;
                        }
                    }
                    if ((brightness = deltaZ * rInv) > 1.0) {
                        brightness = 1.0;
                    }
                    if (zTop <= this.sectionZ) {
                        this.drawPixel(disp, zTop, brightness * color.r255(), brightness * color.g255(), brightness * color.b255());
                    } else if (zBottom <= this.sectionZ) {
                        this.drawSectionPixel(disp, color);
                    }
                }
                ++j;
                ++dx;
                ++disp;
            }
            ++i;
            ++dy;
            lineDisp += this.dimX;
        }
        this.correctOptimizingBlocks(x1, y1, x2, y2);
    }

    private void drawCylinderSector(double x1, double y1, double z1, double x2, double y2, double z2, double radius, double normalX1, double normalY1, double normalZ1, double normalX2, double normalY2, double normalZ2, DrawingColor color, boolean wholeCylinder, boolean findRangeZ) {
        if (radius < 0.0) {
            return;
        }
        int ix1 = net.algart.arrays.Arrays.round32((double)x1);
        int iy1 = net.algart.arrays.Arrays.round32((double)y1);
        int ix2 = net.algart.arrays.Arrays.round32((double)x2);
        int iy2 = net.algart.arrays.Arrays.round32((double)y2);
        if (ix1 == ix2 && iy1 == iy2) {
            return;
        }
        int ir = net.algart.arrays.Arrays.round32((double)radius);
        if (ix1 - ir >= this.dimX && ix2 - ir >= this.dimX || ix1 + ir < 0 && ix2 + ir < 0 || iy1 - ir >= this.dimY && iy2 - ir >= this.dimY || iy1 + ir < 0 && iy2 + ir < 0 || z1 - radius > this.sectionZ && z2 - radius > this.sectionZ && !findRangeZ) {
            return;
        }
        radius = (double)ir + 0.5;
        double gX = x2 - x1;
        double gY = y2 - y1;
        double gZ = z2 - z1;
        double gXYSqr = gX * gX + gY * gY;
        if (gXYSqr < 0.25) {
            return;
        }
        double gXYAbs = Math.sqrt(gXYSqr);
        double gAbs = Math.sqrt(gXYSqr + gZ * gZ);
        double dFi = 0.6 / radius;
        int count = net.algart.arrays.Arrays.round32((double)(Math.PI * 2 / dFi));
        double cosDFi = Math.cos(dFi);
        double sinDFi = Math.sin(dFi);
        double ix = normalX1 * radius;
        double iy = normalY1 * radius;
        double iz = normalZ1 * radius;
        double finalIx = normalX2 * radius;
        double finalIy = normalY2 * radius;
        double finalIz = normalZ2 * radius;
        if (wholeCylinder) {
            iz = 0.0;
            iy = -gX * radius / gXYAbs;
            ix = gY * radius / gXYAbs;
        }
        double jx = (gY * iz - gZ * iy) / gAbs;
        double jy = (gZ * ix - gX * iz) / gAbs;
        double jz = (gX * iy - gY * ix) / gAbs;
        boolean first = true;
        int newDx = net.algart.arrays.Arrays.round32((double)ix);
        int newDy = net.algart.arrays.Arrays.round32((double)iy);
        int dx = newDx;
        int dy = newDy;
        while (count >= 0) {
            boolean needDrawGeneratrix;
            double brightness = Math.abs(iz) / radius;
            if (ir == 0) {
                brightness = 1.0;
            }
            if (brightness > 1.0) {
                brightness = 1.0;
            }
            byte r = (byte)(brightness * color.r255() + 0.5);
            byte g = (byte)(brightness * color.g255() + 0.5);
            byte b = (byte)(brightness * color.b255() + 0.5);
            boolean bl = needDrawGeneratrix = Math.round(iz) > 0L || findRangeZ;
            if (needDrawGeneratrix) {
                if (!first && newDx != dx && newDy != dy) {
                    if (newDx > dx) {
                        if (newDy > dy) {
                            if (dy > 0) {
                                dx = newDx;
                            } else {
                                dy = newDy;
                            }
                        } else if (dy > 0) {
                            dy = newDy;
                        } else {
                            dx = newDx;
                        }
                    } else if (newDy > dy) {
                        if (dy > 0) {
                            dy = newDy;
                        } else {
                            dx = newDx;
                        }
                    } else if (dy > 0) {
                        dx = newDx;
                    } else {
                        dy = newDy;
                    }
                    this.drawLine(ix1 + dx, iy1 + dy, z1 + iz, ix2 + dx, iy2 + dy, z2 + iz, r, g, b, findRangeZ);
                }
                if (first || newDx != dx || newDy != dy) {
                    this.drawLine(ix1 + newDx, iy1 + newDy, z1 + iz, ix2 + newDx, iy2 + newDy, z2 + iz, r, g, b, findRangeZ);
                }
            }
            boolean bl2 = first = !needDrawGeneratrix;
            if (!wholeCylinder && Math.abs(ix - finalIx) < 1.0 && Math.abs(iy - finalIy) < 1.0 && Math.abs(iz - finalIz) < 1.0) {
                ix = finalIx;
                iy = finalIy;
                iz = finalIz;
                count = Math.min(count, 1);
            } else {
                double newIx = cosDFi * ix + sinDFi * jx;
                double newIy = cosDFi * iy + sinDFi * jy;
                double newIz = cosDFi * iz + sinDFi * jz;
                double newJx = -sinDFi * ix + cosDFi * jx;
                double newJy = -sinDFi * iy + cosDFi * jy;
                double newJz = -sinDFi * iz + cosDFi * jz;
                ix = newIx;
                iy = newIy;
                iz = newIz;
                jx = newJx;
                jy = newJy;
                jz = newJz;
            }
            dx = newDx;
            dy = newDy;
            newDx = net.algart.arrays.Arrays.round32((double)ix);
            newDy = net.algart.arrays.Arrays.round32((double)iy);
            --count;
        }
    }

    private void drawSpherePolyhedron(SpherePolyhedron spherePolyhedron, DrawingColor color) {
        boolean isSphereCylinder;
        boolean dissected;
        int y2;
        int y1;
        int x2;
        Objects.requireNonNull(spherePolyhedron, "Null sphere-polyhedron");
        if (spherePolyhedron.isSphere()) {
            this.drawSphere(spherePolyhedron.centerX(), spherePolyhedron.centerY(), spherePolyhedron.centerZ(), spherePolyhedron.containingSphereRadius(), color, false);
            return;
        }
        int x1 = (int)Math.floor(spherePolyhedron.minX());
        if (x1 >= this.dimX) {
            return;
        }
        if (x1 < 0) {
            x1 = 0;
        }
        if ((x2 = (int)Math.ceil(spherePolyhedron.maxX())) < 0) {
            return;
        }
        if (x2 >= this.dimX) {
            x2 = this.dimX - 1;
        }
        if ((y1 = (int)Math.floor(spherePolyhedron.minY())) >= this.dimY) {
            return;
        }
        if (y1 < 0) {
            y1 = 0;
        }
        if ((y2 = (int)Math.ceil(spherePolyhedron.maxY())) < 0) {
            return;
        }
        if (y2 >= this.dimY) {
            y2 = this.dimY - 1;
        }
        double minZ = spherePolyhedron.minZ();
        double maxZ = spherePolyhedron.maxZ();
        if (minZ > this.sectionZ) {
            return;
        }
        boolean bl = dissected = maxZ > this.sectionZ;
        if (dissected && !this.dissectedObjectsVisible) {
            return;
        }
        if (!this.needDrawAccordingOptimizingBlocks(x1, y1, x2, y2, maxZ)) {
            return;
        }
        if (dissected) {
            int i = y1;
            int lineDisp = this.dimX * y1;
            while (i <= y2) {
                int j = x1;
                int disp = lineDisp + x1;
                while (j <= x2) {
                    this.minZ[disp] = Double.POSITIVE_INFINITY;
                    this.maxZ[disp] = Double.NEGATIVE_INFINITY;
                    ++j;
                    ++disp;
                }
                ++i;
                lineDisp += this.dimX;
            }
        }
        double radius = spherePolyhedron.generatrixSphereRadius();
        if (!this.edgesOnly) {
            double correctedRadiusForDrawingFacets = radius == 0.0 ? radius : (double)Math.round(radius) + 0.5;
            double[] xyz = new double[12];
            int n = spherePolyhedron.numberOfFacets();
            for (int f = 0; f < n; ++f) {
                double normalX = Math.round(spherePolyhedron.facetNormalX(f) * correctedRadiusForDrawingFacets);
                double normalY = Math.round(spherePolyhedron.facetNormalY(f) * correctedRadiusForDrawingFacets);
                double normalZUnit = spherePolyhedron.facetNormalZ(f);
                double normalZ = normalZUnit * correctedRadiusForDrawingFacets;
                if (normalZUnit == 0.0 || !dissected && !(normalZUnit > 0.0)) continue;
                double brightness = Math.abs(normalZUnit);
                if (brightness > 1.0) {
                    brightness = 1.0;
                }
                byte r = (byte)(brightness * color.r255() + 0.5);
                byte g = (byte)(brightness * color.g255() + 0.5);
                byte b = (byte)(brightness * color.b255() + 0.5);
                int xyzDisp = 0;
                for (int k = 0; k < 4; ++k) {
                    int v = spherePolyhedron.facetVertexIndex(f, k);
                    xyz[xyzDisp++] = (double)Math.round(spherePolyhedron.vertexX(v)) + normalX;
                    xyz[xyzDisp++] = (double)Math.round(spherePolyhedron.vertexY(v)) + normalY;
                    xyz[xyzDisp++] = spherePolyhedron.vertexZ(v) + normalZ;
                }
                this.drawPolygon(xyz, 4, r, g, b, dissected);
            }
        }
        boolean bl2 = isSphereCylinder = spherePolyhedron.numberOfFacets() == 0;
        if (radius > 0.0 || isSphereCylinder) {
            int n = spherePolyhedron.numberOfEdges();
            for (int e = 0; e < n; ++e) {
                if (spherePolyhedron.isEdgeAlmostDegenerated(e)) continue;
                int f1 = spherePolyhedron.edgeFacet1Index(e);
                int f2 = spherePolyhedron.edgeFacet2Index(e);
                double normalX1 = f1 == -1 ? 0.0 : spherePolyhedron.facetNormalX(f1);
                double normalY1 = f1 == -1 ? 0.0 : spherePolyhedron.facetNormalY(f1);
                double normalZ1 = f1 == -1 ? 0.0 : spherePolyhedron.facetNormalZ(f1);
                double normalX2 = f2 == -1 ? 0.0 : spherePolyhedron.facetNormalX(f2);
                double normalY2 = f2 == -1 ? 0.0 : spherePolyhedron.facetNormalY(f2);
                double normalZ2 = f2 == -1 ? 0.0 : spherePolyhedron.facetNormalZ(f2);
                int v1 = spherePolyhedron.edgeVertex1Index(e);
                int v2 = spherePolyhedron.edgeVertex2Index(e);
                double vX1 = spherePolyhedron.vertexX(v1);
                double vY1 = spherePolyhedron.vertexY(v1);
                double vZ1 = spherePolyhedron.vertexZ(v1);
                double vX2 = spherePolyhedron.vertexX(v2);
                double vY2 = spherePolyhedron.vertexY(v2);
                double vZ2 = spherePolyhedron.vertexZ(v2);
                this.drawCylinderSector(vX1, vY1, vZ1, vX2, vY2, vZ2, radius, normalX1, normalY1, normalZ1, normalX2, normalY2, normalZ2, color, isSphereCylinder || this.edgesOnly, dissected);
            }
            n = spherePolyhedron.numberOfVertices();
            for (int v = 0; v < n; ++v) {
                if (spherePolyhedron.isVertexAlmostDegenerated(v)) continue;
                double vX = spherePolyhedron.vertexX(v);
                double vY = spherePolyhedron.vertexY(v);
                double vZ = spherePolyhedron.vertexZ(v);
                this.drawSphere(vX, vY, vZ, radius, color, dissected);
            }
        }
        if (dissected) {
            int i = y1;
            int lineDisp = this.dimX * y1;
            while (i <= y2) {
                int j = x1;
                int disp = lineDisp + x1;
                while (j <= x2) {
                    if (this.minZ[disp] <= this.sectionZ && this.maxZ[disp] > this.sectionZ) {
                        this.drawSectionPixel(disp, color);
                    }
                    ++j;
                    ++disp;
                }
                ++i;
                lineDisp += this.dimX;
            }
        }
        this.correctOptimizingBlocks(x1, y1, x2, y2);
    }

    private boolean needDrawAccordingOptimizingBlocks(int x1, int y1, int x2, int y2, double maxZ) {
        if (this.zBufferMinInBlocks == null) {
            return true;
        }
        int x1b = x1 >> this.blockLog;
        int x2b = x2 >> this.blockLog;
        int y1b = y1 >> this.blockLog;
        int y2b = y2 >> this.blockLog;
        int ib = y1b;
        int blockLineDisp = y1b * this.nBlocksX;
        while (ib <= y2b) {
            int jb = x1b;
            int blockDisp = blockLineDisp + x1b;
            while (jb <= x2b) {
                if (this.zBufferMinInBlocks[blockDisp] < maxZ) {
                    return true;
                }
                ++jb;
                ++blockDisp;
            }
            ++ib;
            blockLineDisp += this.nBlocksX;
        }
        return false;
    }

    private void correctOptimizingBlocks(int x1, int y1, int x2, int y2) {
        if (this.zBufferMinInBlocks == null) {
            return;
        }
        int x1b = x1 >> this.blockLog;
        int x2b = x2 >> this.blockLog;
        int y1b = y1 >> this.blockLog;
        int y2b = y2 >> this.blockLog;
        int ib = y1b;
        int blockLineDisp = y1b * this.nBlocksX;
        while (ib <= y2b) {
            int jb = x1b;
            int blockDisp = blockLineDisp + x1b;
            while (jb <= x2b) {
                double zMinInBlock = Double.POSITIVE_INFINITY;
                int lineDisp = (ib << this.blockLog) * this.dimX + (jb << this.blockLog);
                for (int i = 0; i < this.blockSize && lineDisp < this.dimXY; ++i, lineDisp += this.dimX) {
                    int maxDisp = Math.min(lineDisp + this.blockSize, this.dimXY);
                    for (int disp = lineDisp; disp < maxDisp; ++disp) {
                        if (!(this.zBuffer[disp] < zMinInBlock)) continue;
                        zMinInBlock = this.zBuffer[disp];
                    }
                }
                this.zBufferMinInBlocks[blockDisp] = zMinInBlock;
                ++jb;
                ++blockDisp;
            }
            ++ib;
            blockLineDisp += this.nBlocksX;
        }
    }

    private void drawPixel(long x, long y, double z) {
        this.drawPixel(x, y, z, this.color.byteR(), this.color.byteG(), this.color.byteB(), false);
    }

    private void drawPixel(long x, long y, double z, byte r, byte g, byte b, boolean findRangeZ) {
        if (x >= 0L && x < (long)this.dimX && y >= 0L && y < (long)this.dimY) {
            this.drawPixel((int)(y * (long)this.dimX + x), z, r, g, b, findRangeZ);
        }
    }

    private void drawPixel(int disp, double z, byte r, byte g, byte b, boolean findRangeZ) {
        if (z <= this.sectionZ && this.zBuffer[disp] < z) {
            this.zBuffer[disp] = z;
            this.r[disp] = r;
            this.g[disp] = g;
            this.b[disp] = b;
        }
        if (findRangeZ) {
            if (z < this.minZ[disp]) {
                this.minZ[disp] = z;
            }
            if (z > this.maxZ[disp]) {
                this.maxZ[disp] = z;
            }
        }
    }

    private void drawPixel(int disp, double z, double r255, double g255, double b255) {
        if (z <= this.sectionZ && this.zBuffer[disp] < z) {
            this.zBuffer[disp] = z;
            this.r[disp] = (byte)(r255 + 0.5);
            this.g[disp] = (byte)(g255 + 0.5);
            this.b[disp] = (byte)(b255 + 0.5);
        }
    }

    private void drawSectionPixel(int disp, DrawingColor color) {
        if (this.zBuffer[disp] < this.sectionZ) {
            this.zBuffer[disp] = this.sectionZ;
            this.r[disp] = color.byteR();
            this.g[disp] = color.byteG();
            this.b[disp] = color.byteB();
        }
    }

    private byte[] channelBytes(byte[] result, byte[] channel, boolean invertLinesYOrder) {
        if (result == null) {
            result = new byte[this.dimXY];
        }
        if (invertLinesYOrder) {
            int i = 0;
            int disp = 0;
            int resultDisp = this.dimXY - this.dimX;
            while (i < this.dimY) {
                System.arraycopy(channel, disp, result, resultDisp, this.dimX);
                ++i;
                disp += this.dimX;
                resultDisp -= this.dimX;
            }
        } else {
            System.arraycopy(channel, 0, result, 0, this.dimXY);
        }
        return result;
    }

    static long packToLong(int index, float value) {
        int x = Float.floatToIntBits(value);
        int corrected = x ^ x >> 30 >>> 1;
        return (long)corrected << 32 | (long)index;
    }

    static int unpackIndex(long indexAndValue) {
        return (int)indexAndValue;
    }

    static float unpackValue(long indexAndValue) {
        int x = (int)(indexAndValue >>> 32);
        int corrected = x ^ x >> 30 >>> 1;
        return Float.intBitsToFloat(corrected);
    }
}

