/*
 * Decompiled with CFR 0.152.
 */
package net.algart.math;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Objects;
import java.util.zip.CRC32;
import net.algart.math.IPoint;
import net.algart.math.IRange;
import net.algart.math.IRectangularArea;
import net.algart.math.Range;
import net.algart.math.RectangularArea;

public class Point
implements Comparable<Point> {
    private static final Point[] originsCache = new Point[16];
    final double[] coordinates;

    Point(double[] coordinates) {
        this.coordinates = coordinates;
    }

    public static Point valueOf(double ... coordinates) {
        Objects.requireNonNull(coordinates, "Null coordinates argument");
        if (coordinates.length == 0) {
            throw new IllegalArgumentException("Empty coordinates array");
        }
        if (coordinates.length <= originsCache.length) {
            boolean origin = true;
            for (double coord : coordinates) {
                if (coord == 0.0) continue;
                origin = false;
                break;
            }
            if (origin) {
                return originsCache[coordinates.length - 1];
            }
        }
        return new Point((double[])coordinates.clone());
    }

    public static Point valueOf(double x) {
        return x == 0.0 ? originsCache[0] : new Point(new double[]{x});
    }

    public static Point valueOf(double x, double y) {
        return x == 0.0 && y == 0.0 ? originsCache[1] : new Point(new double[]{x, y});
    }

    public static Point valueOf(double x, double y, double z) {
        return x == 0.0 && y == 0.0 && z == 0.0 ? originsCache[2] : new Point(new double[]{x, y, z});
    }

    public static Point valueOfEqualCoordinates(int coordCount, double filler) {
        if (filler == 0.0) {
            return Point.origin(coordCount);
        }
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        double[] coordinates = new double[coordCount];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = filler;
        }
        return new Point(coordinates);
    }

    public static Point origin(int coordCount) {
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        if (coordCount <= originsCache.length) {
            return originsCache[coordCount - 1];
        }
        return new Point(new double[coordCount]);
    }

    public static Point minValue(int coordCount) {
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        double[] coordinates = new double[coordCount];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = Double.NEGATIVE_INFINITY;
        }
        return new Point(coordinates);
    }

    public static Point maxValue(int coordCount) {
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        double[] coordinates = new double[coordCount];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = Double.POSITIVE_INFINITY;
        }
        return new Point(coordinates);
    }

    public static Point valueOf(IPoint iPoint) {
        Objects.requireNonNull(iPoint, "Null iPoint argument");
        double[] coordinates = new double[iPoint.coordCount()];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = iPoint.coord(k);
        }
        return new Point(coordinates);
    }

    public int coordCount() {
        return this.coordinates.length;
    }

    public double[] coordinates() {
        return (double[])this.coordinates.clone();
    }

    public double[] coordinates(double[] result) {
        if (result.length < this.coordinates.length) {
            throw new IllegalArgumentException("Too short result array: double[" + result.length + "]; " + this.coordinates.length + " elements required to store coordinates");
        }
        System.arraycopy(this.coordinates, 0, result, 0, this.coordinates.length);
        return result;
    }

    public double coord(int coordIndex) {
        return this.coordinates[coordIndex];
    }

    public double x() {
        return this.coordinates[0];
    }

    public double y() {
        if (this.coordinates.length < 2) {
            throw new IllegalStateException("Cannot get y-coordinate of " + this.coordinates.length + "-dimensional point");
        }
        return this.coordinates[1];
    }

    public double z() {
        if (this.coordinates.length < 3) {
            throw new IllegalStateException("Cannot get z-coordinate of " + this.coordinates.length + "-dimensional point");
        }
        return this.coordinates[2];
    }

    public boolean isOrigin() {
        for (double coord : this.coordinates) {
            if (coord == 0.0) continue;
            return false;
        }
        return true;
    }

    public double distanceFromOrigin() {
        if (this.coordinates.length == 1) {
            return StrictMath.abs(this.coordinates[0]);
        }
        double dSqr = 0.0;
        for (double coord : this.coordinates) {
            dSqr += coord * coord;
        }
        return StrictMath.sqrt(dSqr);
    }

    public double distanceFrom(Collection<Point> points) {
        Objects.requireNonNull(points, "Null points argument");
        double result = Double.POSITIVE_INFINITY;
        for (Point point : points) {
            double d;
            if (point.coordCount() != this.coordinates.length) {
                throw new IllegalArgumentException("Dimensions count mismatch: some of the passed points has " + point.coordCount() + " dimensions instead of " + this.coordinates.length);
            }
            if (this.coordinates.length == 1) {
                d = StrictMath.abs(this.coordinates[0] - point.coordinates[0]);
            } else if (this.coordinates.length == 2) {
                d = StrictMath.hypot(this.coordinates[0] - point.coordinates[0], this.coordinates[1] - point.coordinates[1]);
            } else {
                double dSqr = 0.0;
                for (int k = 0; k < this.coordinates.length; ++k) {
                    double diff = this.coordinates[k] - point.coordinates[k];
                    dSqr += diff * diff;
                }
                d = StrictMath.sqrt(dSqr);
            }
            if (!(d < result)) continue;
            result = d;
        }
        return result;
    }

    public Point add(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] + point.coordinates[k];
        }
        return new Point(coordinates);
    }

    public Point subtract(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] - point.coordinates[k];
        }
        return new Point(coordinates);
    }

    public Point min(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = Math.min(this.coordinates[k], point.coordinates[k]);
        }
        return new Point(coordinates);
    }

    public Point max(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = Math.max(this.coordinates[k], point.coordinates[k]);
        }
        return new Point(coordinates);
    }

    public Point addToAllCoordinates(double increment) {
        if (increment == 0.0) {
            return this;
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] + increment;
        }
        return new Point(coordinates);
    }

    public Point multiply(double multiplier) {
        if (multiplier == 0.0) {
            return Point.origin(this.coordinates.length);
        }
        if (multiplier == 1.0) {
            return this;
        }
        if (multiplier == -1.0) {
            return this.symmetric();
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] * multiplier;
        }
        return new Point(coordinates);
    }

    public Point scale(double ... multipliers) {
        Objects.requireNonNull(multipliers, "Null multipliers argument");
        if (multipliers.length != this.coordinates.length) {
            throw new IllegalArgumentException("Illegal number of multipliers: " + multipliers.length + " instead of " + this.coordinates.length);
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = multipliers[k] == 0.0 ? 0.0 : (multipliers[k] == 1.0 ? this.coordinates[k] : (multipliers[k] == -1.0 ? -this.coordinates[k] : this.coordinates[k] * multipliers[k]));
        }
        return new Point(coordinates);
    }

    public Point scaleAndShift(double[] multipliers, Point shift) {
        double[] coordinates = new double[this.coordinates.length];
        this.scaleAndShift(coordinates, multipliers, shift);
        return new Point(coordinates);
    }

    public void scaleAndShift(double[] resultCoordinates, double[] multipliers, Point shift) {
        if (resultCoordinates.length < this.coordinates.length) {
            throw new IllegalArgumentException("Too short result coordinates array: double[" + resultCoordinates.length + "]; " + this.coordinates.length + " elements required to store coordinates");
        }
        Objects.requireNonNull(multipliers, "Null multipliers argument");
        if (multipliers.length != this.coordinates.length) {
            throw new IllegalArgumentException("Illegal number of multipliers: " + multipliers.length + " instead of " + this.coordinates.length);
        }
        Objects.requireNonNull(shift, "Null shift argument");
        if (shift.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + shift.coordCount() + " instead of " + this.coordinates.length);
        }
        for (int k = 0; k < this.coordinates.length; ++k) {
            resultCoordinates[k] = multipliers[k] == 0.0 ? shift.coord(k) : (multipliers[k] == 1.0 ? shift.coord(k) + this.coordinates[k] : (multipliers[k] == -1.0 ? shift.coord(k) - this.coordinates[k] : shift.coord(k) + this.coordinates[k] * multipliers[k]));
        }
    }

    public Point shiftAlongAxis(int coordIndex, double shift) {
        this.coord(coordIndex);
        if (shift == 0.0) {
            return this;
        }
        double[] coordinates = (double[])this.coordinates.clone();
        int n = coordIndex;
        coordinates[n] = coordinates[n] + shift;
        return new Point(coordinates);
    }

    public double scalarProduct(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double result = 0.0;
        for (int k = 0; k < this.coordinates.length; ++k) {
            result += this.coordinates[k] * point.coordinates[k];
        }
        return result;
    }

    public Point symmetric() {
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = -this.coordinates[k];
        }
        return new Point(coordinates);
    }

    public Point projectionAlongAxis(int coordIndex) {
        this.coord(coordIndex);
        if (this.coordinates.length == 1) {
            throw new IllegalStateException("Cannot perform projection of 1-dimensional figures");
        }
        double[] coordinates = new double[this.coordinates.length - 1];
        System.arraycopy(this.coordinates, 0, coordinates, 0, coordIndex);
        System.arraycopy(this.coordinates, coordIndex + 1, coordinates, coordIndex, coordinates.length - coordIndex);
        return new Point(coordinates);
    }

    boolean projectionAlongAxisEquals(int coordIndex, Point point) {
        int k;
        this.coord(coordIndex);
        if (this.coordinates.length == 1) {
            throw new IllegalStateException("Cannot perform projection of 1-dimensional figures");
        }
        if (this.coordinates.length != point.coordCount() + 1) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " is not equal to " + this.coordinates.length + "-1");
        }
        for (k = 0; k < coordIndex; ++k) {
            if (this.coordinates[k] == point.coordinates[k]) continue;
            return false;
        }
        for (k = coordIndex + 1; k < this.coordinates.length; ++k) {
            if (this.coordinates[k] == point.coordinates[k - 1]) continue;
            return false;
        }
        return true;
    }

    @Override
    public int compareTo(Point o) {
        return this.compareTo(o, 0);
    }

    public int compareTo(Point o, int firstCoordIndex) {
        int n = Math.max(this.coordinates.length, o.coordinates.length);
        if (firstCoordIndex < 0) {
            throw new IllegalArgumentException("Negative firstCoordIndex argument");
        }
        firstCoordIndex %= n;
        for (int k = n - 1; k >= 0; --k) {
            double otherCoord;
            int index = k + firstCoordIndex;
            if (index >= n) {
                index -= n;
            }
            double thisCoord = index >= this.coordinates.length ? 0.0 : this.coordinates[index];
            double d = otherCoord = index >= o.coordinates.length ? 0.0 : o.coordinates[index];
            if (thisCoord > otherCoord) {
                return 1;
            }
            if (!(thisCoord < otherCoord)) continue;
            return -1;
        }
        return Integer.compare(this.coordinates.length, o.coordinates.length);
    }

    public String toString() {
        StringBuilder result = new StringBuilder("(");
        for (int k = 0; k < this.coordinates.length; ++k) {
            if (k > 0) {
                result.append(", ");
            }
            result.append(this.coordinates[k]);
        }
        result.append(")");
        return result.toString();
    }

    public int hashCode() {
        CRC32 sum = new CRC32();
        byte[] bytes = new byte[this.coordinates.length * 8];
        Point.getBytes(this.coordinates, bytes);
        sum.update(bytes, 0, bytes.length);
        return (int)sum.getValue() ^ 0x1C11DD;
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof Point)) {
            return false;
        }
        Point p = (Point)obj;
        if (p.coordinates.length != this.coordinates.length) {
            return false;
        }
        for (int k = 0; k < this.coordinates.length; ++k) {
            if (Double.doubleToLongBits(p.coordinates[k]) == Double.doubleToLongBits(this.coordinates[k])) continue;
            return false;
        }
        return true;
    }

    public boolean isInteger() {
        for (double coordinate : this.coordinates) {
            if (coordinate == (double)((long)coordinate)) continue;
            return false;
        }
        return true;
    }

    public IPoint toIntegerPoint() {
        return IPoint.valueOf(this);
    }

    public IPoint toRoundedPoint() {
        return IPoint.roundOf(this);
    }

    private static void getBytes(double[] array, byte[] result) {
        int disp = 0;
        for (int k = 0; k < array.length; ++k) {
            long l = Double.doubleToLongBits(array[k]);
            int value = (int)l ^ 0x2596115B;
            result[disp++] = (byte)value;
            result[disp++] = (byte)(value >>> 8);
            result[disp++] = (byte)(value >>> 16);
            result[disp++] = (byte)(value >>> 24);
            value = (int)(l >>> 32) ^ 0x375AA545;
            result[disp++] = (byte)value;
            result[disp++] = (byte)(value >>> 8);
            result[disp++] = (byte)(value >>> 16);
            result[disp++] = (byte)(value >>> 24);
        }
    }

    static {
        for (int i = 1; i <= originsCache.length; ++i) {
            Point.originsCache[i - 1] = new Point(new double[i]){

                @Override
                public boolean isOrigin() {
                    return true;
                }
            };
        }
    }

    static class Test {
        Test() {
        }

        public static void main(String[] args) {
            Range[] ranges;
            for (Range r : ranges = new Range[]{Range.valueOf(1.1, 2.7), Range.valueOf(1.6, 2.8), Range.valueOf(-27.1, 24.81), Range.valueOf(0.0, 9.223372036854776E18), Range.valueOf(-9.223372036854776E18, -100.0), Range.valueOf(-9.223372036854776E18, -100.0), Range.valueOf(-9.223372036852776E18, -100.0), Range.valueOf(-9.223372036852776E18, 100.0), Range.valueOf(-1000.0, 1.0E8), Range.valueOf(-4.0E18, 6.0E18)}) {
                IRange ir;
                IRange cast = null;
                IRange rounded = null;
                try {
                    cast = r.toIntegerRange();
                    assert (IRange.valueOf(r).equals(cast));
                    System.out.println(String.valueOf(r) + " casted to " + String.valueOf(cast));
                }
                catch (Exception e) {
                    System.out.println("  Cannot call toIntegerRange for " + String.valueOf(r) + ": " + String.valueOf(e));
                }
                try {
                    rounded = r.toRoundedRange();
                    assert (IRange.roundOf(r).equals(rounded));
                    System.out.println(String.valueOf(r) + " rounded to " + String.valueOf(rounded));
                }
                catch (Exception e) {
                    System.out.println("  Cannot call toRoundedRange for " + String.valueOf(r) + ": " + String.valueOf(e));
                }
                try {
                    ir = IRange.valueOf(r);
                    assert (ir.equals(cast));
                    System.out.println(String.valueOf(r) + " casted to " + String.valueOf(ir));
                }
                catch (Exception e) {
                    System.out.println("  Cannot cast range " + String.valueOf(r) + ": " + String.valueOf(e));
                }
                try {
                    ir = IRange.roundOf(r);
                    assert (ir.equals(rounded));
                    System.out.println(String.valueOf(r) + " rounded to " + String.valueOf(ir));
                }
                catch (Exception e) {
                    System.out.println("  Cannot round range " + String.valueOf(r) + ": " + String.valueOf(e));
                }
            }
            System.out.println();
            Object[] points = new Point[]{Point.valueOf(12.0, 3.0), Point.valueOf(1.2, 3.0, 1.0), Point.valueOf(12.0, 3.0, 0.0), Point.valueOf(1.2, 3.0, 0.0, 1.234), Point.valueOf(12.0, 3.0, 0.0, -21234.0), Point.valueOf(-12.0, 123453.0, 27182.0, 821234.0), Point.valueOf(14.0, -3.0), Point.valueOf(0.0), Point.valueOf(0.0, 0.0), Point.valueOf(-3.0, -2.0), Point.valueOf(-4.0, 5.0), Point.valueOf(15.0, 20.0), Point.valueOf(3.0, 1.33), Point.valueOf(4.1, 5.0), Point.valueOf(0.0, 0.0, 0.0), Point.origin(3), Point.valueOf(new double[18]), Point.valueOf(new double[18]), Point.valueOf(13.0, 0.0), Point.valueOf(13.0, -0.0), Point.valueOf(4413.1, 0.1), Point.valueOf(4413.2, -0.8), Point.valueOf(13.0, 1.0, 0.0), Point.valueOf(3.0, 4.0, 0.0), Point.valueOf(13.0), Point.valueOf(1000.0, 30.0), Point.valueOf(1000.0, 1.0E20), Point.valueOf(-5.0E18, 30.0), Point.valueOf(-5.0E18, -300.0), Point.valueOf(-7.0E18, -5.0E18), Point.valueOf(5.0E18, 1.0E20)};
            Arrays.sort(points);
            for (Object rp : points) {
                System.out.println(String.valueOf(rp) + "; symmetric: " + String.valueOf(((Point)rp).symmetric()) + "; distance from origin: " + ((Point)rp).distanceFromOrigin() + " = " + ((Point)rp).distanceFrom(Arrays.asList(Point.origin(((Point)rp).coordCount()))) + (((Point)rp).coordCount() > 1 && ((Point)rp).projectionAlongAxisEquals(0, Point.origin(((Point)rp).coordCount() - 1)) ? "; x-projection is origin" : "") + "; x-shift: " + String.valueOf(((Point)rp).shiftAlongAxis(0, 100.0)) + "; x-projection: " + String.valueOf(((Point)rp).coordCount() == 1 ? "impossible" : ((Point)rp).projectionAlongAxis(0)) + "; last-axis-projection: " + String.valueOf(((Point)rp).coordCount() == 1 ? "impossible" : ((Point)rp).projectionAlongAxis(((Point)rp).coordCount() - 1)) + "; *2: " + String.valueOf(((Point)rp).multiply(2.0)) + " = " + String.valueOf(((Point)rp).scale(IPoint.valueOfEqualCoordinates(((Point)rp).coordCount(), 2L).toPoint().coordinates())) + " = " + String.valueOf(((Point)rp).scaleAndShift(Point.valueOfEqualCoordinates(((Point)rp).coordCount(), 2.0).coordinates(), Point.origin(((Point)rp).coordCount()))) + "; sqr: " + ((Point)rp).scalarProduct((Point)rp) + "; hash: " + ((Point)rp).hashCode() + "; address: " + System.identityHashCode(rp));
            }
            System.out.println();
            for (int k = 0; k < points.length - 1; k += 2) {
                IRectangularArea ira;
                System.out.print(k + ": ");
                RectangularArea ra = null;
                try {
                    ra = RectangularArea.valueOf((Point)points[k], (Point)points[k + 1]);
                    assert (RectangularArea.valueOf(ra.ranges()).equals(ra));
                    Point point = Point.valueOfEqualCoordinates(ra.coordCount(), -1.5);
                    RectangularArea test = RectangularArea.valueOf(point, Point.origin(ra.coordCount()));
                    assert (!ra.intersects(test) ? ra.intersection(test) == null : ra.intersection(test).equals(RectangularArea.valueOf(ra.min().max(test.min()), ra.max().min(test.max()))));
                    System.out.println(String.valueOf(ra) + "; ranges: " + String.valueOf(Arrays.asList(ra.ranges())) + "; contains(origin): " + ra.contains(Point.origin(ra.coordCount())) + "; expand(origin): " + String.valueOf(ra.expand(Point.origin(ra.coordCount()))) + "; expand(-1,-1..2,2): " + String.valueOf(ra.expand(RectangularArea.valueOf(Point.valueOfEqualCoordinates(ra.coordCount(), -1.0), Point.valueOfEqualCoordinates(ra.coordCount(), 2.0)))) + "; parallel distance to (-1.5,-1.5,...): " + (ra.coordCount() == 2 ? ra.parallelDistance(-1.5, -1.5) : (ra.coordCount() == 3 ? ra.parallelDistance(-1.5, -1.5, -1.5) : ra.parallelDistance(point))) + (ra.contains(point) ? " (inside)" : "") + "; intersection with " + String.valueOf(test) + ": " + String.valueOf(ra.intersection(test)) + "; subtracted " + String.valueOf(test) + ": " + String.valueOf(ra.difference(new ArrayList<RectangularArea>(), test)) + "; hash: " + ra.hashCode());
                }
                catch (Exception e) {
                    System.out.println("  Cannot create area with " + String.valueOf(points[k]) + " and " + String.valueOf(points[k + 1]) + ": " + String.valueOf(e));
                }
                if (ra == null) continue;
                IRectangularArea cast = null;
                IRectangularArea rounded = null;
                try {
                    cast = ra.toIntegerRectangularArea();
                    assert (IRectangularArea.valueOf(ra).equals(cast));
                    System.out.println(String.valueOf(ra) + " casted to " + String.valueOf(cast));
                }
                catch (Exception e) {
                    System.out.println("  Cannot call toIntegerRectangularArea for " + String.valueOf(ra) + ": " + String.valueOf(e));
                }
                try {
                    rounded = ra.toRoundedRectangularArea();
                    assert (IRectangularArea.roundOf(ra).equals(rounded));
                    IPoint point = IPoint.valueOfEqualCoordinates(ra.coordCount(), 10L);
                    IRectangularArea test = IRectangularArea.valueOf(IPoint.origin(ra.coordCount()), point);
                    assert (!rounded.intersects(test) ? rounded.intersection(test) == null : rounded.intersection(test).equals(IRectangularArea.valueOf(rounded.min().max(test.min()), rounded.max().min(test.max()))));
                    System.out.println(String.valueOf(ra) + " rounded to " + String.valueOf(rounded) + "; parallel distance to (10,10,...): " + (ra.coordCount() == 2 ? rounded.parallelDistance(10L, 10L) : (ra.coordCount() == 3 ? rounded.parallelDistance(10L, 10L, 10L) : rounded.parallelDistance(point))) + (rounded.contains(point) ? " (inside)" : "") + "; intersection with " + String.valueOf(test) + ": " + String.valueOf(rounded.intersection(test)) + "; subtracted " + String.valueOf(test) + ": " + String.valueOf(rounded.difference(new ArrayList<IRectangularArea>(), test)) + "; hash: " + rounded.hashCode());
                }
                catch (Exception e) {
                    System.out.println("  Cannot call toRoundedRectangularArea for " + String.valueOf(ra) + ": " + String.valueOf(e));
                }
                try {
                    ira = IRectangularArea.valueOf(ra);
                    assert (ira.equals(cast));
                    System.out.println(String.valueOf(ra) + " casted to " + String.valueOf(ira));
                }
                catch (Exception e) {
                    System.out.println("  Cannot cast range " + String.valueOf(ra) + ": " + String.valueOf(e));
                }
                try {
                    ira = IRectangularArea.roundOf(ra);
                    assert (ira.equals(rounded));
                    System.out.println(String.valueOf(ra) + " rounded to " + String.valueOf(ira));
                    continue;
                }
                catch (Exception e) {
                    System.out.println("  Cannot round range " + String.valueOf(ra) + ": " + String.valueOf(e));
                }
            }
        }
    }
}

