/*
 * Decompiled with CFR 0.152.
 */
package net.algart.math.functions;

import java.util.Objects;
import net.algart.math.Point;
import net.algart.math.functions.CoordinateTransformationOperator;
import net.algart.math.functions.LinearFunc;
import net.algart.math.functions.ProjectiveOperator;

public final class LinearOperator
extends ProjectiveOperator {
    LinearOperator(double[] a, double[] diagonal, double[] b) {
        super(a, diagonal, b, null, 1.0);
    }

    public static LinearOperator getInstance(double[] a, double[] b) {
        Objects.requireNonNull(a, "Null A matrix");
        Objects.requireNonNull(b, "Null b vector");
        return new LinearOperator((double[])a.clone(), null, (double[])b.clone());
    }

    public static LinearOperator getDiagonalInstance(double[] diagonal, double[] b) {
        Objects.requireNonNull(diagonal, "Null diagonal array");
        Objects.requireNonNull(b, "Null b vector");
        return new LinearOperator(null, (double[])diagonal.clone(), (double[])b.clone());
    }

    public static LinearOperator getDiagonalInstance(double ... diagonal) {
        Objects.requireNonNull(diagonal, "Null diagonal array");
        return new LinearOperator(null, (double[])diagonal.clone(), new double[diagonal.length]);
    }

    public static LinearOperator getShiftInstance(double ... b) {
        Objects.requireNonNull(b, "Null b array");
        return new LinearOperator(null, null, b);
    }

    public static LinearOperator getRotation2D(double centerX, double centerY, double angle) {
        double cos = StrictMath.cos(angle);
        double sin = StrictMath.sin(angle);
        for (int k = -4; k <= 4; ++k) {
            if (angle != (double)k / 2.0 * Math.PI) continue;
            assert (StrictMath.abs((double)StrictMath.round(cos) - cos) < 1.0E-6);
            assert (StrictMath.abs((double)StrictMath.round(sin) - sin) < 1.0E-6);
            cos = StrictMath.round(cos);
            sin = StrictMath.round(sin);
        }
        double[] a = new double[]{cos, sin, -sin, cos};
        double[] b = new double[]{centerX - a[0] * centerX - a[1] * centerY, centerY - a[2] * centerX - a[3] * centerY};
        return new LinearOperator(a, null, b);
    }

    public static LinearOperator getInstanceByPoints(Point[] q, Point[] p) {
        Objects.requireNonNull(p, "Null p argument");
        Objects.requireNonNull(q, "Null q argument");
        if (p.length != q.length) {
            throw new IllegalArgumentException("p and q point arrays lengths mismatch: p.length=" + p.length + ", q.length=" + q.length);
        }
        if (p.length == 0) {
            throw new IllegalArgumentException("Empty p and q arrays");
        }
        int n = p.length - 1;
        if (((long)n + 1L) * ((long)n + 1L) > Integer.MAX_VALUE) {
            throw new OutOfMemoryError("Too large necessary matrix (more than Integer.MAX_VALUE elements)");
        }
        for (int k = 0; k < p.length; ++k) {
            if (p[k].coordCount() != n) {
                throw new IllegalArgumentException("n+1 n-dimensional points are necessary to find the linear operator, but we have " + (n + 1) + " points, and the source point #" + k + " is " + p[k].coordCount() + "-dimensional");
            }
            if (q[k].coordCount() == n) continue;
            throw new IllegalArgumentException("n+1 n-dimensional points are necessary to find the linear operator, but we have " + (n + 1) + " points, and the destination point #" + k + " is " + q[k].coordCount() + "-dimensional");
        }
        double[] s = new double[(n + 1) * (n + 1)];
        double[] t = new double[n + 1];
        double[] v = new double[n + 1];
        int sOfs = 0;
        for (int i = 0; i <= n; ++i) {
            int j = 0;
            while (j < n) {
                s[sOfs] = p[i].coord(j);
                ++j;
                ++sOfs;
            }
            s[sOfs++] = 1.0;
        }
        double[] a = new double[n * n];
        double[] b = new double[n];
        int i = 0;
        int aOfs = 0;
        while (i < n) {
            for (int j = 0; j <= n; ++j) {
                t[j] = q[j].coord(i);
            }
            LinearOperator.solveLinearEquationsSet(v, (double[])s.clone(), t);
            System.arraycopy(v, 0, a, aOfs, n);
            b[i] = v[n];
            ++i;
            aOfs += n;
        }
        return new LinearOperator(a, null, b);
    }

    public LinearOperator superposition(LinearOperator operator) {
        double sum;
        Objects.requireNonNull(operator, "Null operator argument");
        if (operator.n != this.n) {
            throw new IllegalArgumentException("Passed and this operators dimensions mismatch: operator.n()=" + operator.n + ", this.n()=" + this.n);
        }
        if (operator.isShift()) {
            double[] b = (double[])this.b.clone();
            for (int i = 0; i < this.n; ++i) {
                int n = i;
                b[n] = b[n] + operator.b[i];
            }
            return new LinearOperator(this.a, this.diagonal, b);
        }
        double[] a = this.a != null ? this.a : this.a();
        double[] aOther = operator.a != null ? operator.a : operator.a();
        double[] aNew = new double[a.length];
        double[] bNew = new double[this.n];
        for (int in = 0; in < a.length; in += this.n) {
            for (int j = 0; j < this.n; ++j) {
                sum = 0.0;
                int k = 0;
                int disp = j;
                while (k < this.n) {
                    sum += aOther[in + k] * a[disp];
                    ++k;
                    disp += this.n;
                }
                aNew[in + j] = sum;
            }
        }
        int i = 0;
        int in = 0;
        while (i < this.n) {
            sum = operator.b[i];
            for (int j = 0; j < this.n; ++j) {
                sum += aOther[in + j] * this.b[j];
            }
            bNew[i] = sum;
            ++i;
            in += this.n;
        }
        return new LinearOperator(aNew, null, bNew);
    }

    public LinearOperator changeB(double ... b) {
        Objects.requireNonNull(b, "Null b array");
        if (b.length != this.n) {
            throw new IllegalArgumentException("Passed b and this.b vector lengths mismatch: b.length=" + b.length + ", this b.length=" + this.n);
        }
        return new LinearOperator(this.a, this.diagonal, b);
    }

    @Override
    public void map(double[] destPoint, double[] srcPoint) {
        this.calculateAxPlusB(destPoint, srcPoint);
    }

    public void inverseMap(double[] srcPoint, double[] destPoint) {
        Objects.requireNonNull(srcPoint, "Null srcPoint");
        Objects.requireNonNull(destPoint, "Null destPoint");
        if (srcPoint.length != this.n) {
            throw new IllegalArgumentException("Illegal length of srcPoint array: " + srcPoint.length + " for " + String.valueOf(this));
        }
        if (destPoint.length != this.n) {
            throw new IllegalArgumentException("Illegal length of destPoint array: " + destPoint.length + " for " + String.valueOf(this));
        }
        if (this.a != null) {
            double[] yMinusB = new double[this.n];
            for (int i = 0; i < this.n; ++i) {
                yMinusB[i] = destPoint[i] - this.b[i];
            }
            double[] aClone = (double[])this.a.clone();
            LinearOperator.solveLinearEquationsSet(srcPoint, aClone, yMinusB);
        } else if (this.diagonal != null) {
            for (int i = 0; i < this.n; ++i) {
                srcPoint[i] = (destPoint[i] - this.b[i]) / this.diagonal[i];
            }
        } else {
            for (int i = 0; i < this.n; ++i) {
                srcPoint[i] = destPoint[i] - this.b[i];
            }
        }
    }

    public static void solveLinearEquationsSet(double[] x, double[] a, double[] y) {
        Objects.requireNonNull(x, "Null x");
        Objects.requireNonNull(y, "Null y");
        Objects.requireNonNull(a, "Null a");
        int n = x.length;
        if (y.length != n) {
            throw new IllegalArgumentException("x and y vector lengths mismatch: x.length=" + n + ", y.length=" + y.length);
        }
        if ((long)a.length != (long)n * (long)n) {
            throw new IllegalArgumentException("Illegal size of A matrix: a.length=" + a.length + " must be equal to x.length^2=" + (long)n * (long)n);
        }
        for (int k = 0; k < n - 1; ++k) {
            int aOfsDiagonal = k * n + k;
            int pivotIndex = k;
            double pivot = a[aOfsDiagonal];
            double pivotAbs = StrictMath.abs(pivot);
            int i = k + 1;
            int aOfs = aOfsDiagonal + n;
            while (i < n) {
                double v = a[aOfs];
                double vAbs = StrictMath.abs(v);
                if (vAbs > pivotAbs) {
                    pivot = v;
                    pivotAbs = vAbs;
                    pivotIndex = i;
                }
                ++i;
                aOfs += n;
            }
            if (pivotAbs == 0.0) {
                for (int j = 0; j < n; ++j) {
                    x[j] = Double.NaN;
                }
                return;
            }
            if (pivotIndex != k) {
                double temp = y[k];
                y[k] = y[pivotIndex];
                y[pivotIndex] = temp;
                int j = k;
                int aOfsK = aOfsDiagonal;
                int aOfs2 = pivotIndex * n + k;
                while (j < n) {
                    temp = a[aOfsK];
                    a[aOfsK] = a[aOfs2];
                    a[aOfs2] = temp;
                    ++j;
                    ++aOfsK;
                    ++aOfs2;
                }
            }
            assert (a[aOfsDiagonal] == pivot) : "Pivot element is not placed to the correct place (k = " + k + ")";
            aOfs = k * n + n;
            for (i = k + 1; i < n; ++i) {
                double q = a[aOfs + k] / pivot;
                int n2 = i;
                y[n2] = y[n2] - y[k] * q;
                int j = 0;
                while (j < k) {
                    assert (a[aOfs] == 0.0) : "The line was not filled by zero in previous iterations 0,...," + (k - 1);
                    ++j;
                    ++aOfs;
                }
                a[aOfs++] = 0.0;
                int aOfsK = aOfsDiagonal + 1;
                assert (aOfs == i * n + k + 1);
                assert (aOfsK == k * n + k + 1);
                int j2 = k + 1;
                while (j2 < n) {
                    int n3 = aOfs++;
                    a[n3] = a[n3] - a[aOfsK] * q;
                    ++j2;
                    ++aOfsK;
                }
            }
        }
        int aOfs = n * n - 1;
        for (int i = n - 1; i >= 0; --i) {
            assert (aOfs == i * n + n - 1);
            double v = y[i];
            int j = n - 1;
            while (j > i) {
                v -= a[aOfs] * x[j];
                --j;
                --aOfs;
            }
            x[i] = v / a[aOfs];
            aOfs -= i + 1;
        }
    }

    @Override
    public String toString() {
        boolean shift = this.isShift();
        StringBuilder sA = new StringBuilder();
        if (this.isDiagonal()) {
            if (this.diagonal != null) {
                sA.append("diag[");
                for (int k = 0; k < this.diagonal.length; ++k) {
                    if (k > 0) {
                        sA.append(",");
                    }
                    sA.append(LinearFunc.goodFormat(this.diagonal[k]));
                }
                sA.append("]");
            }
        } else {
            sA.append("A");
        }
        StringBuilder sB = new StringBuilder();
        for (int k = 0; k < this.n; ++k) {
            if (k > 0) {
                sB.append(",");
            }
            sB.append(LinearFunc.goodFormat(this.b[k]));
        }
        return "linear (affine) " + this.n + "-dimensional operator " + String.valueOf(sA) + "x+b, b=(" + String.valueOf(sB) + ")" + (shift ? " (shift)" : "");
    }

    static void printEquationsSet(double[] x, double[] a, double[] y) {
        int n = y.length;
        System.out.println();
        for (int i = 0; i < n; ++i) {
            double v = 0.0;
            for (int j = 0; j < n; ++j) {
                if (x != null) {
                    v += a[i * n + j] * x[j];
                }
                System.out.printf("%10.6f*x[%d] " + (j < n - 1 ? "+ " : "= "), a[i * n + j], j);
            }
            System.out.printf("%.7f", y[i]);
            if (x != null) {
                System.out.printf(" (error %.7f)", v - y[i]);
            }
            System.out.println();
        }
    }

    static class Test
    extends ProjectiveOperator.Test {
        Test() {
        }

        @Override
        int numberOfPoints(int dimCount) {
            return dimCount + 1;
        }

        @Override
        CoordinateTransformationOperator getOperator() {
            return LinearOperator.getInstanceByPoints(this.q, this.p);
        }

        void badLinearEquationsSetTest() {
            double factorial = 1.0;
            for (int k = 2; k <= 16; ++k) {
                factorial *= (double)k;
            }
            double[] a = new double[this.dimCount * this.dimCount];
            int aOfs = 0;
            for (int i = 1; i <= this.dimCount; ++i) {
                int j = 1;
                while (j <= this.dimCount) {
                    a[aOfs] = factorial / (double)(i + j);
                    ++j;
                    ++aOfs;
                }
            }
            LinearOperator lo = LinearOperator.getInstance(a, new double[this.dimCount]);
            this.newRndPoints(10, this.dimCount);
            for (int k = 0; k < this.p.length; ++k) {
                double[] srcPoint = this.p[k].coordinates();
                double[] destPoint = new double[srcPoint.length];
                this.q[k] = this.p[k];
                lo.map(destPoint, srcPoint);
                lo.inverseMap(srcPoint, destPoint);
                this.r[k] = Point.valueOf(srcPoint);
            }
            double error = this.maxDiff();
            System.out.println("Difference for \"bad\" equation set " + this.dimCount + "*" + this.dimCount + " = " + error);
            LinearOperator lo2 = LinearOperator.getInstance(a, new double[this.dimCount]);
            if (!lo.equals(lo2)) {
                throw new AssertionError((Object)"Error in equals");
            }
            ProjectiveOperator po = ProjectiveOperator.getInstance(a, lo.b(), new double[this.dimCount], 1.0);
            if (!lo.equals(po)) {
                throw new AssertionError((Object)"Error in equals");
            }
        }

        public static void main(String[] args) {
            Test test = new Test();
            test.init(args);
            test.badLinearEquationsSetTest();
            test.mainTest();
        }
    }
}

