/*
 * Decompiled with CFR 0.152.
 */
package net.algart.math;

import java.util.Arrays;
import java.util.Collection;
import java.util.Objects;
import java.util.zip.CRC32;
import net.algart.math.IRange;
import net.algart.math.IRectangularArea;
import net.algart.math.Point;

public class IPoint
implements Comparable<IPoint> {
    private static final IPoint[] originsCache = new IPoint[16];
    final long[] coordinates;

    IPoint(long[] coordinates) {
        this.coordinates = coordinates;
    }

    public static IPoint valueOf(long ... coordinates) {
        Objects.requireNonNull(coordinates, "Null coordinates argument");
        if (coordinates.length == 0) {
            throw new IllegalArgumentException("Empty coordinates array");
        }
        if (coordinates.length <= originsCache.length) {
            boolean origin = true;
            for (long coord : coordinates) {
                if (coord == 0L) continue;
                origin = false;
                break;
            }
            if (origin) {
                return originsCache[coordinates.length - 1];
            }
        }
        return new IPoint((long[])coordinates.clone());
    }

    public static IPoint valueOf(long x) {
        return x == 0L ? originsCache[0] : new IPoint(new long[]{x});
    }

    public static IPoint valueOf(long x, long y) {
        return x == 0L && y == 0L ? originsCache[1] : new IPoint(new long[]{x, y});
    }

    public static IPoint valueOf(long x, long y, long z) {
        return x == 0L && y == 0L && z == 0L ? originsCache[2] : new IPoint(new long[]{x, y, z});
    }

    public static IPoint valueOfEqualCoordinates(int coordCount, long filler) {
        if (filler == 0L) {
            return IPoint.origin(coordCount);
        }
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        long[] coordinates = new long[coordCount];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = filler;
        }
        return new IPoint(coordinates);
    }

    public static IPoint origin(int coordCount) {
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        if (coordCount <= originsCache.length) {
            return originsCache[coordCount - 1];
        }
        return new IPoint(new long[coordCount]);
    }

    public static IPoint minValue(int coordCount) {
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        long[] coordinates = new long[coordCount];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = Long.MIN_VALUE;
        }
        return new IPoint(coordinates);
    }

    public static IPoint maxValue(int coordCount) {
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        long[] coordinates = new long[coordCount];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = Long.MAX_VALUE;
        }
        return new IPoint(coordinates);
    }

    public static IPoint valueOf(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        long[] coordinates = new long[point.coordCount()];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = (long)point.coord(k);
        }
        return new IPoint(coordinates);
    }

    public static IPoint roundOf(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        long[] coordinates = new long[point.coordCount()];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = StrictMath.round(point.coord(k));
        }
        return new IPoint(coordinates);
    }

    public int coordCount() {
        return this.coordinates.length;
    }

    public long[] coordinates() {
        return (long[])this.coordinates.clone();
    }

    public long[] coordinates(long[] result) {
        if (result.length < this.coordinates.length) {
            throw new IllegalArgumentException("Too short result array: long[" + result.length + "]; " + this.coordinates.length + " elements required to store coordinates");
        }
        System.arraycopy(this.coordinates, 0, result, 0, this.coordinates.length);
        return result;
    }

    public long coord(int coordIndex) {
        return this.coordinates[coordIndex];
    }

    public long x() {
        return this.coordinates[0];
    }

    public long y() {
        if (this.coordinates.length < 2) {
            throw new IllegalStateException("Cannot get y-coordinate of " + this.coordinates.length + "-dimensional point");
        }
        return this.coordinates[1];
    }

    public long z() {
        if (this.coordinates.length < 3) {
            throw new IllegalStateException("Cannot get z-coordinate of " + this.coordinates.length + "-dimensional point");
        }
        return this.coordinates[2];
    }

    public boolean isOrigin() {
        for (long coord : this.coordinates) {
            if (coord == 0L) continue;
            return false;
        }
        return true;
    }

    public double distanceFromOrigin() {
        if (this.coordinates.length == 1) {
            return StrictMath.abs(this.coordinates[0]);
        }
        double dSqr = 0.0;
        for (long coord : this.coordinates) {
            dSqr += (double)coord * (double)coord;
        }
        return StrictMath.sqrt(dSqr);
    }

    public double distanceFrom(Collection<IPoint> points) {
        Objects.requireNonNull(points, "Null points argument");
        double result = Double.POSITIVE_INFINITY;
        for (IPoint point : points) {
            double d;
            if (point.coordCount() != this.coordinates.length) {
                throw new IllegalArgumentException("Dimensions count mismatch: some of the passed points has " + point.coordCount() + " dimensions instead of " + this.coordinates.length);
            }
            if (this.coordinates.length == 1) {
                d = StrictMath.abs((double)this.coordinates[0] - (double)point.coordinates[0]);
            } else if (this.coordinates.length == 2) {
                d = StrictMath.hypot((double)this.coordinates[0] - (double)point.coordinates[0], (double)this.coordinates[1] - (double)point.coordinates[1]);
            } else {
                double dSqr = 0.0;
                for (int k = 0; k < this.coordinates.length; ++k) {
                    double diff = (double)this.coordinates[k] - (double)point.coordinates[k];
                    dSqr += diff * diff;
                }
                d = StrictMath.sqrt(dSqr);
            }
            if (!(d < result)) continue;
            result = d;
        }
        return result;
    }

    public IPoint add(IPoint point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] + point.coordinates[k];
        }
        return new IPoint(coordinates);
    }

    public IPoint subtract(IPoint point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] - point.coordinates[k];
        }
        return new IPoint(coordinates);
    }

    public IPoint min(IPoint point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = Math.min(this.coordinates[k], point.coordinates[k]);
        }
        return new IPoint(coordinates);
    }

    public IPoint max(IPoint point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = Math.max(this.coordinates[k], point.coordinates[k]);
        }
        return new IPoint(coordinates);
    }

    public IPoint addToAllCoordinates(long increment) {
        if (increment == 0L) {
            return this;
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] + increment;
        }
        return new IPoint(coordinates);
    }

    public IPoint roundedMultiply(double multiplier) {
        if (multiplier == 0.0) {
            return IPoint.origin(this.coordinates.length);
        }
        if (multiplier == 1.0) {
            return this;
        }
        if (multiplier == -1.0) {
            return this.symmetric();
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = StrictMath.round((double)this.coordinates[k] * multiplier);
        }
        return new IPoint(coordinates);
    }

    public IPoint roundedScale(double ... multipliers) {
        Objects.requireNonNull(multipliers, "Null multipliers argument");
        if (multipliers.length != this.coordinates.length) {
            throw new IllegalArgumentException("Illegal number of multipliers: " + multipliers.length + " instead of " + this.coordinates.length);
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = multipliers[k] == 0.0 ? 0L : (multipliers[k] == 1.0 ? this.coordinates[k] : (multipliers[k] == -1.0 ? -this.coordinates[k] : StrictMath.round((double)this.coordinates[k] * multipliers[k])));
        }
        return new IPoint(coordinates);
    }

    public Point scaleAndShift(double[] multipliers, Point shift) {
        double[] coordinates = new double[this.coordinates.length];
        this.scaleAndShift(coordinates, multipliers, shift);
        return new Point(coordinates);
    }

    public void scaleAndShift(double[] resultCoordinates, double[] multipliers, Point shift) {
        if (resultCoordinates.length < this.coordinates.length) {
            throw new IllegalArgumentException("Too short result coordinates array: double[" + resultCoordinates.length + "]; " + this.coordinates.length + " elements required to store coordinates");
        }
        Objects.requireNonNull(multipliers, "Null multipliers argument");
        if (multipliers.length != this.coordinates.length) {
            throw new IllegalArgumentException("Illegal number of multipliers: " + multipliers.length + " instead of " + this.coordinates.length);
        }
        Objects.requireNonNull(shift, "Null shift argument");
        if (shift.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + shift.coordCount() + " instead of " + this.coordinates.length);
        }
        for (int k = 0; k < this.coordinates.length; ++k) {
            resultCoordinates[k] = multipliers[k] == 0.0 ? shift.coord(k) : (multipliers[k] == 1.0 ? shift.coord(k) + (double)this.coordinates[k] : (multipliers[k] == -1.0 ? shift.coord(k) - (double)this.coordinates[k] : shift.coord(k) + (double)this.coordinates[k] * multipliers[k]));
        }
    }

    public IPoint shiftAlongAxis(int coordIndex, long shift) {
        this.coord(coordIndex);
        if (shift == 0L) {
            return this;
        }
        long[] coordinates = (long[])this.coordinates.clone();
        int n = coordIndex;
        coordinates[n] = coordinates[n] + shift;
        return new IPoint(coordinates);
    }

    public double scalarProduct(IPoint point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double result = 0.0;
        for (int k = 0; k < this.coordinates.length; ++k) {
            result += (double)this.coordinates[k] * (double)point.coordinates[k];
        }
        return result;
    }

    public IPoint symmetric() {
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = -this.coordinates[k];
        }
        return new IPoint(coordinates);
    }

    public IPoint projectionAlongAxis(int coordIndex) {
        this.coord(coordIndex);
        if (this.coordinates.length == 1) {
            throw new IllegalStateException("Cannot perform projection of 1-dimensional figures");
        }
        long[] coordinates = new long[this.coordinates.length - 1];
        System.arraycopy(this.coordinates, 0, coordinates, 0, coordIndex);
        System.arraycopy(this.coordinates, coordIndex + 1, coordinates, coordIndex, coordinates.length - coordIndex);
        return new IPoint(coordinates);
    }

    boolean projectionAlongAxisEquals(int coordIndex, IPoint point) {
        int k;
        this.coord(coordIndex);
        if (this.coordinates.length == 1) {
            throw new IllegalStateException("Cannot perform projection of 1-dimensional figures");
        }
        if (this.coordinates.length != point.coordCount() + 1) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " is not equal to " + this.coordinates.length + "-1");
        }
        for (k = 0; k < coordIndex; ++k) {
            if (this.coordinates[k] == point.coordinates[k]) continue;
            return false;
        }
        for (k = coordIndex + 1; k < this.coordinates.length; ++k) {
            if (this.coordinates[k] == point.coordinates[k - 1]) continue;
            return false;
        }
        return true;
    }

    @Override
    public int compareTo(IPoint o) {
        return this.compareTo(o, 0);
    }

    public int compareTo(IPoint o, int firstCoordIndex) {
        int n = Math.max(this.coordinates.length, o.coordinates.length);
        if (firstCoordIndex < 0) {
            throw new IllegalArgumentException("Negative firstCoordIndex argument");
        }
        firstCoordIndex %= n;
        for (int k = n - 1; k >= 0; --k) {
            long otherCoord;
            int index = k + firstCoordIndex;
            if (index >= n) {
                index -= n;
            }
            long thisCoord = index >= this.coordinates.length ? 0L : this.coordinates[index];
            long l = otherCoord = index >= o.coordinates.length ? 0L : o.coordinates[index];
            if (thisCoord > otherCoord) {
                return 1;
            }
            if (thisCoord >= otherCoord) continue;
            return -1;
        }
        return Integer.compare(this.coordinates.length, o.coordinates.length);
    }

    public String toString() {
        StringBuilder result = new StringBuilder("(");
        for (int k = 0; k < this.coordinates.length; ++k) {
            if (k > 0) {
                result.append(", ");
            }
            result.append(this.coordinates[k]);
        }
        result.append(")");
        return result.toString();
    }

    public int hashCode() {
        CRC32 sum = new CRC32();
        byte[] bytes = new byte[this.coordinates.length * 8];
        IPoint.getBytes(this.coordinates, bytes);
        sum.update(bytes, 0, bytes.length);
        return (int)sum.getValue() ^ 0x1C11DD;
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof IPoint)) {
            return false;
        }
        IPoint ip = (IPoint)obj;
        if (ip.coordinates.length != this.coordinates.length) {
            return false;
        }
        for (int k = 0; k < this.coordinates.length; ++k) {
            if (ip.coordinates[k] == this.coordinates[k]) continue;
            return false;
        }
        return true;
    }

    public IPoint addExact(IPoint point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = IPoint.addExact(this.coordinates[k], point.coordinates[k]);
        }
        return new IPoint(coordinates);
    }

    public IPoint subtractExact(IPoint point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = IPoint.subtractExact(this.coordinates[k], point.coordinates[k]);
        }
        return new IPoint(coordinates);
    }

    public IPoint multiply(double multiplier) {
        if (multiplier == 0.0) {
            return IPoint.origin(this.coordinates.length);
        }
        if (multiplier == 1.0) {
            return this;
        }
        if (multiplier == -1.0) {
            return this.symmetric();
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = (long)((double)this.coordinates[k] * multiplier);
        }
        return new IPoint(coordinates);
    }

    public IPoint scale(double ... multipliers) {
        Objects.requireNonNull(multipliers, "Null multipliers argument");
        if (multipliers.length != this.coordinates.length) {
            throw new IllegalArgumentException("Illegal number of multipliers: " + multipliers.length + " instead of " + this.coordinates.length);
        }
        long[] coordinates = new long[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = multipliers[k] == 0.0 ? 0L : (multipliers[k] == 1.0 ? this.coordinates[k] : (multipliers[k] == -1.0 ? -this.coordinates[k] : (long)((double)this.coordinates[k] * multipliers[k])));
        }
        return new IPoint(coordinates);
    }

    public long toOneDimensional(long[] dimensions, boolean pseudoCyclicTruncation) {
        if (pseudoCyclicTruncation) {
            long result;
            int n = Math.min(this.coordinates.length, dimensions.length);
            if (n == 0) {
                return 0L;
            }
            if (dimensions[--n] < 0L) {
                throw new IllegalArgumentException("Negative dimensions[" + n + "]");
            }
            long limit = dimensions[n];
            long l = result = limit == 0L ? 0L : this.coordinates[n] % limit;
            if (result < 0L) {
                result += limit;
            }
            while (n > 0) {
                long coord;
                if (dimensions[--n] < 0L) {
                    throw new IllegalArgumentException("Negative dimensions[" + n + "]");
                }
                long l2 = coord = (limit *= dimensions[n]) == 0L ? 0L : this.coordinates[n] % limit;
                if (coord < 0L) {
                    coord += limit;
                }
                result *= dimensions[n];
                if ((result += coord) >= 0L && result < limit) continue;
                result -= limit;
            }
            return result;
        }
        int n = this.coordinates.length - 1;
        if (n < dimensions.length && dimensions[n] < 0L) {
            throw new IllegalArgumentException("Negative dimensions[" + n + "]");
        }
        long result = this.coordinates[n];
        for (int k = n - 1; k >= 0; --k) {
            if (k >= dimensions.length) {
                result += this.coordinates[k];
                continue;
            }
            if (dimensions[k] < 0L) {
                throw new IllegalArgumentException("Negative dimensions[" + k + "]");
            }
            result = result * dimensions[k] + this.coordinates[k];
        }
        return result;
    }

    public Point toPoint() {
        return Point.valueOf(this);
    }

    private static void getBytes(long[] array, byte[] result) {
        int disp = 0;
        for (int k = 0; k < array.length; ++k) {
            long l = array[k];
            int value = (int)l ^ 0xAE23256;
            result[disp++] = (byte)value;
            result[disp++] = (byte)(value >>> 8);
            result[disp++] = (byte)(value >>> 16);
            result[disp++] = (byte)(value >>> 24);
            value = (int)(l >>> 32) ^ 0x2ED8234B;
            result[disp++] = (byte)value;
            result[disp++] = (byte)(value >>> 8);
            result[disp++] = (byte)(value >>> 16);
            result[disp++] = (byte)(value >>> 24);
        }
    }

    static long addExact(long x, long y) {
        long r = x + y;
        if (((x ^ r) & (y ^ r)) < 0L) {
            throw new ArithmeticException("long overflow");
        }
        return r;
    }

    static long subtractExact(long x, long y) {
        long r = x - y;
        if (((x ^ y) & (x ^ r)) < 0L) {
            throw new ArithmeticException("long overflow");
        }
        return r;
    }

    static {
        for (int i = 1; i <= originsCache.length; ++i) {
            IPoint.originsCache[i - 1] = new IPoint(new long[i]){

                @Override
                public boolean isOrigin() {
                    return true;
                }
            };
        }
    }

    static class Test {
        Test() {
        }

        /*
         * WARNING - void declaration
         */
        public static void main(String[] args) {
            void var3_6;
            Object[] p = new IPoint[]{IPoint.valueOf(12L, 3L), IPoint.valueOf(12L, 3L, 1L), IPoint.valueOf(12L, 3L, 0L), IPoint.valueOf(12L, 3L, 0L, 1234L), IPoint.valueOf(12L, 3L, 0L, -21234L), IPoint.valueOf(-12L, 123453L, 27182L, 821234L), IPoint.valueOf(14L, -3L), IPoint.valueOf(0L), IPoint.valueOf(0L, 0L), IPoint.valueOf(0L, 2L), IPoint.valueOf(0L, 1L), IPoint.valueOf(-1L, -14L), IPoint.valueOf(-1L, 1L), IPoint.valueOf(1L, 4L), IPoint.valueOf(1L, 1L), IPoint.valueOf(2L, 4L), IPoint.valueOf(2L, 3L), IPoint.valueOf(0L, 0L, 0L), IPoint.origin(3), IPoint.valueOf(new long[18]), IPoint.valueOf(new long[18]), IPoint.valueOf(13L, 0L), IPoint.valueOf(-13L, 0L), IPoint.valueOf(13L, 0L, 1L), IPoint.valueOf(3L, 4L, 0L), IPoint.valueOf(13L), IPoint.valueOf(Long.MIN_VALUE, Long.MIN_VALUE), IPoint.valueOf(100L, Long.MAX_VALUE), IPoint.valueOf(-9223372036854775807L, -2L), IPoint.valueOf(-9223372036854775807L, -2L), IPoint.valueOf(-9223372036854775807L, -3L)};
            Arrays.sort(p);
            long[] dimensions = new long[]{10L, 10L, 10L};
            for (long[] ends : new long[][]{{0L, 10L}, {0L, Long.MAX_VALUE}, {Long.MIN_VALUE, -100L}, {-9223372036854775807L, -100L}, {-9223372036854775806L, -100L}, {-9223372036854775806L, 100L}}) {
                System.out.println("Range " + ends[0] + ".." + ends[1] + " is " + (String)(IRange.isAllowedRange(ends[0], ends[1]) ? "allowed: " + String.valueOf(IRange.valueOf(ends[0], ends[1])) : "not allowed"));
            }
            for (Object ip : p) {
                System.out.println(String.valueOf(ip) + "; symmetric: " + String.valueOf(((IPoint)ip).symmetric()) + "; distance from origin: " + ((IPoint)ip).distanceFromOrigin() + " = " + ((IPoint)ip).distanceFrom(Arrays.asList(IPoint.origin(((IPoint)ip).coordCount()))) + (((IPoint)ip).coordCount() > 1 && ((IPoint)ip).projectionAlongAxisEquals(0, IPoint.origin(((IPoint)ip).coordCount() - 1)) ? "; x-projection is origin" : "") + "; x-shift: " + String.valueOf(((IPoint)ip).shiftAlongAxis(0, 100L)) + "; x-projection: " + String.valueOf(((IPoint)ip).coordCount() == 1 ? "impossible" : ((IPoint)ip).projectionAlongAxis(0)) + "; last-axis-projection: " + String.valueOf(((IPoint)ip).coordCount() == 1 ? "impossible" : ((IPoint)ip).projectionAlongAxis(((IPoint)ip).coordCount() - 1)) + "; shift in 10x10x10: " + ((IPoint)ip).toOneDimensional(dimensions, true) + "; *1.1: " + String.valueOf(((IPoint)ip).multiply(1.1)) + " = " + String.valueOf(((IPoint)ip).scale(Point.valueOfEqualCoordinates(((IPoint)ip).coordCount(), 1.1).coordinates())) + "; round *1.1: " + String.valueOf(((IPoint)ip).roundedMultiply(1.1)) + " = " + String.valueOf(((IPoint)ip).roundedScale(Point.valueOfEqualCoordinates(((IPoint)ip).coordCount(), 1.1).coordinates())) + " ~ " + String.valueOf(((IPoint)ip).scaleAndShift(Point.valueOfEqualCoordinates(((IPoint)ip).coordCount(), 1.1).coordinates(), Point.origin(((IPoint)ip).coordCount()))) + "; sqr: " + ((IPoint)ip).scalarProduct((IPoint)ip) + "; hash: " + ((IPoint)ip).hashCode() + "; address: " + System.identityHashCode(ip));
            }
            System.out.println();
            boolean bl = false;
            while (var3_6 < p.length - 1) {
                try {
                    IRectangularArea ra = IRectangularArea.valueOf((IPoint)p[var3_6], (IPoint)p[var3_6 + true]);
                    assert (IRectangularArea.valueOf(ra.ranges()).equals(ra));
                    System.out.println(String.valueOf(ra) + "; ranges: " + String.valueOf(Arrays.asList(ra.ranges())) + "; contains(origin): " + ra.contains(IPoint.origin(ra.coordCount())) + "; expand(origin): " + String.valueOf(ra.expand(IPoint.origin(ra.coordCount()))) + "; expand(-1,-1..2,2): " + String.valueOf(ra.expand(IRectangularArea.valueOf(IPoint.valueOfEqualCoordinates(ra.coordCount(), -1L), IPoint.valueOfEqualCoordinates(ra.coordCount(), 2L)))) + " hash: " + ra.hashCode());
                }
                catch (Exception e) {
                    System.out.println("  Cannot create area with " + String.valueOf(p[var3_6]) + " and " + String.valueOf(p[var3_6 + true]) + ": " + String.valueOf(e));
                }
                var3_6 += 2;
            }
        }
    }
}

