/*
 * Decompiled with CFR 0.152.
 */
package net.algart.math.rectangles;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import net.algart.arrays.Arrays;
import net.algart.math.IPoint;
import net.algart.math.IRectangularArea;
import net.algart.math.rectangles.HorizontalBoundaryIBracketSet;
import net.algart.math.rectangles.HorizontalIBracketSet;
import net.algart.math.rectangles.IBracket;
import net.algart.math.rectangles.SearchIRectangleInHypograph;

public class IRectanglesUnion {
    static final int DEBUG_LEVEL = Arrays.SystemSettings.getIntProperty("net.algart.math.rectangles.debugLevel", 0);
    private static final boolean USE_SECOND_SIDES_WHILE_SEARCHING_CONNECTIONS = false;
    private final List<Frame> frames;
    private final IRectangularArea circumscribedRectangle;
    private List<HorizontalSide> horizontalSides = null;
    private List<VerticalSide> verticalSides = null;
    private List<List<Frame>> connectedComponents = null;
    private List<HorizontalSideSeries> horizontalSideSeries = null;
    private List<VerticalSideSeries> verticalSideSeries = null;
    private List<HorizontalSideSeries> horizontalSideSeriesAtBoundary = null;
    private List<VerticalSideSeries> verticalSideSeriesAtBoundary = null;
    private long[] allDifferentXAtBoundary = null;
    private double unionArea = Double.NaN;
    private List<List<BoundaryLink>> allBoundaries = null;
    private List<HorizontalSection> horizontalSectionsByLowerSides = null;
    private IRectangularArea largestRectangleInUnion = null;
    private final Object lock = new Object();

    IRectanglesUnion(List<Frame> frames) {
        this.frames = frames;
        if (frames.isEmpty()) {
            this.circumscribedRectangle = null;
        } else {
            long minX = Long.MAX_VALUE;
            long minY = Long.MAX_VALUE;
            long maxX = Long.MIN_VALUE;
            long maxY = Long.MIN_VALUE;
            for (Frame frame : frames) {
                minX = Math.min(minX, frame.fromX);
                minY = Math.min(minY, frame.fromY);
                maxX = Math.max(maxX, frame.toX - 1L);
                maxY = Math.max(maxY, frame.toY - 1L);
            }
            this.circumscribedRectangle = IRectangularArea.valueOf(minX, minY, maxX, maxY);
        }
    }

    public static IRectanglesUnion newInstance(Collection<IRectangularArea> rectangles) {
        return new IRectanglesUnion(IRectanglesUnion.checkAndConvertToFrames(rectangles));
    }

    public IRectanglesUnion subtractRectangle(IRectangularArea whatToSubtract) {
        Objects.requireNonNull(whatToSubtract, "Null rectangle");
        if (whatToSubtract.coordCount() != 2) {
            throw new IllegalArgumentException("Only 2-dimensional rectangle can be subtracted");
        }
        LinkedList<IRectangularArea> rectangles = new LinkedList<IRectangularArea>();
        for (Frame frame : this.frames) {
            rectangles.add(frame.rectangle);
        }
        IRectangularArea.subtractCollection(rectangles, whatToSubtract);
        return IRectanglesUnion.newInstance(rectangles);
    }

    public IRectanglesUnion subtractLargestRectangle() {
        if (this.frames.isEmpty()) {
            return this;
        }
        return this.subtractRectangle(this.largestRectangleInUnion());
    }

    public List<Frame> frames() {
        return Collections.unmodifiableList(this.frames);
    }

    public IRectangularArea circumscribedRectangle() {
        return this.circumscribedRectangle;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<HorizontalSide> horizontalSides() {
        Object object = this.lock;
        synchronized (object) {
            return Collections.unmodifiableList(this.horizontalSides);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<VerticalSide> verticalSides() {
        Object object = this.lock;
        synchronized (object) {
            return Collections.unmodifiableList(this.verticalSides);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int connectedComponentCount() {
        Object object = this.lock;
        synchronized (object) {
            this.findConnectedComponents();
            return this.connectedComponents.size();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public IRectanglesUnion connectedComponent(int index) {
        Object object = this.lock;
        synchronized (object) {
            this.findConnectedComponents();
            List<Frame> resultFrames = IRectanglesUnion.cloneFrames((Collection<Frame>)this.connectedComponents.get(index));
            IRectanglesUnion result = new IRectanglesUnion(resultFrames);
            result.connectedComponents = Collections.singletonList(resultFrames);
            return result;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<HorizontalBoundaryLink> allHorizontalBoundaryLinks() {
        Object object = this.lock;
        synchronized (object) {
            this.findBoundaries();
            ArrayList<HorizontalBoundaryLink> result = new ArrayList<HorizontalBoundaryLink>();
            for (HorizontalSideSeries series : this.horizontalSideSeriesAtBoundary) {
                result.addAll(series.containedBoundaryLinks);
            }
            return result;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<VerticalBoundaryLink> allVerticalBoundaryLinks() {
        Object object = this.lock;
        synchronized (object) {
            this.findBoundaries();
            ArrayList<VerticalBoundaryLink> result = new ArrayList<VerticalBoundaryLink>();
            for (VerticalSideSeries series : this.verticalSideSeriesAtBoundary) {
                result.addAll(series.containedBoundaryLinks);
            }
            return result;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<List<BoundaryLink>> allBoundaries() {
        Object object = this.lock;
        synchronized (object) {
            this.findBoundaries();
            return Collections.unmodifiableList(this.allBoundaries);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public double unionArea() {
        Object object = this.lock;
        synchronized (object) {
            this.findBoundaries();
            return this.unionArea;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public IRectangularArea largestRectangleInUnion() {
        Object object = this.lock;
        synchronized (object) {
            this.findLargestRectangleInUnion();
            return this.largestRectangleInUnion;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void findConnectedComponents() {
        Object object = this.lock;
        synchronized (object) {
            this.doCreateSideLists();
            if (this.connectedComponents != null) {
                return;
            }
            if (this.frames.isEmpty()) {
                this.connectedComponents = Collections.emptyList();
                return;
            }
            long t1 = System.nanoTime();
            List<List<Frame>> connectionLists = IRectanglesUnion.createListOfLists(this.frames.size());
            long t2 = System.nanoTime();
            long nConnections = this.doFillConnectionLists(connectionLists);
            long t3 = System.nanoTime();
            List<List<Frame>> result = this.doFindConnectedComponents(connectionLists);
            long t4 = System.nanoTime();
            this.connectedComponents = result;
            IRectanglesUnion.debug(1, "Rectangle union (%d rectangles), finding %d connected components: %.3f ms = %.3f ms initializing + %.3f finding %d connections + %.3f breadth-first search (%.3f mcs / rectangle)%n", this.frames.size(), result.size(), (double)(t4 - t1) * 1.0E-6, (double)(t2 - t1) * 1.0E-6, (double)(t3 - t2) * 1.0E-6, nConnections, (double)(t4 - t3) * 1.0E-6, (double)(t4 - t1) * 0.001 / (double)this.frames.size());
            if (DEBUG_LEVEL >= 2 && result.size() >= 2) {
                t1 = System.nanoTime();
                for (int i = 0; i < result.size(); ++i) {
                    for (Frame frame1 : result.get(i)) {
                        for (int j = i + 1; j < result.size(); ++j) {
                            for (Frame frame2 : result.get(j)) {
                                if (frame1.rectangle.intersects(frame2.rectangle)) {
                                    throw new AssertionError((Object)("First 2 connected component really have intersection: " + String.valueOf(frame1) + " (component " + i + ") intersects " + String.valueOf(frame2) + " (component " + j + ")"));
                                }
                            }
                        }
                    }
                }
                t2 = System.nanoTime();
                IRectanglesUnion.debug(2, "Testing connected components: %.3f ms%n", (double)(t2 - t1) * 1.0E-6);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void findBoundaries() {
        Object object = this.lock;
        synchronized (object) {
            this.doCreateSideLists();
            if (this.allBoundaries != null) {
                return;
            }
            if (this.frames.isEmpty()) {
                this.horizontalSideSeries = Collections.emptyList();
                this.verticalSideSeries = Collections.emptyList();
                this.horizontalSideSeriesAtBoundary = Collections.emptyList();
                this.verticalSideSeriesAtBoundary = Collections.emptyList();
                this.allDifferentXAtBoundary = new long[0];
                this.unionArea = 0.0;
                this.allBoundaries = Collections.emptyList();
                return;
            }
            long t1 = System.nanoTime();
            this.horizontalSideSeries = this.createHorizontalSideSeriesLists();
            this.verticalSideSeries = this.createVerticalSideSeriesLists();
            long t2 = System.nanoTime();
            long hCount = this.doFindHorizontalBoundaries();
            long t3 = System.nanoTime();
            this.horizontalSideSeriesAtBoundary = new ArrayList<HorizontalSideSeries>();
            this.doExtractHorizontalSeriesAtBoundary();
            long t4 = System.nanoTime();
            long vCount = this.doConvertHorizontalToVerticalLinks();
            this.verticalSideSeriesAtBoundary = new ArrayList<VerticalSideSeries>();
            long t5 = System.nanoTime();
            this.doExtractVerticalSeriesAtBoundary();
            if (vCount != hCount) {
                throw new AssertionError((Object)"Different numbers of horizontal and vertical links found");
            }
            long t6 = System.nanoTime();
            this.allDifferentXAtBoundary = this.doExtractAllDifferentXAtBoundary();
            this.unionArea = this.doCalculateArea();
            this.doSetLinkIndexes(hCount);
            assert (hCount <= Integer.MAX_VALUE);
            this.allBoundaries = this.doJoinBoundaries(hCount);
            long t7 = System.nanoTime();
            if (DEBUG_LEVEL >= 1) {
                long totalLinkCount = IRectanglesUnion.totalCount(this.allBoundaries);
                IRectanglesUnion.debug(1, "Rectangle union (%d rectangles), area %.1f, finding %d boundaries with %d links: %.3f ms = %.3f ms initializing + %.3f ms %d horizontal links + %.3f ms %d/%d horizontals at boundary + %.3f ms %d vertical links + %.3f ms %d/%d verticals at boundary + %.3f ms postprocessing and joining links (%.3f mcs / rectangle, %.3f mcs / link)%n", this.frames.size(), this.unionArea, this.allBoundaries().size(), totalLinkCount, (double)(t7 - t1) * 1.0E-6, (double)(t2 - t1) * 1.0E-6, (double)(t3 - t2) * 1.0E-6, hCount, (double)(t4 - t3) * 1.0E-6, this.horizontalSideSeriesAtBoundary.size(), this.horizontalSideSeries.size(), (double)(t5 - t4) * 1.0E-6, vCount, (double)(t6 - t5) * 1.0E-6, this.verticalSideSeriesAtBoundary.size(), this.verticalSideSeries.size(), (double)(t7 - t6) * 1.0E-6, (double)(t7 - t1) * 0.001 / (double)this.frames.size(), (double)(t7 - t1) * 0.001 / (double)totalLinkCount);
            }
            if (DEBUG_LEVEL >= 2) {
                StringBuilder sb = new StringBuilder();
                for (int k = 0; k < this.allDifferentXAtBoundary.length; ++k) {
                    if (k == 100) {
                        sb.append("...");
                        break;
                    }
                    sb.append("(").append(k).append(":)").append(this.allDifferentXAtBoundary[k]).append(", ");
                }
                IRectanglesUnion.debug(2, "Found %d verticals with different x: %s%n", this.allDifferentXAtBoundary.length, sb);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void findLargestRectangleInUnion() {
        Object object = this.lock;
        synchronized (object) {
            this.findBoundaries();
            if (this.largestRectangleInUnion != null) {
                return;
            }
            if (this.frames.isEmpty()) {
                this.horizontalSectionsByLowerSides = Collections.emptyList();
                this.largestRectangleInUnion = null;
                return;
            }
            long t1 = System.nanoTime();
            List<HorizontalSection> horizontalSectionsByLowerSides = this.doFindHorizontalSections();
            long t2 = System.nanoTime();
            List closingLinksIntersectingEachVertical = this.doFindClosingLinksIntersectingEachVertical();
            long t3 = System.nanoTime();
            SearchIRectangleInHypograph searcher = this.doFindLargestRegtangles(horizontalSectionsByLowerSides, closingLinksIntersectingEachVertical);
            long t4 = System.nanoTime();
            this.horizontalSectionsByLowerSides = horizontalSectionsByLowerSides;
            this.largestRectangleInUnion = searcher.largestRectangle();
            if (DEBUG_LEVEL >= 1) {
                long totalLinkCount = IRectanglesUnion.totalCount(this.allBoundaries);
                IRectanglesUnion.debug(1, "Rectangle union (%d rectangles, %d links), finding largest rectangle %s (area %.1f): %.3f ms = %.3f ms %d horizontal sections + %.3f ms %d links intersecting with %d verticals + %.3f ms searching largest rectangle (%.3f mcs / rectangle, %.3f mcs / link)%n", this.frames.size(), totalLinkCount, this.largestRectangleInUnion, this.largestRectangleInUnion.volume(), (double)(t4 - t1) * 1.0E-6, (double)(t2 - t1) * 1.0E-6, horizontalSectionsByLowerSides.size(), (double)(t3 - t2) * 1.0E-6, IRectanglesUnion.totalCount(closingLinksIntersectingEachVertical), this.allDifferentXAtBoundary.length - 1, (double)(t4 - t3) * 1.0E-6, (double)(t4 - t1) * 0.001 / (double)this.frames.size(), (double)(t4 - t1) * 0.001 / (double)totalLinkCount);
            }
            if (DEBUG_LEVEL >= 2) {
                t1 = System.nanoTime();
                for (HorizontalSection section : horizontalSectionsByLowerSides) {
                    IRectangularArea r = section.equivalentRectangle();
                    LinkedList<IRectangularArea> queue = new LinkedList<IRectangularArea>();
                    queue.add(r);
                    for (Frame frame : this.frames) {
                        IRectangularArea.subtractCollection(queue, IRectangularArea.valueOf(frame.fromX, frame.fromY, frame.toX - 1L, frame.toY));
                    }
                    if (!queue.isEmpty()) {
                        throw new AssertionError((Object)("Section " + String.valueOf(section) + " is not a subset of the union"));
                    }
                }
                t2 = System.nanoTime();
                IRectanglesUnion.debug(2, "Testing horizontal sections: %.3f ms%n", (double)(t2 - t1) * 1.0E-6);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String toString() {
        Object object = this.lock;
        synchronized (object) {
            return "union of " + this.frames.size() + " rectangles" + (String)(this.connectedComponents == null ? "" : ", " + this.connectedComponents.size() + " connected components");
        }
    }

    public static double areaInBoundary(List<BoundaryLink> boundary) {
        Objects.requireNonNull(boundary, "Null boundary");
        double result = 0.0;
        for (BoundaryLink link : boundary) {
            if (!link.isHorizontal()) continue;
            result += link.areaUnderLink();
        }
        return result;
    }

    public static List<IPoint> boundaryVerticesPlusHalf(List<BoundaryLink> boundary) {
        Objects.requireNonNull(boundary, "Null boundary");
        ArrayList<IPoint> result = new ArrayList<IPoint>();
        BoundaryLink last = null;
        for (BoundaryLink link : boundary) {
            long coord = link.coord();
            long secondCoord = last == link.linkTo() ? link.to : link.from;
            result.add(link.isHorizontal() ? IPoint.valueOf(secondCoord, coord) : IPoint.valueOf(coord, secondCoord));
            last = link;
        }
        if (DEBUG_LEVEL >= 3) {
            IRectanglesUnion.debug(3, "Boundary precise vertices +0.5: %s%n", result);
        }
        return result;
    }

    public static List<IPoint> boundaryVerticesAtRectangles(List<BoundaryLink> boundary) {
        Objects.requireNonNull(boundary, "Null boundary");
        ArrayList<IPoint> result = new ArrayList<IPoint>();
        int n = boundary.size();
        if (n == 0) {
            return result;
        }
        BoundaryLink last = boundary.get(n - 1);
        for (BoundaryLink link : boundary) {
            long coord;
            boolean lastFirst = last.atFirstOfTwoParallelSides();
            boolean thisFirst = link.atFirstOfTwoParallelSides();
            long l = coord = thisFirst ? link.coord() : link.coord() - 1L;
            long secondCoord = last == link.linkTo() ? (lastFirst ? link.to : link.to - 1L) : (lastFirst ? link.from : link.from - 1L);
            result.add(link.isHorizontal() ? IPoint.valueOf(secondCoord, coord) : IPoint.valueOf(coord, secondCoord));
            last = link;
        }
        if (DEBUG_LEVEL >= 1) {
            int k = 0;
            for (BoundaryLink link : boundary) {
                IRectangularArea onLink;
                IPoint v2;
                IPoint v1 = (IPoint)result.get(k);
                IRectangularArea onVertices = IRectangularArea.valueOf(v1.min(v2 = (IPoint)result.get(k == n - 1 ? 0 : k + 1)), v1.max(v2));
                if (!onVertices.contains(onLink = link.equivalentRectangle())) {
                    throw new AssertionError((Object)("Boundary rectangle does not contain the link #" + k + ": " + String.valueOf(v1) + ", " + String.valueOf(v2) + ", " + String.valueOf(onLink)));
                }
                if (onVertices.volume() - onLink.volume() > 2.0001) {
                    throw new AssertionError((Object)("Boundary rectangle is too large: link #" + k + ": " + String.valueOf(v1) + ", " + String.valueOf(v2) + ", " + String.valueOf(onLink)));
                }
                ++k;
            }
        }
        if (DEBUG_LEVEL >= 3) {
            IRectanglesUnion.debug(3, "Boundary vertices at rectangles: %s%n", result);
        }
        return result;
    }

    private void doCreateSideLists() {
        if (this.horizontalSides != null) {
            return;
        }
        long t1 = System.nanoTime();
        ArrayList<HorizontalSide> horizontalSides = new ArrayList<HorizontalSide>();
        ArrayList<VerticalSide> verticalSides = new ArrayList<VerticalSide>();
        for (Frame frame : this.frames) {
            horizontalSides.add(frame.lessHorizontalSide);
            horizontalSides.add(frame.higherHorizontalSide);
            verticalSides.add(frame.lessVerticalSide);
            verticalSides.add(frame.higherVerticalSide);
        }
        Collections.sort(horizontalSides);
        Collections.sort(verticalSides);
        long t2 = System.nanoTime();
        this.horizontalSides = horizontalSides;
        this.verticalSides = verticalSides;
        IRectanglesUnion.debug(1, "Rectangle union (%d rectangles), allocating and sorting sides: %.3f ms%n", this.frames.size(), (double)(t2 - t1) * 1.0E-6);
    }

    private long doFillConnectionLists(List<List<Frame>> connectionLists) {
        assert (!this.frames.isEmpty());
        HorizontalIBracketSet<HorizontalSide> bracketSet = new HorizontalIBracketSet<HorizontalSide>(this.horizontalSides, false);
        long count = 0L;
        while (bracketSet.next()) {
            IBracket lastBefore;
            if (((HorizontalSide)bracketSet.horizontal).first && (lastBefore = bracketSet.lastIntersectionBeforeLeft()) != null && lastBefore.followingCoveringDepth > 0) {
                IRectanglesUnion.addConnection(connectionLists, lastBefore.intersectingSide.frame, ((HorizontalSide)bracketSet.horizontal).frame);
                ++count;
            }
            if (!((HorizontalSide)bracketSet.horizontal).first) continue;
            for (IBracket bracket : bracketSet.currentIntersections()) {
                if (DEBUG_LEVEL >= 2) assert (bracket.covers(bracketSet.coord));
                IRectanglesUnion.addConnection(connectionLists, bracket.intersectingSide.frame, ((HorizontalSide)bracketSet.horizontal).frame);
                ++count;
            }
        }
        return count;
    }

    private List<List<Frame>> doFindConnectedComponents(List<List<Frame>> connectionLists) {
        assert (!this.frames.isEmpty());
        ArrayList<List<Frame>> result = new ArrayList<List<Frame>>();
        boolean[] frameVisited = new boolean[this.frames.size()];
        LinkedList<Frame> queue = new LinkedList<Frame>();
        int index = 0;
        while (true) {
            if (index < frameVisited.length && frameVisited[index]) {
                ++index;
                continue;
            }
            if (index >= frameVisited.length) break;
            ArrayList<Frame> component = new ArrayList<Frame>();
            queue.add(this.frames.get(index));
            frameVisited[index] = true;
            while (!queue.isEmpty()) {
                Frame frame = (Frame)queue.poll();
                component.add(frame);
                List<Frame> neighbours = connectionLists.get(frame.index);
                for (Frame neighbour : neighbours) {
                    if (frameVisited[neighbour.index]) continue;
                    queue.add(neighbour);
                    frameVisited[neighbour.index] = true;
                }
                if (DEBUG_LEVEL < 4) continue;
                IRectanglesUnion.debug(4, "  Neighbours of %s:%n", frame);
                for (Frame neighbour : neighbours) {
                    IRectanglesUnion.debug(4, "    %s%n", neighbour);
                }
            }
            result.add(component);
        }
        return result;
    }

    private List<HorizontalSideSeries> createHorizontalSideSeriesLists() {
        ArrayList<HorizontalSideSeries> result = new ArrayList<HorizontalSideSeries>();
        SideSeries last = null;
        for (HorizontalSide side : this.horizontalSides) {
            boolean expanded = last != null && last.expand(side);
            if (expanded) continue;
            if (last != null) {
                result.add((HorizontalSideSeries)last);
            }
            last = new HorizontalSideSeries(side);
        }
        if (last != null) {
            result.add((HorizontalSideSeries)last);
        }
        this.checkSidesSeriesList(result);
        return result;
    }

    private List<VerticalSideSeries> createVerticalSideSeriesLists() {
        ArrayList<VerticalSideSeries> result = new ArrayList<VerticalSideSeries>();
        SideSeries last = null;
        for (VerticalSide side : this.verticalSides) {
            boolean expanded = last != null && last.expand(side);
            if (expanded) continue;
            if (last != null) {
                result.add((VerticalSideSeries)last);
            }
            last = new VerticalSideSeries(side);
        }
        if (last != null) {
            result.add((VerticalSideSeries)last);
        }
        this.checkSidesSeriesList(result);
        return result;
    }

    private void checkSidesSeriesList(List<? extends SideSeries> sideSeries) {
        int k;
        int n;
        if (DEBUG_LEVEL >= 2) {
            n = sideSeries.size();
            for (k = 1; k < n; ++k) {
                assert (sideSeries.get(k - 1).compareTo(sideSeries.get(k)) <= 0);
            }
        }
        if (DEBUG_LEVEL >= 3) {
            IRectanglesUnion.debug(3, "  %d side series:%n", sideSeries.size());
            n = sideSeries.size();
            for (k = 0; k < n; ++k) {
                IRectanglesUnion.debug(3, "    side series %d/%d: %s%n", k, n, sideSeries.get(k));
            }
        }
    }

    private long doFindHorizontalBoundaries() {
        assert (!this.frames.isEmpty());
        HorizontalIBracketSet<HorizontalSideSeries> bracketSet = new HorizontalIBracketSet<HorizontalSideSeries>(this.horizontalSideSeries, true);
        long count = 0L;
        while (bracketSet.next()) {
            Set<IBracket> brackets = bracketSet.currentIntersections();
            IBracket lastBefore = bracketSet.lastIntersectionBeforeLeft();
            boolean lastRightAtBoundary = lastBefore == null || lastBefore.followingCoveringDepth == 0;
            FrameSide lastLeftVertical = lastRightAtBoundary ? ((HorizontalSideSeries)bracketSet.horizontal).transversalFrameSideFrom() : null;
            for (IBracket bracket : brackets) {
                if (DEBUG_LEVEL >= 2) assert (bracket.covers(bracketSet.coord));
                boolean rightAtBoundary = bracket.followingCoveringDepth == 0;
                if (rightAtBoundary == lastRightAtBoundary) continue;
                if (rightAtBoundary) {
                    lastLeftVertical = bracket.intersectingSide;
                } else {
                    IRectanglesUnion.addHorizontalLink(bracketSet, lastLeftVertical, bracket.intersectingSide);
                }
                lastRightAtBoundary = rightAtBoundary;
            }
            if (lastRightAtBoundary) {
                IRectanglesUnion.addHorizontalLink(bracketSet, lastLeftVertical, ((HorizontalSideSeries)bracketSet.horizontal).transversalFrameSideTo());
            }
            count += (long)((HorizontalSideSeries)bracketSet.horizontal).containedLinksCount();
        }
        return count;
    }

    private long doConvertHorizontalToVerticalLinks() {
        assert (!this.frames.isEmpty());
        for (VerticalSideSeries verticalSeries : this.verticalSideSeries) {
            assert (verticalSeries.containedBoundaryLinks == null) : "non-null containedBoundaryLinks = " + String.valueOf(verticalSeries.containedBoundaryLinks);
            assert (verticalSeries.intersectingBoundaryLinks == null) : "non-null intersectingBoundaryLinks = " + String.valueOf(verticalSeries.intersectingBoundaryLinks);
        }
        for (HorizontalSideSeries series : this.horizontalSideSeriesAtBoundary) {
            for (HorizontalBoundaryLink link : series.containedBoundaryLinks) {
                link.transversalSeriesFrom.addIntersectingLink(link);
                link.transversalSeriesTo.addIntersectingLink(link);
            }
        }
        Object[] horizontalLinks = new HorizontalBoundaryLink[]{};
        long count = 0L;
        for (VerticalSideSeries verticalSeries : this.verticalSideSeries) {
            if (verticalSeries.intersectingBoundaryLinks == null) continue;
            int horizontalsCount = verticalSeries.intersectingBoundaryLinks.size();
            assert (horizontalsCount > 0 && horizontalsCount % 2 == 0) : "Invalid horizontalsCount=" + horizontalsCount;
            horizontalLinks = verticalSeries.intersectingBoundaryLinks.toArray(horizontalLinks);
            Arrays.sort(horizontalLinks, 0, horizontalsCount);
            for (int k = 0; k < horizontalsCount; k += 2) {
                Object linkFrom = horizontalLinks[k];
                Object linkTo = horizontalLinks[k + 1];
                long from = ((BoundaryLink)linkFrom).coord();
                long to = ((BoundaryLink)linkTo).coord();
                assert (k == 0 || from > ((BoundaryLink)horizontalLinks[k - 1]).coord()) : "Two horizontal links with the same ordinate " + from + " (=" + ((BoundaryLink)horizontalLinks[k - 1]).coord() + ") are incident with the same vertical side";
                assert (from < to) : "Empty vertical link #" + k / 2 + ": " + from + ".." + to;
                VerticalBoundaryLink link = new VerticalBoundaryLink(verticalSeries, (HorizontalBoundaryLink)linkFrom, (HorizontalBoundaryLink)linkTo);
                ((HorizontalBoundaryLink)linkFrom).setNeighbour(link);
                ((HorizontalBoundaryLink)linkTo).setNeighbour(link);
                verticalSeries.addLink(link);
            }
            count += (long)verticalSeries.containedLinksCount();
        }
        return count;
    }

    private void doExtractHorizontalSeriesAtBoundary() {
        int count = 0;
        for (HorizontalSideSeries series : this.horizontalSideSeries) {
            if (!series.containsBoundary()) continue;
            series.indexInSortedListAtBoundary = count++;
            this.horizontalSideSeriesAtBoundary.add(series);
        }
    }

    private void doExtractVerticalSeriesAtBoundary() {
        int count = 0;
        for (VerticalSideSeries series : this.verticalSideSeries) {
            if (!series.containsBoundary()) continue;
            series.indexInSortedListAtBoundary = count++;
            this.verticalSideSeriesAtBoundary.add(series);
        }
    }

    private long[] doExtractAllDifferentXAtBoundary() {
        int count = 0;
        long lastX = 157L;
        for (VerticalSideSeries series : this.verticalSideSeriesAtBoundary) {
            assert (series.containsBoundary());
            long coord = series.coord();
            if (count == 0 || coord != lastX) {
                lastX = coord;
                ++count;
            }
            series.numberOfLessCoordinatesAtBoundary = count - 1;
        }
        long[] result = new long[count];
        count = 0;
        for (VerticalSideSeries series : this.verticalSideSeriesAtBoundary) {
            long coord = series.coord();
            if (count != 0 && coord == lastX) continue;
            lastX = coord;
            result[count] = coord;
            ++count;
        }
        return result;
    }

    private void doSetLinkIndexes(long horizontalOrVerticalLinksCount) {
        int count = 0;
        for (HorizontalSideSeries horizontalSideSeries : this.horizontalSideSeriesAtBoundary) {
            for (HorizontalBoundaryLink horizontalBoundaryLink : horizontalSideSeries.containedBoundaryLinks) {
                if (count == Integer.MAX_VALUE) {
                    throw new OutOfMemoryError("Number of horizontal links must be < 2^31");
                }
                horizontalBoundaryLink.indexInSortedList = count++;
            }
        }
        assert ((long)count == horizontalOrVerticalLinksCount);
        count = 0;
        for (VerticalSideSeries verticalSideSeries : this.verticalSideSeriesAtBoundary) {
            for (VerticalBoundaryLink verticalBoundaryLink : verticalSideSeries.containedBoundaryLinks) {
                if (count == Integer.MAX_VALUE) {
                    throw new OutOfMemoryError("Number of vertical links must be < 2^31");
                }
                verticalBoundaryLink.indexInSortedList = count++;
            }
        }
        assert ((long)count == horizontalOrVerticalLinksCount);
    }

    private double doCalculateArea() {
        double result = 0.0;
        for (HorizontalSideSeries series : this.horizontalSideSeriesAtBoundary) {
            for (HorizontalBoundaryLink link : series.containedBoundaryLinks) {
                result += link.areaUnderLink();
            }
        }
        return result;
    }

    private List<List<BoundaryLink>> doJoinBoundaries(long numberOfHorizontalLinks) {
        assert (!this.frames.isEmpty());
        long maxCount = 10L * Math.min(numberOfHorizontalLinks, Integer.MAX_VALUE);
        ArrayList<List<BoundaryLink>> result = new ArrayList<List<BoundaryLink>>();
        for (HorizontalSideSeries series : this.horizontalSideSeriesAtBoundary) {
            for (HorizontalBoundaryLink link : series.containedBoundaryLinks) {
                if (link.joinedIntoAllBoundaries) continue;
                List<BoundaryLink> boundary = IRectanglesUnion.scanBoundary(link, maxCount);
                result.add(Collections.unmodifiableList(boundary));
            }
        }
        return result;
    }

    private List<HorizontalSection> doFindHorizontalSectionsSlowly() {
        assert (!this.frames.isEmpty());
        ArrayList<HorizontalSection> result = new ArrayList<HorizontalSection>();
        HorizontalIBracketSet<HorizontalSideSeries> bracketSet = new HorizontalIBracketSet<HorizontalSideSeries>(this.horizontalSideSeries, false);
        HorizontalSection lastSection = null;
        while (bracketSet.next()) {
            if (!((HorizontalSideSeries)bracketSet.horizontal).first || !((HorizontalSideSeries)bracketSet.horizontal).containsBoundary()) continue;
            if (lastSection != null && bracketSet.coord == lastSection.coord) {
                assert (lastSection.from() <= ((HorizontalSideSeries)bracketSet.horizontal).from) : "sides series sorted incorrectly";
                if (((HorizontalSideSeries)bracketSet.horizontal).to <= lastSection.to()) {
                    lastSection.boundaryLinksAtSection.addAll(((HorizontalSideSeries)bracketSet.horizontal).containedBoundaryLinks);
                    continue;
                }
            }
            FrameSide left = bracketSet.maxLeftBeloningToUnion();
            FrameSide right = bracketSet.minRightBeloningToUnion();
            assert (left != null);
            assert (right != null);
            assert (left.coord() <= right.coord());
            lastSection = new HorizontalSection(true, bracketSet.coord, left, right);
            lastSection.boundaryLinksAtSection.addAll(((HorizontalSideSeries)bracketSet.horizontal).containedBoundaryLinks);
            result.add(lastSection);
        }
        return result;
    }

    private List<HorizontalSection> doFindHorizontalSections() {
        assert (!this.frames.isEmpty());
        ArrayList<HorizontalSection> result = new ArrayList<HorizontalSection>();
        HorizontalBoundaryIBracketSet<HorizontalBoundaryLink> bracketSet = new HorizontalBoundaryIBracketSet<HorizontalBoundaryLink>(this.allHorizontalBoundaryLinks());
        HorizontalSection lastSection = null;
        while (bracketSet.next()) {
            if (!((HorizontalBoundaryLink)bracketSet.horizontal).atFirstOfTwoParallelSides()) continue;
            if (lastSection != null && bracketSet.coord == lastSection.coord) {
                assert (lastSection.from() <= ((HorizontalBoundaryLink)bracketSet.horizontal).from) : "sides series sorted incorrectly";
                if (((HorizontalBoundaryLink)bracketSet.horizontal).to <= lastSection.to()) {
                    lastSection.boundaryLinksAtSection.add((HorizontalBoundaryLink)bracketSet.horizontal);
                    continue;
                }
            }
            int leftIndex = bracketSet.maxLeftIndexBeloningToUnion();
            int rightIndex = bracketSet.minRightIndexBeloningToUnion();
            assert (leftIndex <= rightIndex);
            long left = this.allDifferentXAtBoundary[leftIndex];
            long right = this.allDifferentXAtBoundary[rightIndex];
            lastSection = new HorizontalSection(true, bracketSet.coord, left, right, leftIndex, rightIndex);
            lastSection.boundaryLinksAtSection.add((HorizontalBoundaryLink)bracketSet.horizontal);
            result.add(lastSection);
        }
        return result;
    }

    private List<List<HorizontalBoundaryLink>> doFindClosingLinksIntersectingEachVertical() {
        assert (!this.frames.isEmpty());
        List<List<HorizontalBoundaryLink>> result = IRectanglesUnion.createListOfLists(this.allDifferentXAtBoundary.length - 1);
        for (HorizontalSideSeries series : this.horizontalSideSeriesAtBoundary) {
            assert (series.containsBoundary());
            if (series.first) continue;
            for (HorizontalBoundaryLink closingLink : series.containedBoundaryLinks) {
                int from = closingLink.transversalSeriesFrom.numberOfLessCoordinatesAtBoundary;
                int to = closingLink.transversalSeriesTo.numberOfLessCoordinatesAtBoundary;
                for (int k = from; k < to; ++k) {
                    result.get(k).add(closingLink);
                }
            }
        }
        return result;
    }

    private SearchIRectangleInHypograph doFindLargestRegtangles(List<HorizontalSection> horizontalSectionsByLowerSides, List<List<HorizontalBoundaryLink>> closingLinksIntersectingEachVertical) {
        assert (!this.frames.isEmpty());
        assert (this.allDifferentXAtBoundary.length >= 2) : "If frames exist, they cannot have <2 vertical boundary links";
        SearchIRectangleInHypograph searcher = new SearchIRectangleInHypograph(this.allDifferentXAtBoundary);
        long[] workY = new long[this.allDifferentXAtBoundary.length - 1];
        for (HorizontalSection section : horizontalSectionsByLowerSides) {
            searcher.setCurrentFromY(section.coord);
            for (HorizontalBoundaryLink openingLink : section.boundaryLinksAtSection) {
                int k;
                int from = openingLink.transversalSeriesFrom.numberOfLessCoordinatesAtBoundary;
                int to = openingLink.transversalSeriesTo.numberOfLessCoordinatesAtBoundary;
                Arrays.fill(workY, from, to, Long.MAX_VALUE);
                for (k = from; k < to; ++k) {
                    for (HorizontalBoundaryLink closingLink : closingLinksIntersectingEachVertical.get(k)) {
                        long y = closingLink.coord();
                        if (y < section.coord || y >= workY[k]) continue;
                        workY[k] = y;
                    }
                }
                for (k = from; k < to; ++k) {
                    searcher.setY(k, workY[k]);
                }
            }
            if (DEBUG_LEVEL >= 3) {
                searcher.resetAlreadyFoundRectangle();
            }
            searcher.resetMaxRectangleCorrected();
            searcher.correctMaximalRectangle(section.leftNumberOfLessCoordinatesAtBoundary, section.rightNumberOfLessCoordinatesAtBoundary);
            if (!searcher.isMaxRectangleCorrected()) continue;
            section.largestRectangle = searcher.largestRectangle();
        }
        return searcher;
    }

    static void debug(int level, String format, Object ... args) {
        if (DEBUG_LEVEL >= level) {
            System.out.printf(Locale.US, " IRU " + format, args);
        }
    }

    private static void addConnection(List<List<Frame>> connectionLists, Frame a, Frame b) {
        connectionLists.get(a.index).add(b);
        connectionLists.get(b.index).add(a);
    }

    private static void addHorizontalLink(HorizontalIBracketSet<HorizontalSideSeries> bracketSet, FrameSide firstTransveral, FrameSide secondTransveral) {
        assert (firstTransveral.coord() <= secondTransveral.coord());
        HorizontalBoundaryLink link = new HorizontalBoundaryLink((HorizontalSideSeries)bracketSet.horizontal, (VerticalSideSeries)firstTransveral.parentSeries, (VerticalSideSeries)secondTransveral.parentSeries);
        if (link.from < link.to) {
            if (DEBUG_LEVEL >= 3) {
                IRectanglesUnion.debug(3, "    adding %s%n", link);
            }
            ((HorizontalSideSeries)bracketSet.horizontal).addLink(link);
        }
    }

    private static List<BoundaryLink> scanBoundary(HorizontalBoundaryLink start, long maxCount) {
        ArrayList<BoundaryLink> result = new ArrayList<BoundaryLink>();
        HorizontalBoundaryLink hLink = start;
        VerticalBoundaryLink vLink = start.linkTo;
        long count = 0L;
        do {
            assert (vLink.linkFrom != null && vLink.linkTo != null);
            assert (hLink == vLink.linkFrom || hLink == vLink.linkTo);
            assert (!hLink.joinedIntoAllBoundaries);
            assert (!vLink.joinedIntoAllBoundaries);
            result.add(hLink);
            result.add(vLink);
            hLink.joinedIntoAllBoundaries = true;
            vLink.joinedIntoAllBoundaries = true;
            HorizontalBoundaryLink horizontalBoundaryLink = hLink = hLink == vLink.linkFrom ? vLink.linkTo : vLink.linkFrom;
            assert (hLink.linkFrom != null && hLink.linkTo != null);
            VerticalBoundaryLink verticalBoundaryLink = vLink = vLink == hLink.linkFrom ? hLink.linkTo : hLink.linkFrom;
            if (++count > maxCount) {
                throw new AssertionError((Object)"Infinite loop detected while scanning the boundary");
            }
        } while (hLink != start);
        return result;
    }

    private static List<Frame> checkAndConvertToFrames(Collection<IRectangularArea> rectangles) {
        Objects.requireNonNull(rectangles, "Null rectangles argument");
        for (IRectangularArea rectangle : rectangles) {
            Objects.requireNonNull(rectangle, "Null rectangle in a collection");
            if (rectangle.coordCount() == 2) continue;
            throw new IllegalArgumentException("Only 2-dimensional rectangles can be joined");
        }
        long t1 = System.nanoTime();
        ArrayList<Frame> result = new ArrayList<Frame>();
        int index = 0;
        for (IRectangularArea rectangle : rectangles) {
            result.add(new Frame(rectangle, index++));
        }
        long t2 = System.nanoTime();
        IRectanglesUnion.debug(1, "Rectangle union (%d rectangles), initial allocating frames: %.3f ms%n", result.size(), (double)(t2 - t1) * 1.0E-6);
        return result;
    }

    private static List<Frame> cloneFrames(Collection<Frame> frames) {
        assert (frames != null);
        long t1 = System.nanoTime();
        ArrayList<Frame> result = new ArrayList<Frame>();
        int index = 0;
        for (Frame frame : frames) {
            result.add(new Frame(frame.rectangle, index++));
        }
        long t2 = System.nanoTime();
        IRectanglesUnion.debug(1, "Rectangle union (%d rectangles), initial cloning frames: %.3f ms%n", result.size(), (double)(t2 - t1) * 1.0E-6);
        return result;
    }

    private static <T> List<List<T>> createListOfLists(int n) {
        ArrayList<List<T>> result = new ArrayList<List<T>>();
        for (int k = 0; k < n; ++k) {
            result.add(new ArrayList());
        }
        return result;
    }

    private static <T> long totalCount(List<List<T>> listOfLists) {
        long result = 0L;
        for (List<T> list : listOfLists) {
            result += (long)list.size();
        }
        return result;
    }

    public static class Frame {
        final HorizontalSide lessHorizontalSide;
        final HorizontalSide higherHorizontalSide;
        final VerticalSide lessVerticalSide;
        final VerticalSide higherVerticalSide;
        private final IRectangularArea rectangle;
        private final long fromX;
        private final long toX;
        private final long fromY;
        private final long toY;
        final int index;

        private Frame(IRectangularArea rectangle, int index) {
            assert (rectangle != null);
            this.rectangle = rectangle;
            this.lessHorizontalSide = new HorizontalSide(true, this);
            this.higherHorizontalSide = new HorizontalSide(false, this);
            this.lessVerticalSide = new VerticalSide(true, this);
            this.higherVerticalSide = new VerticalSide(false, this);
            this.fromX = rectangle.min(0);
            this.toX = rectangle.max(0) + 1L;
            this.fromY = rectangle.min(1);
            this.toY = rectangle.max(1) + 1L;
            this.index = index;
        }

        public IRectangularArea rectangle() {
            return this.rectangle;
        }

        public String toString() {
            return "Frame #" + this.index + " (" + String.valueOf(this.rectangle) + ")";
        }
    }

    static class HorizontalSideSeries
    extends SideSeries {
        private List<HorizontalBoundaryLink> containedBoundaryLinks = null;

        private HorizontalSideSeries(FrameSide initialSide) {
            super(initialSide);
        }

        @Override
        public boolean isHorizontal() {
            return true;
        }

        private void addLink(HorizontalBoundaryLink link) {
            if (this.containedBoundaryLinks == null) {
                this.containedBoundaryLinks = new ArrayList<HorizontalBoundaryLink>();
            }
            this.containedBoundaryLinks.add(link);
        }

        private boolean containsBoundary() {
            return this.containedBoundaryLinks != null;
        }

        private int containedLinksCount() {
            return this.containedBoundaryLinks == null ? 0 : this.containedBoundaryLinks.size();
        }
    }

    static class VerticalSideSeries
    extends SideSeries {
        private List<VerticalBoundaryLink> containedBoundaryLinks = null;
        private List<HorizontalBoundaryLink> intersectingBoundaryLinks = null;

        private VerticalSideSeries(FrameSide initialSide) {
            super(initialSide);
        }

        @Override
        public boolean isHorizontal() {
            return false;
        }

        private void addLink(VerticalBoundaryLink link) {
            if (this.containedBoundaryLinks == null) {
                this.containedBoundaryLinks = new ArrayList<VerticalBoundaryLink>();
            }
            this.containedBoundaryLinks.add(link);
        }

        private void addIntersectingLink(HorizontalBoundaryLink link) {
            if (this.intersectingBoundaryLinks == null) {
                this.intersectingBoundaryLinks = new ArrayList<HorizontalBoundaryLink>();
            }
            this.intersectingBoundaryLinks.add(link);
        }

        private boolean containsBoundary() {
            return this.containedBoundaryLinks != null;
        }

        private int containedLinksCount() {
            return this.containedBoundaryLinks == null ? 0 : this.containedBoundaryLinks.size();
        }
    }

    static class HorizontalSection
    extends Side {
        private final long coord;
        private final long left;
        private final long right;
        private final int leftNumberOfLessCoordinatesAtBoundary;
        private final int rightNumberOfLessCoordinatesAtBoundary;
        private final List<HorizontalBoundaryLink> boundaryLinksAtSection = new ArrayList<HorizontalBoundaryLink>();
        private IRectangularArea largestRectangle = null;

        private HorizontalSection(boolean first, long coord, FrameSide left, FrameSide right) {
            this(first, coord, left.coord(), right.coord(), left.parentSeries.numberOfLessCoordinatesAtBoundary, right.parentSeries.numberOfLessCoordinatesAtBoundary);
        }

        private HorizontalSection(boolean first, long coord, long left, long right, int leftNumberOfLessCoordinatesAtBoundary, int rightNumberOfLessCoordinatesAtBoundary) {
            super(first);
            this.coord = coord;
            this.left = left;
            this.right = right;
            this.leftNumberOfLessCoordinatesAtBoundary = leftNumberOfLessCoordinatesAtBoundary;
            this.rightNumberOfLessCoordinatesAtBoundary = rightNumberOfLessCoordinatesAtBoundary;
        }

        @Override
        public boolean isHorizontal() {
            return true;
        }

        @Override
        public long coord() {
            return this.coord;
        }

        @Override
        public long from() {
            return this.left;
        }

        @Override
        public long to() {
            return this.right;
        }

        @Override
        void allContainedFrameSides(List<FrameSide> result) {
            throw new UnsupportedOperationException();
        }

        @Override
        FrameSide transversalFrameSideFrom() {
            throw new UnsupportedOperationException();
        }

        @Override
        FrameSide transversalFrameSideTo() {
            throw new UnsupportedOperationException();
        }
    }

    public static abstract class BoundaryLink
    implements Comparable<BoundaryLink> {
        final SideSeries parentSeries;
        long from;
        long to;
        int indexInSortedList = -1;
        boolean joinedIntoAllBoundaries = false;

        private BoundaryLink(SideSeries parentSeries, long from, long to) {
            assert (parentSeries != null);
            this.parentSeries = parentSeries;
            assert (from <= to);
            this.from = from;
            this.to = to;
        }

        public boolean atFirstOfTwoParallelSides() {
            return this.parentSeries.first;
        }

        public boolean atSecondOfTwoParallelSides() {
            return !this.parentSeries.first;
        }

        public abstract boolean isHorizontal();

        public long coord() {
            return this.parentSeries.coord();
        }

        public long from() {
            return this.from;
        }

        public long to() {
            return this.to;
        }

        public abstract BoundaryLink linkFrom();

        public abstract BoundaryLink linkTo();

        public abstract IRectangularArea equivalentRectangle();

        @Override
        public int compareTo(BoundaryLink o) {
            if (this.coord() < o.coord()) {
                return -1;
            }
            if (this.coord() > o.coord()) {
                return 1;
            }
            if (this.from < o.from) {
                return -1;
            }
            if (this.from > o.from) {
                return 1;
            }
            if (this.to < o.to) {
                return -1;
            }
            if (this.to > o.to) {
                return 1;
            }
            return this.parentSeries.compareTo(o.parentSeries);
        }

        public String toString() {
            return (this instanceof HorizontalBoundaryLink ? "horizontal" : "vertical") + " boundary link " + this.from + ".." + this.to + " at " + String.valueOf(this.parentSeries);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            BoundaryLink that = (BoundaryLink)o;
            if (this.from != that.from) {
                return false;
            }
            if (this.to != that.to) {
                return false;
            }
            return this.parentSeries.equals(that.parentSeries);
        }

        public int hashCode() {
            int result = this.parentSeries.hashCode();
            result = 31 * result + Long.hashCode(this.from);
            result = 31 * result + Long.hashCode(this.to);
            return result;
        }

        double areaUnderLink() {
            double area = ((double)this.to - (double)this.from) * (double)this.coord();
            return this.atFirstOfTwoParallelSides() ? -area : area;
        }
    }

    public static class HorizontalSide
    extends FrameSide {
        private HorizontalSide(boolean first, Frame frame) {
            super(first, frame);
        }

        @Override
        public boolean isHorizontal() {
            return true;
        }

        @Override
        public long frameSideCoord() {
            return this.first ? this.frame.fromY : this.frame.toY - 1L;
        }

        @Override
        public long coord() {
            return this.first ? this.frame.fromY : this.frame.toY;
        }

        @Override
        public long from() {
            return this.frame.fromX;
        }

        @Override
        public long to() {
            return this.frame.toX;
        }

        @Override
        FrameSide transversalFrameSideFrom() {
            return this.frame.lessVerticalSide;
        }

        @Override
        FrameSide transversalFrameSideTo() {
            return this.frame.higherVerticalSide;
        }
    }

    public static class VerticalSide
    extends FrameSide {
        private VerticalSide(boolean first, Frame frame) {
            super(first, frame);
        }

        @Override
        public boolean isHorizontal() {
            return false;
        }

        @Override
        public long frameSideCoord() {
            return this.first ? this.frame.fromX : this.frame.toX - 1L;
        }

        @Override
        public long coord() {
            return this.first ? this.frame.fromX : this.frame.toX;
        }

        @Override
        public long from() {
            return this.frame.fromY;
        }

        @Override
        public long to() {
            return this.frame.toY;
        }

        @Override
        FrameSide transversalFrameSideFrom() {
            return this.frame.lessVerticalSide;
        }

        @Override
        FrameSide transversalFrameSideTo() {
            return this.frame.higherVerticalSide;
        }
    }

    public static abstract class Side
    implements Comparable<Side> {
        final boolean first;

        private Side(boolean first) {
            this.first = first;
        }

        public boolean isFirstOfTwoParallelSides() {
            return this.first;
        }

        public boolean isSecondOfTwoParallelSides() {
            return !this.first;
        }

        public abstract boolean isHorizontal();

        public long frameSideCoord() {
            return this.isFirstOfTwoParallelSides() ? this.coord() : this.coord() - 1L;
        }

        public abstract long coord();

        public abstract long from();

        public abstract long to();

        public IRectangularArea equivalentRectangle() {
            long coord = this.frameSideCoord();
            long from = this.from();
            long to = this.to();
            assert (from <= to);
            if (from == to) {
                return null;
            }
            if (this.isHorizontal()) {
                return IRectangularArea.valueOf(from, coord, to - 1L, coord);
            }
            return IRectangularArea.valueOf(coord, from, coord, to - 1L);
        }

        @Override
        public int compareTo(Side o) {
            long otherFrom;
            long otherCoord;
            if (this.getClass() != o.getClass()) {
                throw new ClassCastException("Comparison of sides with different types: " + String.valueOf(this.getClass()) + " != " + String.valueOf(o.getClass()));
            }
            long thisCoord = this.coord();
            if (thisCoord < (otherCoord = o.coord())) {
                return -1;
            }
            if (thisCoord > otherCoord) {
                return 1;
            }
            if (this.first && !o.first) {
                return -1;
            }
            if (!this.first && o.first) {
                return 1;
            }
            long thisFrom = this.from();
            if (thisFrom < (otherFrom = o.from())) {
                return -1;
            }
            if (thisFrom > otherFrom) {
                return 1;
            }
            long thisTo = this.to();
            long otherTo = o.to();
            return Long.compare(thisTo, otherTo);
        }

        public String toString() {
            return (this.isHorizontal() ? (this.first ? "top" : "bottom") : (this.first ? "left" : "right")) + " side" + (String)(this instanceof FrameSide ? " of frame #" + ((FrameSide)this).frame.index : "") + ": " + this.from() + ".." + this.to() + " at " + this.coord();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Side side = (Side)o;
            if (this.first != side.first) {
                return false;
            }
            return this.coord() == side.coord() && this.from() == side.from() && this.to() == side.to();
        }

        public int hashCode() {
            int result = this.first ? 1 : 0;
            long coord = this.coord();
            long from = this.from();
            long to = this.to();
            result = 31 * result + Long.hashCode(coord);
            result = 31 * result + Long.hashCode(from);
            result = 31 * result + Long.hashCode(to);
            return result;
        }

        abstract void allContainedFrameSides(List<FrameSide> var1);

        abstract FrameSide transversalFrameSideFrom();

        abstract FrameSide transversalFrameSideTo();
    }

    public static abstract class FrameSide
    extends Side {
        final Frame frame;
        SideSeries parentSeries = null;

        private FrameSide(boolean first, Frame frame) {
            super(first);
            assert (frame != null);
            this.frame = frame;
        }

        public Frame frame() {
            return this.frame;
        }

        @Override
        void allContainedFrameSides(List<FrameSide> result) {
            result.clear();
            result.add(this);
        }
    }

    static abstract class SideSeries
    extends Side {
        final long coord;
        long from;
        long to;
        private FrameSide sideFrom;
        private FrameSide sideTo;
        private final FrameSide firstSide;
        private List<FrameSide> otherSides = null;
        int indexInSortedListAtBoundary = -1;
        int numberOfLessCoordinatesAtBoundary = -1;

        private SideSeries(FrameSide initialSide) {
            super(initialSide.first);
            this.coord = initialSide.coord();
            this.from = initialSide.from();
            this.to = initialSide.to();
            this.sideFrom = initialSide.transversalFrameSideFrom();
            this.sideTo = initialSide.transversalFrameSideTo();
            this.firstSide = initialSide;
            initialSide.parentSeries = this;
        }

        @Override
        public long coord() {
            return this.coord;
        }

        @Override
        public long from() {
            return this.from;
        }

        @Override
        public long to() {
            return this.to;
        }

        @Override
        void allContainedFrameSides(List<FrameSide> result) {
            result.clear();
            result.add(this.firstSide);
            if (this.otherSides != null) {
                result.addAll(this.otherSides);
            }
        }

        @Override
        FrameSide transversalFrameSideFrom() {
            return this.sideFrom;
        }

        @Override
        FrameSide transversalFrameSideTo() {
            return this.sideTo;
        }

        boolean expand(FrameSide followingSide) {
            if (followingSide.isHorizontal() != this.isHorizontal() || followingSide.coord() != this.coord() || followingSide.first != this.first) {
                return false;
            }
            long followingFrom = followingSide.from();
            long followingTo = followingSide.to();
            if (followingFrom > this.to || followingTo < this.from) {
                return false;
            }
            if (followingFrom < this.from) {
                this.from = followingFrom;
                this.sideFrom = followingSide.transversalFrameSideFrom();
            }
            if (followingTo > this.to) {
                this.to = followingTo;
                this.sideTo = followingSide.transversalFrameSideTo();
            }
            if (this.otherSides == null) {
                this.otherSides = new ArrayList<FrameSide>();
            }
            this.otherSides.add(followingSide);
            followingSide.parentSeries = this;
            return true;
        }
    }

    public static class HorizontalBoundaryLink
    extends BoundaryLink {
        final VerticalSideSeries transversalSeriesFrom;
        final VerticalSideSeries transversalSeriesTo;
        VerticalBoundaryLink linkFrom = null;
        VerticalBoundaryLink linkTo = null;

        private HorizontalBoundaryLink(HorizontalSideSeries parentSeries, VerticalSideSeries transversalSeriesFrom, VerticalSideSeries transversalSeriesTo) {
            super(parentSeries, transversalSeriesFrom.coord(), transversalSeriesTo.coord());
            this.transversalSeriesFrom = transversalSeriesFrom;
            this.transversalSeriesTo = transversalSeriesTo;
        }

        @Override
        public boolean isHorizontal() {
            return true;
        }

        @Override
        public VerticalBoundaryLink linkFrom() {
            return this.linkFrom;
        }

        @Override
        public VerticalBoundaryLink linkTo() {
            return this.linkTo;
        }

        @Override
        public IRectangularArea equivalentRectangle() {
            return IRectangularArea.valueOf(this.from, this.parentSeries.frameSideCoord(), this.to - 1L, this.parentSeries.frameSideCoord());
        }

        void setNeighbour(VerticalBoundaryLink neighbour) {
            if (neighbour.parentSeries == this.transversalSeriesFrom) {
                this.linkFrom = neighbour;
            } else if (neighbour.parentSeries == this.transversalSeriesTo) {
                this.linkTo = neighbour;
            } else {
                throw new AssertionError((Object)"Attempt to assing vertical neighbour from alien side series");
            }
        }
    }

    public static class VerticalBoundaryLink
    extends BoundaryLink {
        final HorizontalBoundaryLink linkFrom;
        final HorizontalBoundaryLink linkTo;

        private VerticalBoundaryLink(VerticalSideSeries parentSeries, HorizontalBoundaryLink linkFrom, HorizontalBoundaryLink linkTo) {
            super(parentSeries, linkFrom.coord(), linkTo.coord());
            this.linkFrom = linkFrom;
            this.linkTo = linkTo;
        }

        @Override
        public boolean isHorizontal() {
            return false;
        }

        @Override
        public HorizontalBoundaryLink linkFrom() {
            return this.linkFrom;
        }

        @Override
        public HorizontalBoundaryLink linkTo() {
            return this.linkTo;
        }

        @Override
        public IRectangularArea equivalentRectangle() {
            return IRectangularArea.valueOf(this.parentSeries.frameSideCoord(), this.from, this.parentSeries.frameSideCoord(), this.to - 1L);
        }
    }
}

