/*
 * Decompiled with CFR 0.152.
 */
package net.algart.additions.arrays;

import java.util.Arrays;
import java.util.Objects;
import java.util.stream.IntStream;
import net.algart.arrays.Arrays;
import net.algart.arrays.BitArray;
import net.algart.arrays.ByteArray;
import net.algart.arrays.CharArray;
import net.algart.arrays.DirectAccessible;
import net.algart.arrays.DoubleArray;
import net.algart.arrays.FloatArray;
import net.algart.arrays.IntArray;
import net.algart.arrays.LongArray;
import net.algart.arrays.Matrix;
import net.algart.arrays.PArray;
import net.algart.arrays.ShortArray;

public final class UniformHistogram256Finder {
    private static final int DEFAULT_BUFFER_LENGTH = 16384;
    private final boolean multithreading;
    private final int bufferLength;
    private final long[][] threadHistograms;
    private final boolean[][] threadMaskBuffers;
    private final long[] splitters;
    private long[] resultHistogram = null;
    private long resultCardinality = -1L;
    private double resultBarWidth = Double.NaN;
    private double resultSum = Double.NaN;

    private UniformHistogram256Finder(int bufferLength, boolean multithreading) {
        if (bufferLength < 256) {
            throw new IllegalArgumentException("Too little buffer length " + bufferLength + " (must not be < 256)");
        }
        this.bufferLength = bufferLength;
        int cpuCount = Arrays.SystemSettings.cpuCount();
        int numberOfTasks = !multithreading || cpuCount == 1 ? 1 : 4 * cpuCount;
        this.multithreading = numberOfTasks > 1;
        this.threadHistograms = new long[numberOfTasks][256];
        this.threadMaskBuffers = new boolean[numberOfTasks][bufferLength];
        this.splitters = new long[numberOfTasks + 1];
    }

    public static UniformHistogram256Finder newInstance(boolean multithreading) {
        return UniformHistogram256Finder.newInstance(16384, multithreading);
    }

    public static UniformHistogram256Finder newInstance(int bufferLength, boolean multithreading) {
        return new UniformHistogram256Finder(bufferLength, multithreading);
    }

    public UniformHistogram256Finder find(Matrix<? extends PArray> data, Matrix<? extends BitArray> mask) {
        Objects.requireNonNull(data, "Null data matrix");
        if (mask != null && !data.dimEquals(mask)) {
            throw new IllegalArgumentException("Data and mask matrices have different dimensions: " + String.valueOf(data) + " and " + String.valueOf(mask));
        }
        return this.find((PArray)data.array(), mask == null ? null : (BitArray)mask.array());
    }

    public UniformHistogram256Finder find(PArray data, BitArray mask) {
        return this.clear().accumulate(data, mask);
    }

    public UniformHistogram256Finder clear() {
        for (long[] histogram : this.threadHistograms) {
            Arrays.fill(histogram, 0L);
        }
        return this;
    }

    public UniformHistogram256Finder accumulate(PArray data, BitArray mask) {
        Objects.requireNonNull(data, "Null data");
        return this.accumulate(data, mask, data.length());
    }

    public UniformHistogram256Finder accumulate(PArray data, BitArray mask, long length) {
        Objects.requireNonNull(data, "Null data");
        if (length < 0L) {
            throw new IllegalArgumentException("Negative length = " + length);
        }
        if (data.length() < length) {
            throw new IllegalArgumentException("Length of data " + data.length() + " is less than specified length " + length);
        }
        if (mask != null && mask.length() != data.length()) {
            throw new IllegalArgumentException("Different lengths of data (" + data.length() + ") and mask (" + mask.length() + ")");
        }
        net.algart.arrays.Arrays.splitToRanges((long[])this.splitters, (long)length);
        Class elementType = data.elementType();
        IntStream stream = IntStream.range(0, this.splitters.length - 1);
        if (this.multithreading) {
            stream = stream.parallel();
        }
        if (elementType == Character.TYPE) {
            CharArray array = (CharArray)data;
            stream.forEach(rangeIndex -> this.histogramOfChars(array, mask, rangeIndex));
        } else if (elementType == Boolean.TYPE) {
            BitArray array = (BitArray)data;
            stream.forEach(rangeIndex -> this.histogramOfBits(array, mask, rangeIndex));
        } else if (elementType == Byte.TYPE) {
            ByteArray array = (ByteArray)data;
            stream.forEach(rangeIndex -> this.histogramOfBytes(array, mask, rangeIndex));
        } else if (elementType == Short.TYPE) {
            ShortArray array = (ShortArray)data;
            stream.forEach(rangeIndex -> this.histogramOfShorts(array, mask, rangeIndex));
        } else if (elementType == Integer.TYPE) {
            IntArray array = (IntArray)data;
            stream.forEach(rangeIndex -> this.histogramOfInts(array, mask, rangeIndex));
        } else if (elementType == Long.TYPE) {
            LongArray array = (LongArray)data;
            stream.forEach(rangeIndex -> this.histogramOfLongs(array, mask, rangeIndex));
        } else if (elementType == Float.TYPE) {
            FloatArray array = (FloatArray)data;
            stream.forEach(rangeIndex -> this.histogramOfFloats(array, mask, rangeIndex));
        } else if (elementType == Double.TYPE) {
            DoubleArray array = (DoubleArray)data;
            stream.forEach(rangeIndex -> this.histogramOfDoubles(array, mask, rangeIndex));
        } else {
            throw new AssertionError((Object)("Impossible element type " + String.valueOf(elementType)));
        }
        this.buildHistogram(elementType);
        this.findCardinality();
        this.findSumAndBarWidth(elementType);
        return this;
    }

    public long[] histogram() {
        this.checkReady();
        return this.resultHistogram;
    }

    public double barWidth() {
        this.checkReady();
        return this.resultBarWidth;
    }

    public long cardinality() {
        this.checkReady();
        return this.resultCardinality;
    }

    public double sum() {
        this.checkReady();
        return this.resultSum;
    }

    public double mean() {
        this.checkReady();
        return this.resultSum / (double)this.resultCardinality;
    }

    public void checkReady() {
        if (this.resultHistogram == null) {
            throw new IllegalStateException("Histogram is not calculated yet");
        }
    }

    public static double barWidth(Class<?> elementType) {
        Objects.requireNonNull(elementType, "Null elementType");
        if (elementType == Character.TYPE) {
            return 256.0;
        }
        if (elementType == Boolean.TYPE) {
            return 1.0;
        }
        if (elementType == Byte.TYPE) {
            return 1.0;
        }
        if (elementType == Short.TYPE) {
            return 256.0;
        }
        if (elementType == Integer.TYPE) {
            return 8388608.0;
        }
        if (elementType == Long.TYPE) {
            return 1.40737488355328E14;
        }
        if (elementType == Float.TYPE) {
            return 0.00390625;
        }
        if (elementType == Double.TYPE) {
            return 0.00390625;
        }
        throw new UnsupportedOperationException("Non-primitive element type " + String.valueOf(elementType) + " is not supported");
    }

    private void buildHistogram(Class<?> elementType) {
        int numberOfBars;
        int n = numberOfBars = elementType == Boolean.TYPE ? 2 : 256;
        if (this.resultHistogram == null || this.resultHistogram.length != numberOfBars) {
            this.resultHistogram = new long[numberOfBars];
        } else {
            Arrays.fill(this.resultHistogram, 0L);
        }
        for (long[] histogram : this.threadHistograms) {
            for (int k = 0; k < this.resultHistogram.length; ++k) {
                int n2 = k;
                this.resultHistogram[n2] = this.resultHistogram[n2] + histogram[k];
            }
        }
    }

    private void findCardinality() {
        this.resultCardinality = Arrays.stream(this.resultHistogram).sum();
    }

    private void findSumAndBarWidth(Class<?> elementType) {
        double sum = 0.0;
        for (int i = 1; i < this.resultHistogram.length; ++i) {
            sum += (double)i * (double)this.resultHistogram[i];
        }
        this.resultBarWidth = UniformHistogram256Finder.barWidth(elementType);
        if (elementType == Character.TYPE) {
            this.resultSum = sum * this.resultBarWidth;
        } else if (elementType == Boolean.TYPE) {
            this.resultSum = sum;
        } else if (elementType == Byte.TYPE) {
            this.resultSum = sum;
        } else if (elementType == Short.TYPE) {
            this.resultSum = sum * this.resultBarWidth;
        } else if (elementType == Integer.TYPE) {
            this.resultSum = sum * this.resultBarWidth;
        } else if (elementType == Long.TYPE) {
            this.resultSum = sum * this.resultBarWidth;
        } else if (elementType == Float.TYPE) {
            this.resultSum = sum / 256.0;
        } else if (elementType == Double.TYPE) {
            this.resultSum = sum / 256.0;
        } else {
            throw new UnsupportedOperationException("Non-primitive element type " + String.valueOf(elementType) + " is not supported");
        }
    }

    private void histogramOfChars(CharArray data, BitArray mask, int rangeIndex) {
        DirectAccessible da;
        long from = this.splitters[rangeIndex];
        long to = this.splitters[rangeIndex + 1];
        long[] histogram = this.threadHistograms[rangeIndex];
        if (mask != null) {
            DirectAccessible da2;
            boolean[] maskBuffer = this.threadMaskBuffers[rangeIndex];
            if (data instanceof DirectAccessible && (da2 = (DirectAccessible)data).hasJavaArray()) {
                int offset = da2.javaArrayOffset();
                char[] array = (char[])da2.javaArray();
                int intTo = (int)to;
                assert ((long)intTo == to);
                int p = (int)from;
                while ((long)p < to) {
                    int length = Math.min(this.bufferLength, intTo - p);
                    mask.getData((long)p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.charRange(array, histogram, maskBuffer, offset + p, length);
                    p += length;
                }
            } else {
                int length;
                for (long p = from; p < to; p += (long)length) {
                    length = (int)Math.min((long)this.bufferLength, to - p);
                    mask.getData(p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.charRange(data, histogram, maskBuffer, p, length);
                }
            }
        } else if (data instanceof DirectAccessible && (da = (DirectAccessible)data).hasJavaArray()) {
            int offset = da.javaArrayOffset();
            char[] array = (char[])da.javaArray();
            int intTo = (int)to;
            int intFrom = (int)from;
            assert ((long)intFrom == from && (long)intTo == to);
            UniformHistogram256Finder.charRange(array, histogram, offset + intFrom, intTo - intFrom);
        } else {
            UniformHistogram256Finder.charRange(data, histogram, from, to - from);
        }
    }

    private static void charRange(CharArray data, long[] histogram, boolean[] mask, long p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            char v = data.getChar(p + (long)k);
            int n = index = v >> 8;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void charRange(char[] data, long[] histogram, boolean[] mask, int p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            char v = data[p + k];
            int n = index = v >> 8;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void charRange(CharArray data, long[] histogram, long p, long length) {
        long to = p + length;
        for (long i = p; i < to; ++i) {
            int index;
            char v = data.getChar(i);
            int n = index = v >> 8;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void charRange(char[] data, long[] histogram, int p, int length) {
        int to = p + length;
        for (int i = p; i < to; ++i) {
            int index;
            char v = data[i];
            int n = index = v >> 8;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private void histogramOfBits(BitArray data, BitArray mask, int rangeIndex) {
        DirectAccessible da;
        long from = this.splitters[rangeIndex];
        long to = this.splitters[rangeIndex + 1];
        long[] histogram = this.threadHistograms[rangeIndex];
        if (mask != null) {
            DirectAccessible da2;
            boolean[] maskBuffer = this.threadMaskBuffers[rangeIndex];
            if (data instanceof DirectAccessible && (da2 = (DirectAccessible)data).hasJavaArray()) {
                int offset = da2.javaArrayOffset();
                boolean[] array = (boolean[])da2.javaArray();
                int intTo = (int)to;
                assert ((long)intTo == to);
                int p = (int)from;
                while ((long)p < to) {
                    int length = Math.min(this.bufferLength, intTo - p);
                    mask.getData((long)p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.booleanRange(array, histogram, maskBuffer, offset + p, length);
                    p += length;
                }
            } else {
                int length;
                for (long p = from; p < to; p += (long)length) {
                    length = (int)Math.min((long)this.bufferLength, to - p);
                    mask.getData(p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.booleanRange(data, histogram, maskBuffer, p, length);
                }
            }
        } else if (data instanceof DirectAccessible && (da = (DirectAccessible)data).hasJavaArray()) {
            int offset = da.javaArrayOffset();
            boolean[] array = (boolean[])da.javaArray();
            int intTo = (int)to;
            int intFrom = (int)from;
            assert ((long)intFrom == from && (long)intTo == to);
            UniformHistogram256Finder.booleanRange(array, histogram, offset + intFrom, intTo - intFrom);
        } else {
            UniformHistogram256Finder.booleanRange(data, histogram, from, to - from);
        }
    }

    private static void booleanRange(BitArray data, long[] histogram, boolean[] mask, long p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            boolean v = data.getBit(p + (long)k);
            int n = index = v ? 1 : 0;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void booleanRange(boolean[] data, long[] histogram, boolean[] mask, int p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            boolean v = data[p + k];
            int n = index = v ? 1 : 0;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void booleanRange(BitArray data, long[] histogram, long p, long length) {
        long to = p + length;
        for (long i = p; i < to; ++i) {
            int index;
            boolean v = data.getBit(i);
            int n = index = v ? 1 : 0;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void booleanRange(boolean[] data, long[] histogram, int p, int length) {
        int to = p + length;
        for (int i = p; i < to; ++i) {
            int index;
            boolean v = data[i];
            int n = index = v ? 1 : 0;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private void histogramOfBytes(ByteArray data, BitArray mask, int rangeIndex) {
        DirectAccessible da;
        long from = this.splitters[rangeIndex];
        long to = this.splitters[rangeIndex + 1];
        long[] histogram = this.threadHistograms[rangeIndex];
        if (mask != null) {
            DirectAccessible da2;
            boolean[] maskBuffer = this.threadMaskBuffers[rangeIndex];
            if (data instanceof DirectAccessible && (da2 = (DirectAccessible)data).hasJavaArray()) {
                int offset = da2.javaArrayOffset();
                byte[] array = (byte[])da2.javaArray();
                int intTo = (int)to;
                assert ((long)intTo == to);
                int p = (int)from;
                while ((long)p < to) {
                    int length = Math.min(this.bufferLength, intTo - p);
                    mask.getData((long)p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.byteRange(array, histogram, maskBuffer, offset + p, length);
                    p += length;
                }
            } else {
                int length;
                for (long p = from; p < to; p += (long)length) {
                    length = (int)Math.min((long)this.bufferLength, to - p);
                    mask.getData(p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.byteRange(data, histogram, maskBuffer, p, length);
                }
            }
        } else if (data instanceof DirectAccessible && (da = (DirectAccessible)data).hasJavaArray()) {
            int offset = da.javaArrayOffset();
            byte[] array = (byte[])da.javaArray();
            int intTo = (int)to;
            int intFrom = (int)from;
            assert ((long)intFrom == from && (long)intTo == to);
            UniformHistogram256Finder.byteRange(array, histogram, offset + intFrom, intTo - intFrom);
        } else {
            UniformHistogram256Finder.byteRange(data, histogram, from, to - from);
        }
    }

    private static void byteRange(ByteArray data, long[] histogram, boolean[] mask, long p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            int v = data.getByte(p + (long)k);
            int n = index = v & 0xFF;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void byteRange(byte[] data, long[] histogram, boolean[] mask, int p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            byte v = data[p + k];
            int n = index = v & 0xFF;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void byteRange(ByteArray data, long[] histogram, long p, long length) {
        long to = p + length;
        for (long i = p; i < to; ++i) {
            int index;
            int v = data.getByte(i);
            int n = index = v & 0xFF;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void byteRange(byte[] data, long[] histogram, int p, int length) {
        int to = p + length;
        for (int i = p; i < to; ++i) {
            int index;
            byte v = data[i];
            int n = index = v & 0xFF;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private void histogramOfShorts(ShortArray data, BitArray mask, int rangeIndex) {
        DirectAccessible da;
        long from = this.splitters[rangeIndex];
        long to = this.splitters[rangeIndex + 1];
        long[] histogram = this.threadHistograms[rangeIndex];
        if (mask != null) {
            DirectAccessible da2;
            boolean[] maskBuffer = this.threadMaskBuffers[rangeIndex];
            if (data instanceof DirectAccessible && (da2 = (DirectAccessible)data).hasJavaArray()) {
                int offset = da2.javaArrayOffset();
                short[] array = (short[])da2.javaArray();
                int intTo = (int)to;
                assert ((long)intTo == to);
                int p = (int)from;
                while ((long)p < to) {
                    int length = Math.min(this.bufferLength, intTo - p);
                    mask.getData((long)p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.shortRange(array, histogram, maskBuffer, offset + p, length);
                    p += length;
                }
            } else {
                int length;
                for (long p = from; p < to; p += (long)length) {
                    length = (int)Math.min((long)this.bufferLength, to - p);
                    mask.getData(p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.shortRange(data, histogram, maskBuffer, p, length);
                }
            }
        } else if (data instanceof DirectAccessible && (da = (DirectAccessible)data).hasJavaArray()) {
            int offset = da.javaArrayOffset();
            short[] array = (short[])da.javaArray();
            int intTo = (int)to;
            int intFrom = (int)from;
            assert ((long)intFrom == from && (long)intTo == to);
            UniformHistogram256Finder.shortRange(array, histogram, offset + intFrom, intTo - intFrom);
        } else {
            UniformHistogram256Finder.shortRange(data, histogram, from, to - from);
        }
    }

    private static void shortRange(ShortArray data, long[] histogram, boolean[] mask, long p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            int v = data.getShort(p + (long)k);
            int n = index = (v & 0xFFFF) >> 8;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void shortRange(short[] data, long[] histogram, boolean[] mask, int p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            short v = data[p + k];
            int n = index = (v & 0xFFFF) >> 8;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void shortRange(ShortArray data, long[] histogram, long p, long length) {
        long to = p + length;
        for (long i = p; i < to; ++i) {
            int index;
            int v = data.getShort(i);
            int n = index = (v & 0xFFFF) >> 8;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void shortRange(short[] data, long[] histogram, int p, int length) {
        int to = p + length;
        for (int i = p; i < to; ++i) {
            int index;
            short v = data[i];
            int n = index = (v & 0xFFFF) >> 8;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private void histogramOfInts(IntArray data, BitArray mask, int rangeIndex) {
        DirectAccessible da;
        long from = this.splitters[rangeIndex];
        long to = this.splitters[rangeIndex + 1];
        long[] histogram = this.threadHistograms[rangeIndex];
        if (mask != null) {
            DirectAccessible da2;
            boolean[] maskBuffer = this.threadMaskBuffers[rangeIndex];
            if (data instanceof DirectAccessible && (da2 = (DirectAccessible)data).hasJavaArray()) {
                int offset = da2.javaArrayOffset();
                int[] array = (int[])da2.javaArray();
                int intTo = (int)to;
                assert ((long)intTo == to);
                int p = (int)from;
                while ((long)p < to) {
                    int length = Math.min(this.bufferLength, intTo - p);
                    mask.getData((long)p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.intRange(array, histogram, maskBuffer, offset + p, length);
                    p += length;
                }
            } else {
                int length;
                for (long p = from; p < to; p += (long)length) {
                    length = (int)Math.min((long)this.bufferLength, to - p);
                    mask.getData(p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.intRange(data, histogram, maskBuffer, p, length);
                }
            }
        } else if (data instanceof DirectAccessible && (da = (DirectAccessible)data).hasJavaArray()) {
            int offset = da.javaArrayOffset();
            int[] array = (int[])da.javaArray();
            int intTo = (int)to;
            int intFrom = (int)from;
            assert ((long)intFrom == from && (long)intTo == to);
            UniformHistogram256Finder.intRange(array, histogram, offset + intFrom, intTo - intFrom);
        } else {
            UniformHistogram256Finder.intRange(data, histogram, from, to - from);
        }
    }

    private static void intRange(IntArray data, long[] histogram, boolean[] mask, long p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            int v = data.getInt(p + (long)k);
            int n = index = v < 0 ? 0 : v >>> 23;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void intRange(int[] data, long[] histogram, boolean[] mask, int p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            int v = data[p + k];
            int n = index = v < 0 ? 0 : v >>> 23;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void intRange(IntArray data, long[] histogram, long p, long length) {
        long to = p + length;
        for (long i = p; i < to; ++i) {
            int index;
            int v = data.getInt(i);
            int n = index = v < 0 ? 0 : v >>> 23;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void intRange(int[] data, long[] histogram, int p, int length) {
        int to = p + length;
        for (int i = p; i < to; ++i) {
            int index;
            int v = data[i];
            int n = index = v < 0 ? 0 : v >>> 23;
            histogram[n] = histogram[n] + 1L;
        }
    }

    private void histogramOfLongs(LongArray data, BitArray mask, int rangeIndex) {
        DirectAccessible da;
        long from = this.splitters[rangeIndex];
        long to = this.splitters[rangeIndex + 1];
        long[] histogram = this.threadHistograms[rangeIndex];
        if (mask != null) {
            DirectAccessible da2;
            boolean[] maskBuffer = this.threadMaskBuffers[rangeIndex];
            if (data instanceof DirectAccessible && (da2 = (DirectAccessible)data).hasJavaArray()) {
                int offset = da2.javaArrayOffset();
                long[] array = (long[])da2.javaArray();
                int intTo = (int)to;
                assert ((long)intTo == to);
                int p = (int)from;
                while ((long)p < to) {
                    int length = Math.min(this.bufferLength, intTo - p);
                    mask.getData((long)p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.longRange(array, histogram, maskBuffer, offset + p, length);
                    p += length;
                }
            } else {
                int length;
                for (long p = from; p < to; p += (long)length) {
                    length = (int)Math.min((long)this.bufferLength, to - p);
                    mask.getData(p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.longRange(data, histogram, maskBuffer, p, length);
                }
            }
        } else if (data instanceof DirectAccessible && (da = (DirectAccessible)data).hasJavaArray()) {
            int offset = da.javaArrayOffset();
            long[] array = (long[])da.javaArray();
            int intTo = (int)to;
            int intFrom = (int)from;
            assert ((long)intFrom == from && (long)intTo == to);
            UniformHistogram256Finder.longRange(array, histogram, offset + intFrom, intTo - intFrom);
        } else {
            UniformHistogram256Finder.longRange(data, histogram, from, to - from);
        }
    }

    private static void longRange(LongArray data, long[] histogram, boolean[] mask, long p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            long v = data.getLong(p + (long)k);
            int n = index = v < 0L ? 0 : (int)(v >>> 47);
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void longRange(long[] data, long[] histogram, boolean[] mask, int p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            long v = data[p + k];
            int n = index = v < 0L ? 0 : (int)(v >>> 47);
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void longRange(LongArray data, long[] histogram, long p, long length) {
        long to = p + length;
        for (long i = p; i < to; ++i) {
            int index;
            long v = data.getLong(i);
            int n = index = v < 0L ? 0 : (int)(v >>> 47);
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void longRange(long[] data, long[] histogram, int p, int length) {
        int to = p + length;
        for (int i = p; i < to; ++i) {
            int index;
            long v = data[i];
            int n = index = v < 0L ? 0 : (int)(v >>> 47);
            histogram[n] = histogram[n] + 1L;
        }
    }

    private void histogramOfFloats(FloatArray data, BitArray mask, int rangeIndex) {
        DirectAccessible da;
        long from = this.splitters[rangeIndex];
        long to = this.splitters[rangeIndex + 1];
        long[] histogram = this.threadHistograms[rangeIndex];
        if (mask != null) {
            DirectAccessible da2;
            boolean[] maskBuffer = this.threadMaskBuffers[rangeIndex];
            if (data instanceof DirectAccessible && (da2 = (DirectAccessible)data).hasJavaArray()) {
                int offset = da2.javaArrayOffset();
                float[] array = (float[])da2.javaArray();
                int intTo = (int)to;
                assert ((long)intTo == to);
                int p = (int)from;
                while ((long)p < to) {
                    int length = Math.min(this.bufferLength, intTo - p);
                    mask.getData((long)p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.floatRange(array, histogram, maskBuffer, offset + p, length);
                    p += length;
                }
            } else {
                int length;
                for (long p = from; p < to; p += (long)length) {
                    length = (int)Math.min((long)this.bufferLength, to - p);
                    mask.getData(p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.floatRange(data, histogram, maskBuffer, p, length);
                }
            }
        } else if (data instanceof DirectAccessible && (da = (DirectAccessible)data).hasJavaArray()) {
            int offset = da.javaArrayOffset();
            float[] array = (float[])da.javaArray();
            int intTo = (int)to;
            int intFrom = (int)from;
            assert ((long)intFrom == from && (long)intTo == to);
            UniformHistogram256Finder.floatRange(array, histogram, offset + intFrom, intTo - intFrom);
        } else {
            UniformHistogram256Finder.floatRange(data, histogram, from, to - from);
        }
    }

    private static void floatRange(FloatArray data, long[] histogram, boolean[] mask, long p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            float v = data.getFloat(p + (long)k);
            int n = index = (double)v < 0.0 ? 0 : ((double)v > 0.999999 ? 255 : (int)((double)v * 256.0));
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void floatRange(float[] data, long[] histogram, boolean[] mask, int p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            float v = data[p + k];
            int n = index = (double)v < 0.0 ? 0 : ((double)v > 0.999999 ? 255 : (int)((double)v * 256.0));
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void floatRange(FloatArray data, long[] histogram, long p, long length) {
        long to = p + length;
        for (long i = p; i < to; ++i) {
            int index;
            float v = data.getFloat(i);
            int n = index = (double)v < 0.0 ? 0 : ((double)v > 0.999999 ? 255 : (int)((double)v * 256.0));
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void floatRange(float[] data, long[] histogram, int p, int length) {
        int to = p + length;
        for (int i = p; i < to; ++i) {
            int index;
            float v = data[i];
            int n = index = (double)v < 0.0 ? 0 : ((double)v > 0.999999 ? 255 : (int)((double)v * 256.0));
            histogram[n] = histogram[n] + 1L;
        }
    }

    private void histogramOfDoubles(DoubleArray data, BitArray mask, int rangeIndex) {
        DirectAccessible da;
        long from = this.splitters[rangeIndex];
        long to = this.splitters[rangeIndex + 1];
        long[] histogram = this.threadHistograms[rangeIndex];
        if (mask != null) {
            DirectAccessible da2;
            boolean[] maskBuffer = this.threadMaskBuffers[rangeIndex];
            if (data instanceof DirectAccessible && (da2 = (DirectAccessible)data).hasJavaArray()) {
                int offset = da2.javaArrayOffset();
                double[] array = (double[])da2.javaArray();
                int intTo = (int)to;
                assert ((long)intTo == to);
                int p = (int)from;
                while ((long)p < to) {
                    int length = Math.min(this.bufferLength, intTo - p);
                    mask.getData((long)p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.doubleRange(array, histogram, maskBuffer, offset + p, length);
                    p += length;
                }
            } else {
                int length;
                for (long p = from; p < to; p += (long)length) {
                    length = (int)Math.min((long)this.bufferLength, to - p);
                    mask.getData(p, (Object)maskBuffer, 0, length);
                    UniformHistogram256Finder.doubleRange(data, histogram, maskBuffer, p, length);
                }
            }
        } else if (data instanceof DirectAccessible && (da = (DirectAccessible)data).hasJavaArray()) {
            int offset = da.javaArrayOffset();
            double[] array = (double[])da.javaArray();
            int intTo = (int)to;
            int intFrom = (int)from;
            assert ((long)intFrom == from && (long)intTo == to);
            UniformHistogram256Finder.doubleRange(array, histogram, offset + intFrom, intTo - intFrom);
        } else {
            UniformHistogram256Finder.doubleRange(data, histogram, from, to - from);
        }
    }

    private static void doubleRange(DoubleArray data, long[] histogram, boolean[] mask, long p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            double v = data.getDouble(p + (long)k);
            int n = index = v < 0.0 ? 0 : (v > 0.999999 ? 255 : (int)(v * 256.0));
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void doubleRange(double[] data, long[] histogram, boolean[] mask, int p, int length) {
        for (int k = 0; k < length; ++k) {
            int index;
            if (!mask[k]) continue;
            double v = data[p + k];
            int n = index = v < 0.0 ? 0 : (v > 0.999999 ? 255 : (int)(v * 256.0));
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void doubleRange(DoubleArray data, long[] histogram, long p, long length) {
        long to = p + length;
        for (long i = p; i < to; ++i) {
            int index;
            double v = data.getDouble(i);
            int n = index = v < 0.0 ? 0 : (v > 0.999999 ? 255 : (int)(v * 256.0));
            histogram[n] = histogram[n] + 1L;
        }
    }

    private static void doubleRange(double[] data, long[] histogram, int p, int length) {
        int to = p + length;
        for (int i = p; i < to; ++i) {
            int index;
            double v = data[i];
            int n = index = v < 0.0 ? 0 : (v > 0.999999 ? 255 : (int)(v * 256.0));
            histogram[n] = histogram[n] + 1L;
        }
    }
}

