/*
 * Decompiled with CFR 0.152.
 */
package net.algart.matrices.tiff.tiles;

import java.nio.ByteOrder;
import java.util.Collection;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import net.algart.arrays.Matrices;
import net.algart.arrays.Matrix;
import net.algart.arrays.TooLargeArrayException;
import net.algart.matrices.tiff.TiffException;
import net.algart.matrices.tiff.TiffIFD;
import net.algart.matrices.tiff.TiffSampleType;
import net.algart.matrices.tiff.tags.TagCompression;
import net.algart.matrices.tiff.tiles.TiffIOMap;
import net.algart.matrices.tiff.tiles.TiffTile;
import net.algart.matrices.tiff.tiles.TiffTileIndex;

public sealed class TiffMap
permits TiffIOMap {
    public static final int MAX_TILE_INDEX = 1000000000;
    private final TiffIFD ifd;
    private final boolean resizable;
    private final Map<TiffTileIndex, TiffTile> tileMap = new LinkedHashMap<TiffTileIndex, TiffTile>();
    private final boolean planarSeparated;
    private final int numberOfChannels;
    private final int numberOfSeparatedPlanes;
    private final int tileSamplesPerPixel;
    private final int[] bitsPerSample;
    private final int alignedBitsPerSample;
    private final int bitsPerUnpackedSample;
    private final int tileAlignedBitsPerPixel;
    private final int totalAlignedBitsPerPixel;
    private final TiffSampleType sampleType;
    private final boolean wholeBytes;
    private final Class<?> elementType;
    private final ByteOrder byteOrder;
    private final long maxNumberOfSamplesInArray;
    private final TilingMode tilingMode;
    private final int tileSizeX;
    private final int tileSizeY;
    private final int tileSizeInPixels;
    private final int tileSizeInBytes;
    private final int compressionCode;
    private final TagCompression compression;
    private final int photometricInterpretationCode;
    private volatile int dimX = 0;
    private volatile int dimY = 0;
    private volatile int gridCountX = 0;
    private volatile int gridCountY = 0;
    private volatile int numberOfGridTiles = 0;

    public TiffMap(TiffIFD ifd, boolean resizable) {
        this.ifd = Objects.requireNonNull(ifd, "Null IFD");
        this.resizable = resizable;
        boolean hasImageDimensions = ifd.hasImageDimensions();
        try {
            if (!hasImageDimensions && !resizable) {
                throw new IllegalArgumentException("TIFF image sizes (ImageWidth and ImageLength tags) are not specified; it is not allowed for non-resizable tile map");
            }
            TilingMode tilingMode = this.tilingMode = ifd.hasTileInformation() ? TilingMode.TILE_GRID : TilingMode.STRIPS;
            if (resizable && !this.tilingMode.isTileGrid()) {
                throw new IllegalArgumentException("TIFF image is not tiled (TileWidth and TileLength tags are not specified); it is not allowed for resizable tile map: any processing TIFF image, such as writing its fragments, requires either knowing its final fixed sizes, or splitting image into tiles with known fixed sizes");
            }
            this.planarSeparated = ifd.isPlanarSeparated();
            this.numberOfChannels = ifd.getSamplesPerPixel();
            assert (this.numberOfChannels <= 128);
            this.numberOfSeparatedPlanes = this.planarSeparated ? this.numberOfChannels : 1;
            this.tileSamplesPerPixel = this.planarSeparated ? 1 : this.numberOfChannels;
            this.bitsPerSample = (int[])ifd.getBitsPerSample().clone();
            this.alignedBitsPerSample = TiffIFD.alignedBitDepth(this.bitsPerSample);
            assert ((long)this.numberOfChannels * (long)this.alignedBitsPerSample < 32768L);
            this.tileAlignedBitsPerPixel = this.tileSamplesPerPixel * this.alignedBitsPerSample;
            this.totalAlignedBitsPerPixel = this.numberOfChannels * this.alignedBitsPerSample;
            this.sampleType = ifd.sampleType();
            this.wholeBytes = this.sampleType.isWholeBytes();
            if (this.wholeBytes != ((this.alignedBitsPerSample & 7) == 0)) {
                throw new ConcurrentModificationException("Corrupted IFD, probably from a parallel thread (sample type " + String.valueOf((Object)this.sampleType) + " is" + (this.wholeBytes ? "" : " NOT") + " whole-bytes, but we have " + this.alignedBitsPerSample + " bits/sample)");
            }
            if (this.totalAlignedBitsPerPixel == 1 != this.sampleType.isBinary()) {
                throw new ConcurrentModificationException("Corrupted IFD, probably from a parallel thread (sample type is " + String.valueOf((Object)this.sampleType) + ", but we have " + this.totalAlignedBitsPerPixel + " bits/pixel)");
            }
            if (this.sampleType.isBinary() && this.numberOfChannels > 1) {
                throw new AssertionError((Object)("Binary IFD for " + this.numberOfChannels + " > 1 channels is not supported: invalid TiffIFD class"));
            }
            this.bitsPerUnpackedSample = this.sampleType.bitsPerSample();
            if (this.bitsPerUnpackedSample < this.alignedBitsPerSample) {
                throw new AssertionError((Object)(String.valueOf((Object)this.sampleType) + ".bitsPerSample() = " + this.bitsPerUnpackedSample + " is too little: less than ifd.alignedBitDepth() = " + this.alignedBitsPerSample));
            }
            this.elementType = this.sampleType.elementType();
            this.byteOrder = ifd.getByteOrder();
            this.maxNumberOfSamplesInArray = this.sampleType.maxNumberOfSamplesInArray();
            this.tileSizeX = ifd.getTileSizeX();
            this.tileSizeY = ifd.getTileSizeY();
            assert (this.tileSizeX > 0 && this.tileSizeY > 0) : "non-positive tile sizes are not checked in IFD methods";
            this.compressionCode = ifd.getCompressionCode();
            this.compression = ifd.optCompression();
            this.photometricInterpretationCode = ifd.getPhotometricInterpretationCode();
            if (hasImageDimensions) {
                this.setDimensions(ifd.getImageDimX(), ifd.getImageDimY(), false);
            }
            if ((long)this.tileSizeX * (long)this.tileSizeY > Integer.MAX_VALUE) {
                throw new IllegalArgumentException("Very large " + (this.tilingMode.isTileGrid() ? "TIFF tiles " : "TIFF strips ") + this.tileSizeX + "x" + this.tileSizeY + " >= 2^31 pixels are not supported");
            }
            this.tileSizeInPixels = this.tileSizeX * this.tileSizeY;
            if ((long)this.tileSizeInPixels * (long)this.tileAlignedBitsPerPixel > Integer.MAX_VALUE) {
                throw new IllegalArgumentException("Very large TIFF tiles " + this.tileSizeX + "x" + this.tileSizeY + ", " + this.tileSamplesPerPixel + " channels per " + this.alignedBitsPerSample + " bits >= 2^31 bits (256 MB) are not supported");
            }
            this.tileSizeInBytes = this.tileSizeInPixels * this.tileAlignedBitsPerPixel + 7 >>> 3;
        }
        catch (TiffException e) {
            throw new IllegalArgumentException("Illegal IFD: " + e.getMessage(), e);
        }
    }

    public TiffIFD ifd() {
        return this.ifd;
    }

    public Map<TiffTileIndex, TiffTile> tileMap() {
        return Collections.unmodifiableMap(this.tileMap);
    }

    public Set<TiffTileIndex> indexes() {
        return Collections.unmodifiableSet(this.tileMap.keySet());
    }

    public Collection<TiffTile> tiles() {
        return Collections.unmodifiableCollection(this.tileMap.values());
    }

    public boolean isResizable() {
        return this.resizable;
    }

    public boolean isPlanarSeparated() {
        return this.planarSeparated;
    }

    public int numberOfChannels() {
        return this.numberOfChannels;
    }

    public int numberOfSeparatedPlanes() {
        return this.numberOfSeparatedPlanes;
    }

    public int tileSamplesPerPixel() {
        return this.tileSamplesPerPixel;
    }

    public TiffSampleType sampleType() {
        return this.sampleType;
    }

    public boolean isBinary() {
        return this.sampleType.isBinary();
    }

    public boolean isWholeBytes() {
        return this.wholeBytes;
    }

    public int[] bitsPerSample() {
        return (int[])this.bitsPerSample.clone();
    }

    public int alignedBitsPerSample() {
        return this.alignedBitsPerSample;
    }

    public OptionalInt bytesPerSample() {
        assert (this.wholeBytes == ((this.alignedBitsPerSample & 7) == 0)) : "must be checked ins the constructor";
        return this.wholeBytes ? OptionalInt.of(this.alignedBitsPerSample >>> 3) : OptionalInt.empty();
    }

    public int bitsPerUnpackedSample() {
        return this.bitsPerUnpackedSample;
    }

    public int tileAlignedBitsPerPixel() {
        return this.tileAlignedBitsPerPixel;
    }

    public int totalAlignedBitsPerPixel() {
        return this.totalAlignedBitsPerPixel;
    }

    public Class<?> elementType() {
        return this.elementType;
    }

    public ByteOrder byteOrder() {
        return this.byteOrder;
    }

    public int sizeOfRegionWithPossibleNonStandardPrecisions(long sizeX, long sizeY) throws TiffException {
        return TiffIFD.sizeOfRegionInBytes(sizeX, sizeY, this.numberOfChannels, this.alignedBitsPerSample);
    }

    public long maxNumberOfSamplesInArray() {
        return this.maxNumberOfSamplesInArray;
    }

    public Optional<String> description() {
        return this.ifd.optDescription();
    }

    public TilingMode getTilingMode() {
        return this.tilingMode;
    }

    public int tileSizeX() {
        return this.tileSizeX;
    }

    public int tileSizeY() {
        return this.tileSizeY;
    }

    public int tileSizeInPixels() {
        return this.tileSizeInPixels;
    }

    public int tileSizeInBytes() {
        return this.tileSizeInBytes;
    }

    public int compressionCode() {
        return this.compressionCode;
    }

    public TagCompression compression() {
        return this.compression;
    }

    public int photometricInterpretationCode() {
        return this.photometricInterpretationCode;
    }

    public int dimX() {
        return this.dimX;
    }

    public int dimY() {
        return this.dimY;
    }

    public long totalSizeInPixels() {
        return (long)this.dimX * (long)this.dimY;
    }

    public long totalSizeInBytes() {
        return Math.multiplyExact(this.totalSizeInPixels(), (long)this.totalAlignedBitsPerPixel) + 7L >>> 3;
    }

    public void setDimensions(int dimX, int dimY) {
        this.setDimensions(dimX, dimY, true);
    }

    public void checkZeroDimensions() {
        if (this.dimX == 0 || this.dimY == 0) {
            throw new IllegalStateException("Zero/unset map dimensions " + this.dimX + "x" + this.dimY + " are not allowed here");
        }
    }

    public void checkTooSmallDimensionsForCurrentGrid() {
        int gridCountX = (int)((long)this.dimX + (long)this.tileSizeX - 1L) / this.tileSizeX;
        int gridCountY = (int)((long)this.dimY + (long)this.tileSizeY - 1L) / this.tileSizeY;
        assert (gridCountX <= this.gridCountX && gridCountY <= this.gridCountY) : "Grid dimensions were not correctly grown according map dimensions";
        if (gridCountX != this.gridCountX || gridCountY != this.gridCountY) {
            assert (this.resizable) : "Map dimensions mismatch to grid dimensions: impossible for non-resizable map";
            throw new IllegalStateException("Map dimensions " + this.dimX + "x" + this.dimY + " are too small for current tile grid: " + String.valueOf(this));
        }
    }

    public void expandDimensions(int newMinimalDimX, int newMinimalDimY) {
        if (this.needToExpandDimensions(newMinimalDimX, newMinimalDimY)) {
            this.setDimensions(Math.max(this.dimX, newMinimalDimX), Math.max(this.dimY, newMinimalDimY));
        }
    }

    public boolean needToExpandDimensions(int newMinimalDimX, int newMinimalDimY) {
        return newMinimalDimX > this.dimX || newMinimalDimY > this.dimY;
    }

    public int gridCountX() {
        return this.gridCountX;
    }

    public int gridCountY() {
        return this.gridCountY;
    }

    public int numberOfGridTiles() {
        return this.numberOfGridTiles;
    }

    public void expandGrid(int newMinimalGridCountX, int newMinimalGridCountY) {
        this.expandGrid(newMinimalGridCountX, newMinimalGridCountY, true);
    }

    public void checkPixelCompatibility(int numberOfChannels, TiffSampleType sampleType) throws TiffException {
        Objects.requireNonNull(sampleType, "Null sampleType");
        if (numberOfChannels <= 0) {
            throw new IllegalArgumentException("Zero or negative numberOfChannels = " + numberOfChannels);
        }
        if (numberOfChannels != this.numberOfChannels) {
            throw new TiffException("Number of channel mismatch: expected " + numberOfChannels + " channels, but TIFF image contains " + this.numberOfChannels + " channels");
        }
        if (sampleType != this.sampleType) {
            throw new TiffException("Sample type mismatch: expected elements are " + sampleType.prettyName() + ", but TIFF image contains elements " + this.sampleType.prettyName());
        }
    }

    public int linearIndex(int xIndex, int yIndex, int separatedPlaneIndex) {
        if (separatedPlaneIndex < 0 || separatedPlaneIndex >= this.numberOfSeparatedPlanes) {
            throw new IndexOutOfBoundsException("Separated plane index " + separatedPlaneIndex + " is out of range 0.." + (this.numberOfSeparatedPlanes - 1));
        }
        if (xIndex < 0 || xIndex >= this.gridCountX || yIndex < 0 || yIndex >= this.gridCountY) {
            throw new IndexOutOfBoundsException("One of X/Y-indexes (" + xIndex + ", " + yIndex + ") of the tile is out of ranges 0.." + (this.gridCountX - 1) + ", 0.." + (this.gridCountY - 1));
        }
        return (separatedPlaneIndex * this.gridCountY + yIndex) * this.gridCountX + xIndex;
    }

    public TiffTileIndex index(int x, int y) {
        return new TiffTileIndex(this, x, y, 0);
    }

    public TiffTileIndex index(int x, int y, int separatedPlaneIndex) {
        return new TiffTileIndex(this, x, y, separatedPlaneIndex);
    }

    public TiffTileIndex copyIndex(TiffTileIndex other) {
        Objects.requireNonNull(other, "Null other index");
        return new TiffTileIndex(this, other.xIndex(), other.yIndex(), other.separatedPlaneIndex());
    }

    public void checkTileIndexIFD(TiffTileIndex tileIndex) {
        Objects.requireNonNull(tileIndex, "Null tile index");
        if (tileIndex.ifd() != this.ifd) {
            throw new IllegalArgumentException("Illegal tile index: tile map cannot process tiles from different IFD");
        }
    }

    public int numberOfTiles() {
        return this.tileMap.size();
    }

    public TiffTile getOrNew(int x, int y) {
        return this.getOrNew(this.index(x, y));
    }

    public TiffTile getOrNew(int x, int y, int separatedPlaneIndex) {
        return this.getOrNew(this.index(x, y, separatedPlaneIndex));
    }

    public TiffTile getOrNew(TiffTileIndex tileIndex) {
        TiffTile result = this.get(tileIndex);
        if (result == null) {
            result = new TiffTile(tileIndex);
            this.put(result);
        }
        return result;
    }

    public TiffTile get(TiffTileIndex tileIndex) {
        this.checkTileIndexIFD(tileIndex);
        return this.tileMap.get(tileIndex);
    }

    public void put(TiffTile tile) {
        Objects.requireNonNull(tile, "Null tile");
        TiffTileIndex tileIndex = tile.index();
        this.checkTileIndexIFD(tileIndex);
        if (this.resizable) {
            this.expandGrid(tileIndex.xIndex() + 1, tileIndex.yIndex() + 1);
        } else if (tileIndex.xIndex() >= this.gridCountX || tileIndex.yIndex() >= this.gridCountY) {
            throw new IndexOutOfBoundsException("New tile is completely outside the image (out of maximal tilemap sizes) " + this.dimX + "x" + this.dimY + ": " + String.valueOf(tileIndex));
        }
        this.tileMap.put(tileIndex, tile);
    }

    public TiffTile remove(TiffTileIndex tileIndex) {
        this.checkTileIndexIFD(tileIndex);
        return this.tileMap.remove(tileIndex);
    }

    public void putAll(Collection<TiffTile> tiles) {
        Objects.requireNonNull(tiles, "Null tiles");
        tiles.forEach(this::put);
    }

    public void buildTileGrid() {
        for (int p = 0; p < this.numberOfSeparatedPlanes; ++p) {
            for (int y = 0; y < this.gridCountY; ++y) {
                for (int x = 0; x < this.gridCountX; ++x) {
                    this.getOrNew(x, y, p).cropStripToMap();
                }
            }
        }
    }

    public void cropAllStrips() {
        this.cropAll(true);
    }

    public void cropAll(boolean strippedOnly) {
        this.tileMap.values().forEach(tile -> tile.cropToMap(strippedOnly));
    }

    public boolean hasUnset() {
        return this.tileMap.values().stream().anyMatch(TiffTile::hasUnsetArea);
    }

    public void markAllAsUnset() {
        this.tileMap.values().forEach(TiffTile::markWholeTileAsUnset);
    }

    public void cropAllUnset() {
        this.tileMap.values().forEach(TiffTile::cropUnsetAreaToMap);
    }

    public List<TiffTile> findCompletedTiles() {
        return this.findTiles(TiffTile::isCompleted);
    }

    public List<TiffTile> findTiles(Predicate<TiffTile> filter) {
        Objects.requireNonNull(filter, "Null filter");
        return this.tileMap.values().stream().filter(filter).collect(Collectors.toList());
    }

    public void freeAllData() {
        this.tileMap.values().forEach(TiffTile::freeData);
    }

    public void clear(boolean clearDimensions) {
        this.tileMap.clear();
        if (clearDimensions) {
            this.setDimensions(0, 0);
            this.gridCountX = 0;
            this.gridCountY = 0;
            this.numberOfGridTiles = 0;
        }
    }

    public void copyAllData(TiffMap source, boolean cloneData) {
        Objects.requireNonNull(source, "Null source TIFF map");
        if (source.numberOfSeparatedPlanes != this.numberOfSeparatedPlanes) {
            throw new IllegalArgumentException("Number of separated planes in the source (" + source.numberOfSeparatedPlanes + ") and this map (" + this.numberOfSeparatedPlanes + ") do not match");
        }
        if (source.gridCountX != this.gridCountX || source.gridCountY != this.gridCountY) {
            throw new IllegalArgumentException("Grid sizes in the source map (" + source.gridCountX + "x" + source.gridCountY + " tiles) and this map (" + this.gridCountX + "x" + source.gridCountY + " tiles) do not match");
        }
        for (TiffTile tile : source.tiles()) {
            this.getOrNew(this.copyIndex(tile.index())).copyData(tile, cloneData);
        }
    }

    public boolean isByteOrderCompatible(ByteOrder byteOrder) {
        Objects.requireNonNull(byteOrder, "Null byte order");
        return byteOrder == this.byteOrder || this.sampleType.isBinary() || this.sampleType.bitsPerSample() == 8;
    }

    public byte[] toInterleavedSamples(byte[] samples, int numberOfChannels, long numberOfPixels) {
        return this.toInterleaveOrSeparatedSamples(samples, numberOfChannels, numberOfPixels, true);
    }

    public byte[] toSeparatedSamples(byte[] samples, int numberOfChannels, long numberOfPixels) {
        return this.toInterleaveOrSeparatedSamples(samples, numberOfChannels, numberOfPixels, false);
    }

    public String toString() {
        return (this.resizable ? "resizable " : "") + this.mapKindName() + " " + this.dimX + "x" + this.dimY + "x" + this.numberOfChannels + " (" + this.alignedBitsPerSample + " bits) of " + this.tileMap.size() + " TIFF tiles (grid " + this.gridCountX + "x" + this.gridCountY + (String)(this.numberOfSeparatedPlanes == 1 ? "" : "x" + this.numberOfSeparatedPlanes) + ") at the image " + String.valueOf(this.ifd);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TiffMap that = (TiffMap)o;
        return this.ifd == that.ifd && this.resizable == that.resizable && this.dimX == that.dimX && this.dimY == that.dimY && Objects.equals(this.tileMap, that.tileMap) && this.planarSeparated == that.planarSeparated && this.numberOfChannels == that.numberOfChannels && this.alignedBitsPerSample == that.alignedBitsPerSample && this.tileSizeX == that.tileSizeX && this.tileSizeY == that.tileSizeY && this.tileSizeInBytes == that.tileSizeInBytes;
    }

    public int hashCode() {
        return Objects.hash(System.identityHashCode(this.ifd), this.tileMap, this.resizable, this.dimX, this.dimY);
    }

    public static long checkRequestedArea(long fromX, long fromY, long sizeX, long sizeY) {
        long result = TiffIFD.multiplySizes(sizeX, sizeY);
        assert (sizeX >= 0L && sizeY >= 0L) : "multiplySizes did not check sizes";
        assert (sizeX <= Integer.MAX_VALUE && sizeY <= Integer.MAX_VALUE) : "multiplySizes did not check sizes";
        if (fromX != (long)((int)fromX) || fromY != (long)((int)fromY)) {
            throw new IllegalArgumentException("Too large absolute values of fromX = " + fromX + " or fromY = " + fromY + " (out of -2^31..2^31-1 ranges)");
        }
        if (sizeX >= Integer.MAX_VALUE - fromX || sizeY >= Integer.MAX_VALUE - fromY) {
            throw new IllegalArgumentException("Requested area [" + fromX + ".." + (fromX + sizeX - 1L) + " x " + fromY + ".." + (fromY + sizeY - 1L) + "] is outside the 0..2^31-2 ranges");
        }
        return result;
    }

    public static byte[] toInterleavedBytes(byte[] bytes, int numberOfChannels, int bytesPerSample, long numberOfPixels) {
        Objects.requireNonNull(bytes, "Null bytes");
        int size = TiffMap.checkSizes(numberOfChannels, bytesPerSample, numberOfPixels);
        if (bytes.length < size) {
            throw new IllegalArgumentException("Too short samples array: " + bytes.length + " < " + size);
        }
        if (numberOfChannels == 1) {
            return bytes;
        }
        int bandSize = bytesPerSample * (int)numberOfPixels;
        byte[] interleavedBytes = new byte[size];
        if (bytesPerSample == 1) {
            Matrix mI = Matrix.as((Object)interleavedBytes, (long[])new long[]{numberOfChannels, numberOfPixels});
            Matrix mS = Matrix.as((Object)bytes, (long[])new long[]{numberOfPixels, numberOfChannels});
            Matrices.interleave(null, (Matrix)mI, (List)mS.asLayers());
        } else {
            int disp = 0;
            for (int i = 0; i < bandSize; i += bytesPerSample) {
                int bandDisp = i;
                int j = 0;
                while (j < numberOfChannels) {
                    for (int k = 0; k < bytesPerSample; ++k) {
                        interleavedBytes[disp++] = bytes[bandDisp + k];
                    }
                    ++j;
                    bandDisp += bandSize;
                }
            }
        }
        return interleavedBytes;
    }

    public static byte[] toSeparatedBytes(byte[] bytes, int numberOfChannels, int bytesPerSample, long numberOfPixels) {
        Objects.requireNonNull(bytes, "Null bytes");
        int size = TiffMap.checkSizes(numberOfChannels, bytesPerSample, numberOfPixels);
        if (bytes.length < size) {
            throw new IllegalArgumentException("Too short samples array: " + bytes.length + " < " + size);
        }
        if (numberOfChannels == 1) {
            return bytes;
        }
        int bandSize = bytesPerSample * (int)numberOfPixels;
        byte[] separatedBytes = new byte[size];
        if (bytesPerSample == 1) {
            Matrix mI = Matrix.as((Object)bytes, (long[])new long[]{numberOfChannels, numberOfPixels});
            Matrix mS = Matrix.as((Object)separatedBytes, (long[])new long[]{numberOfPixels, numberOfChannels});
            Matrices.separate(null, (List)mS.asLayers(), (Matrix)mI);
        } else {
            int disp = 0;
            for (int i = 0; i < bandSize; i += bytesPerSample) {
                int bandDisp = i;
                int j = 0;
                while (j < numberOfChannels) {
                    for (int k = 0; k < bytesPerSample; ++k) {
                        separatedBytes[bandDisp + k] = bytes[disp++];
                    }
                    ++j;
                    bandDisp += bandSize;
                }
            }
        }
        return separatedBytes;
    }

    byte[] toInterleaveOrSeparatedSamples(byte[] samples, int numberOfChannels, long numberOfPixels, boolean interleave) {
        Objects.requireNonNull(samples, "Null samples");
        if (numberOfPixels < 0L) {
            throw new IllegalArgumentException("Negative numberOfPixels = " + numberOfPixels);
        }
        if (numberOfChannels == 1) {
            return samples;
        }
        if (!this.isWholeBytes()) {
            throw new AssertionError((Object)"Non-whole bytes are impossible in valid TiffMap with 1 channel");
        }
        int bytesPerSample = this.alignedBitsPerSample >>> 3;
        assert (this.alignedBitsPerSample == bytesPerSample * 8) : "unaligned bitsPerSample impossible for whole bytes";
        return interleave ? TiffMap.toInterleavedBytes(samples, numberOfChannels, bytesPerSample, numberOfPixels) : TiffMap.toSeparatedBytes(samples, numberOfChannels, bytesPerSample, numberOfPixels);
    }

    String mapKindName() {
        return "map";
    }

    private static int checkSizes(int numberOfChannels, int bytesPerSample, long numberOfPixels) {
        long size;
        TiffIFD.checkNumberOfChannels(numberOfChannels);
        TiffIFD.checkBitsPerSample(8L * (long)bytesPerSample);
        if (numberOfPixels < 0L) {
            throw new IllegalArgumentException("Negative numberOfPixels = " + numberOfPixels);
        }
        if (numberOfPixels > Integer.MAX_VALUE || (size = numberOfPixels * (long)numberOfChannels * (long)bytesPerSample) > Integer.MAX_VALUE) {
            throw new TooLargeArrayException("Too large number of pixels " + numberOfPixels + " (" + numberOfChannels + " samples/pixel, " + bytesPerSample + " bytes/sample): it requires > 2 GB to store");
        }
        return (int)size;
    }

    private void setDimensions(int dimX, int dimY, boolean checkResizable) {
        if (checkResizable && !this.resizable) {
            throw new IllegalArgumentException("Cannot change dimensions of a non-resizable tile map");
        }
        if (dimX < 0) {
            throw new IllegalArgumentException("Negative x-dimension: " + dimX);
        }
        if (dimY < 0) {
            throw new IllegalArgumentException("Negative y-dimension: " + dimY);
        }
        if ((long)dimX * (long)dimY > Long.MAX_VALUE / (long)this.totalAlignedBitsPerPixel) {
            throw new TooLargeArrayException("Extremely large image sizes " + dimX + "x" + dimY + ", " + this.totalAlignedBitsPerPixel + " bits/pixel: total number of bits is greater than 2^63-1 (!)");
        }
        int gridCountX = (int)((long)dimX + (long)this.tileSizeX - 1L) / this.tileSizeX;
        int gridCountY = (int)((long)dimY + (long)this.tileSizeY - 1L) / this.tileSizeY;
        this.expandGrid(gridCountX, gridCountY, checkResizable);
        this.dimX = dimX;
        this.dimY = dimY;
    }

    private void expandGrid(int newMinimalGridCountX, int newMinimalGridCountY, boolean checkResizable) {
        int gridCountY;
        if (newMinimalGridCountX < 0) {
            throw new IllegalArgumentException("Negative new minimal tiles x-count: " + newMinimalGridCountX);
        }
        if (newMinimalGridCountY < 0) {
            throw new IllegalArgumentException("Negative new minimal tiles y-count: " + newMinimalGridCountY);
        }
        if (newMinimalGridCountX <= this.gridCountX && newMinimalGridCountY <= this.gridCountY) {
            return;
        }
        if (checkResizable && !this.resizable) {
            throw new IllegalArgumentException("Cannot expand tile counts in a non-resizable tile map");
        }
        int gridCountX = Math.max(this.gridCountX, newMinimalGridCountX);
        if ((long)gridCountX * (long)(gridCountY = Math.max(this.gridCountY, newMinimalGridCountY)) > (long)(Integer.MAX_VALUE / this.numberOfSeparatedPlanes)) {
            throw new IllegalArgumentException("Too large number of tiles/strips: " + (String)(this.numberOfSeparatedPlanes > 1 ? this.numberOfSeparatedPlanes + " separated planes * " : "") + gridCountX + " * " + gridCountY + " > 2^31-1");
        }
        this.gridCountX = gridCountX;
        this.gridCountY = gridCountY;
        this.numberOfGridTiles = gridCountX * gridCountY * this.numberOfSeparatedPlanes;
    }

    public static enum TilingMode {
        TILE_GRID,
        STRIPS;


        public final boolean isTileGrid() {
            return this == TILE_GRID;
        }
    }
}

