import numpy as np
from PIL import Image, ImageDraw

# Made by ChatGPT, fixed: sort faces by depth for correct rendering
def make_octahedron(angle_x=0, angle_y=0, angle_z=0, img_size=400):
    angle_x = float(angle_x)
    angle_y = float(angle_y)
    angle_z = float(angle_z)
    # Convert angles to radians
    ax, ay, az = np.deg2rad([angle_x, angle_y, angle_z])

    # Vertices of octahedron
    verts = np.array([
        [ 1,  0,  0],
        [-1,  0,  0],
        [ 0,  1,  0],
        [ 0, -1,  0],
        [ 0,  0,  1],
        [ 0,  0, -1],
    ], dtype=float)

    # Rotation matrices
    Rx = np.array([
        [1, 0, 0],
        [0, np.cos(ax), -np.sin(ax)],
        [0, np.sin(ax),  np.cos(ax)]
    ])

    Ry = np.array([
        [ np.cos(ay), 0, np.sin(ay)],
        [0, 1, 0],
        [-np.sin(ay), 0, np.cos(ay)]
    ])

    Rz = np.array([
        [np.cos(az), -np.sin(az), 0],
        [np.sin(az),  np.cos(az), 0],
        [0, 0, 1]
    ])

    # Combined rotation
    R = Rz @ Ry @ Rx
    verts_rot = verts @ R.T

    # Faces (triangles), indices into verts array
    faces = [
        (0, 2, 4),
        (2, 1, 4),
        (1, 3, 4),
        (3, 0, 4),
        (0, 5, 2),
        (2, 5, 1),
        (1, 5, 3),
        (3, 5, 0),
    ]

    # Orthogonal projection: just drop z
    verts_2d = verts_rot[:, :2]

    # Scale and translate to image coords
    margin = 40
    scale = (img_size - 2 * margin) / (verts_2d.max() - verts_2d.min())
    verts_img = (verts_2d - verts_2d.min()) * scale + margin
    verts_img = verts_img.astype(int)

    # Compute average z for each face (for painter's algorithm)
    face_depths = [verts_rot[list(face), 2].mean() for face in faces]

    # Sort faces by depth (farther faces first)
    faces_sorted = [f for _, f in sorted(zip(face_depths, faces), key=lambda x: x[0], reverse=True)]

    # Create blank image and drawing context
    img = Image.new("RGB", (img_size, img_size), (255, 255, 255))
    draw = ImageDraw.Draw(img)

    # Colors for faces
    face_colors = [
        (255, 128, 128),
        (255, 200, 200),
        (255, 100, 100),
        (200, 50, 50),
        (128, 128, 255),
        (200, 200, 255),
        (100, 100, 255),
        (50, 50, 200),
    ]

    # Draw faces in sorted order
    for face, color in zip(faces_sorted, face_colors):
        polygon = [tuple(verts_img[i]) for i in face]
        draw.polygon(polygon, fill=color, outline=(0, 0, 0))

    # Return as numpy array (RGB)
    return np.array(img)

def execute(params, inputs, outputs):
    outputs.m1 = make_octahedron(params.a, params.b, params.c)
    return "Hello! figure is in m1"


# Example usage:
if __name__ == "__main__":
    img = make_octahedron(30, 45, 60)
    # img.show()
    # img.save("octahedron.png")
