/*
 * Decompiled with CFR 0.152.
 */
package net.algart.matrices.tiff;

import java.io.IOException;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import net.algart.arrays.JArrays;
import net.algart.matrices.tiff.TemporaryFileCreator;
import net.algart.matrices.tiff.TiffCreateMode;
import net.algart.matrices.tiff.TiffException;
import net.algart.matrices.tiff.TiffIFD;
import net.algart.matrices.tiff.TiffIFDCorrector;
import net.algart.matrices.tiff.TiffIO;
import net.algart.matrices.tiff.TiffOpenMode;
import net.algart.matrices.tiff.TiffReader;
import net.algart.matrices.tiff.TiffWriter;
import net.algart.matrices.tiff.tags.TagCompression;
import net.algart.matrices.tiff.tiles.TiffMap;
import net.algart.matrices.tiff.tiles.TiffReadMap;
import net.algart.matrices.tiff.tiles.TiffTile;
import net.algart.matrices.tiff.tiles.TiffTileIndex;
import net.algart.matrices.tiff.tiles.TiffWriteMap;
import org.scijava.io.handle.BytesHandle;
import org.scijava.io.handle.DataHandle;
import org.scijava.io.location.BytesLocation;

public final class TiffCopier {
    private static final System.Logger LOG = System.getLogger(TiffCopier.class.getName());
    private static final boolean LOGGABLE_DEBUG = LOG.isLoggable(System.Logger.Level.DEBUG);
    private boolean directCopy = true;
    private int maxInMemoryTemporaryFileSize = 0;
    private TemporaryFileCreator temporaryFileCreator = TemporaryFileCreator.DEFAULT;
    private TiffIFDCorrector ifdCorrector = null;
    private Consumer<ProgressInformation> progressUpdater = null;
    private BooleanSupplier interruptionChecker = null;
    private final ProgressInformation progressInformation = new ProgressInformation();
    private boolean actuallyDirectCopy = false;

    public boolean isDirectCopy() {
        return this.directCopy;
    }

    public TiffCopier setDirectCopy(boolean directCopy) {
        this.directCopy = directCopy;
        return this;
    }

    public int getMaxInMemoryTemporaryFileSize() {
        return this.maxInMemoryTemporaryFileSize;
    }

    public TiffCopier setMaxInMemoryTemporaryFileSize(int maxInMemoryTemporaryFileSize) {
        if (maxInMemoryTemporaryFileSize < 0) {
            throw new IllegalArgumentException("Negative maxInMemoryTemporaryFileSize: " + maxInMemoryTemporaryFileSize);
        }
        this.maxInMemoryTemporaryFileSize = maxInMemoryTemporaryFileSize;
        return this;
    }

    public TemporaryFileCreator getTemporaryFileCreator() {
        return this.temporaryFileCreator;
    }

    public TiffCopier setTemporaryFileCreator(TemporaryFileCreator temporaryFileCreator) {
        this.temporaryFileCreator = Objects.requireNonNull(temporaryFileCreator, "Null temporary file creator");
        return this;
    }

    public TiffIFDCorrector getIfdCorrector() {
        return this.ifdCorrector;
    }

    public TiffCopier setIfdCorrector(TiffIFDCorrector tiffIfdCorrector) {
        this.ifdCorrector = tiffIfdCorrector;
        return this;
    }

    public Consumer<ProgressInformation> progressUpdater() {
        return this.progressUpdater;
    }

    public TiffCopier setProgressUpdater(Consumer<ProgressInformation> progressUpdater) {
        this.progressUpdater = progressUpdater;
        return this;
    }

    public BooleanSupplier cancellationChecker() {
        return this.interruptionChecker;
    }

    public TiffCopier setInterruptionChecker(BooleanSupplier interruptionChecker) {
        this.interruptionChecker = interruptionChecker;
        return this;
    }

    public ProgressInformation progressInformation() {
        return this.progressInformation;
    }

    public boolean actuallyDirectCopy() {
        return this.actuallyDirectCopy;
    }

    public static void copyFile(Path targetTiffFile, Path sourceTiffFile) throws IOException {
        new TiffCopier().copyAllTiff(targetTiffFile, sourceTiffFile);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void compact(Path tiffFile) throws IOException {
        boolean inMemory;
        Objects.requireNonNull(tiffFile, "Null TIFF file");
        boolean bl = inMemory = Files.size(tiffFile) <= (long)this.maxInMemoryTemporaryFileSize;
        if (inMemory) {
            BytesHandle tempBytes = new BytesHandle(new BytesLocation(0));
            try (TiffReader reader = new TiffReader(tiffFile, TiffOpenMode.VALID_TIFF);
                 TiffWriter writer = new TiffWriter((DataHandle<?>)tempBytes);){
                this.copyAllTiff(writer, reader);
            }
            this.copyTempFileBackToTiff(tiffFile, tempBytes);
            return;
        }
        Path tempFile = null;
        try {
            try (TiffReader reader = new TiffReader(tiffFile);){
                tempFile = this.temporaryFileCreator.createTemporaryFile();
                try (TiffWriter writer = new TiffWriter(tempFile, TiffCreateMode.NO_ACTIONS);){
                    this.copyAllTiff(writer, reader);
                }
            }
            this.copyTempFileBackToTiff(tiffFile, tempFile);
            if (tempFile == null) return;
        }
        catch (Throwable throwable) {
            if (tempFile == null) throw throwable;
            Files.delete(tempFile);
            throw throwable;
        }
        Files.delete(tempFile);
    }

    public void copyAllTiff(Path targetTiffFile, Path sourceTiffFile) throws IOException {
        Objects.requireNonNull(targetTiffFile, "Null target TIFF file");
        Objects.requireNonNull(sourceTiffFile, "Null source TIFF file");
        try (TiffReader reader = new TiffReader(sourceTiffFile);
             TiffWriter writer = new TiffWriter(targetTiffFile, TiffCreateMode.NO_ACTIONS);){
            this.copyAllTiff(writer, reader);
        }
    }

    public void copyAllTiff(TiffWriter writer, TiffReader reader) throws IOException {
        Objects.requireNonNull(writer, "Null TIFF writer");
        Objects.requireNonNull(reader, "Null TIFF reader");
        writer.setFormatLike(reader);
        writer.create();
        this.copyImages(writer, reader);
    }

    public void copyImages(TiffWriter writer, TiffReader reader) throws IOException {
        this.copyImages(writer, reader, 0, reader.numberOfImages());
    }

    public void copyImages(TiffWriter writer, TiffReader reader, int fromIndex, int toIndex) throws IOException {
        Objects.requireNonNull(writer, "Null TIFF writer");
        Objects.requireNonNull(reader, "Null TIFF reader");
        if (fromIndex < 0) {
            throw new IllegalArgumentException("Negative fromIndex: " + fromIndex);
        }
        if (toIndex < fromIndex) {
            throw new IllegalArgumentException("toIndex < fromIndex: " + toIndex + " < " + fromIndex);
        }
        int n = reader.numberOfImages();
        if (toIndex > n) {
            throw new IndexOutOfBoundsException("toIndex > numberOfImages: " + toIndex + " > " + n);
        }
        this.resetAllCounters();
        for (int i = fromIndex; i < toIndex; ++i) {
            this.progressInformation.imageIndex = i - fromIndex;
            this.progressInformation.imageCount = toIndex - fromIndex;
            this.copyImage(writer, reader, i);
        }
    }

    public TiffWriteMap copyImage(TiffWriter writer, TiffReader reader, int sourceIfdIndex) throws IOException {
        Objects.requireNonNull(writer, "Null TIFF writer");
        Objects.requireNonNull(reader, "Null TIFF reader");
        TiffReadMap readMap = reader.map(sourceIfdIndex);
        return this.copyImage(writer, readMap);
    }

    public TiffWriteMap copyImage(TiffWriter writer, TiffReadMap readMap) throws IOException {
        boolean actuallyDirectCopy;
        Objects.requireNonNull(writer, "Null TIFF writer");
        Objects.requireNonNull(readMap, "Null TIFF read map");
        long t1 = TiffIO.debugTime();
        this.resetImageCounters();
        TiffIFD writeIFD = new TiffIFD(readMap.ifd());
        if (this.ifdCorrector != null) {
            this.ifdCorrector.correct(writeIFD);
        }
        boolean correctForEncoding = !(actuallyDirectCopy = this.canBeImageCopiedDirectly(writeIFD, writer.getByteOrder(), readMap));
        TiffWriteMap writeMap = writer.newMap(writeIFD, false, correctForEncoding);
        TiffCopier.checkImageCompatibility(writeMap, readMap);
        this.actuallyDirectCopy = actuallyDirectCopy;
        writer.writeForward(writeMap);
        Collection<TiffTile> targetTiles = writeMap.tiles();
        this.progressInformation.tileCount = targetTiles.size();
        int tileCount = 0;
        long t2 = TiffIO.debugTime();
        long timeReading = 0L;
        long timeCopying = 0L;
        long timeWriting = 0L;
        for (TiffTile targetTile : targetTiles) {
            long t2Tile;
            TiffTileIndex readIndex = readMap.copyIndex(targetTile.index());
            long t1Tile = TiffIO.debugTime();
            if (this.actuallyDirectCopy) {
                sourceTile = readMap.readEncodedTile(readIndex);
                t2Tile = TiffIO.debugTime();
                targetTile.copyData(sourceTile, false);
            } else {
                sourceTile = readMap.readCachedTile(readIndex);
                t2Tile = TiffIO.debugTime();
                targetTile.copyUnpackedSamples(sourceTile, readMap.isAutoScaleWhenIncreasingBitDepth());
            }
            writeMap.put(targetTile);
            long t3Tile = TiffIO.debugTime();
            writeMap.writeTile(targetTile, true);
            long t4Tile = TiffIO.debugTime();
            timeReading += t2Tile - t1Tile;
            timeCopying += t3Tile - t2Tile;
            timeWriting += t4Tile - t3Tile;
            this.progressInformation.tileIndex = tileCount++;
            if (this.shouldBreak()) break;
        }
        long t3 = TiffIO.debugTime();
        writeMap.completeWriting();
        if (TiffIO.BUILT_IN_TIMING && LOGGABLE_DEBUG) {
            long sizeInBytes = writeMap.totalSizeInBytes();
            long t4 = TiffIO.debugTime();
            LOG.log(System.Logger.Level.DEBUG, String.format(Locale.US, "%s copied entire image %s %dx%dx%d (%d tiles, %.3f MB) in %.3f ms = %.3f prepare + %.3f copy (%.3f read + %.3f copy data + %.3f write) + %.3f complete, %.3f MB/s", this.getClass().getSimpleName(), actuallyDirectCopy ? "directly" : "with repacking" + (this.directCopy ? " (direct mode rejected)" : ""), writeMap.dimX(), writeMap.dimY(), writeMap.numberOfChannels(), tileCount, (double)sizeInBytes / 1048576.0, (double)(t4 - t1) * 1.0E-6, (double)(t2 - t1) * 1.0E-6, (double)(t3 - t2) * 1.0E-6, (double)timeReading * 1.0E-6, (double)timeCopying * 1.0E-6, (double)timeWriting * 1.0E-6, (double)(t4 - t3) * 1.0E-6, (double)sizeInBytes / 1048576.0 / ((double)(t4 - t1) * 1.0E-9)));
        }
        return writeMap;
    }

    public TiffWriteMap copyImage(TiffWriter writer, TiffReadMap readMap, int fromX, int fromY, int sizeX, int sizeY) throws IOException {
        boolean swapOrder;
        Objects.requireNonNull(writer, "Null TIFF writer");
        Objects.requireNonNull(readMap, "Null TIFF read map");
        TiffMap.checkRequestedArea(fromX, fromY, sizeX, sizeY);
        long t1 = TiffIO.debugTime();
        this.resetImageCounters();
        TiffIFD writeIFD = new TiffIFD(readMap.ifd());
        writeIFD.putImageDimensions(sizeX, sizeY, false);
        if (this.ifdCorrector != null) {
            this.ifdCorrector.correct(writeIFD);
        }
        TiffWriteMap writeMap = writer.newMap(writeIFD, false, true);
        TiffCopier.checkImageCompatibility(writeMap, readMap);
        writer.writeForward(writeMap);
        this.progressInformation.tileCount = writeMap.numberOfGridTiles();
        int readDimX = readMap.dimX();
        int readDimY = readMap.dimY();
        int gridCountY = writeMap.gridCountY();
        int gridCountX = writeMap.gridCountX();
        int mapTileSizeX = writeMap.tileSizeX();
        int mapTileSizeY = writeMap.tileSizeY();
        boolean directCopy = this.canBeRectangleCopiedDirectly(writeMap, readMap, fromX, fromY);
        boolean bl = swapOrder = !readMap.isByteOrderCompatible(writeMap.byteOrder());
        if (swapOrder && directCopy) {
            throw new AssertionError((Object)"If we use swapOrder branch, we must be sure that no tiles will be coped directly");
        }
        if (swapOrder) {
            assert (readMap.byteOrder() != writeMap.byteOrder());
            assert (readMap.elementType() != Boolean.TYPE);
        }
        int fromXIndex = directCopy ? fromX / mapTileSizeX : Integer.MIN_VALUE;
        int fromYIndex = directCopy ? fromY / mapTileSizeY : Integer.MIN_VALUE;
        long t2 = TiffIO.debugTime();
        int repackCount = 0;
        int tileCount = 0;
        int yIndex = 0;
        int y = 0;
        while (yIndex < gridCountY) {
            int sizeYInTile = Math.min(sizeY - y, mapTileSizeY);
            int readY = fromY + y;
            int xIndex = 0;
            int x = 0;
            while (xIndex < gridCountX) {
                int sizeXInTile = Math.min(sizeX - x, mapTileSizeX);
                int readX = fromX + x;
                boolean bl2 = this.actuallyDirectCopy = directCopy && sizeXInTile == mapTileSizeX && sizeYInTile == mapTileSizeY && readY <= readDimY - sizeYInTile && readX <= readDimX - sizeXInTile;
                if (this.actuallyDirectCopy) {
                    int readXIndex = fromXIndex + xIndex;
                    int readYIndex = fromYIndex + yIndex;
                    TiffCopier.copyEncodedTile(writeMap, readMap, xIndex, yIndex, readXIndex, readYIndex);
                } else {
                    int written = TiffCopier.copyRectangle(writeMap, readMap, x, y, readX, readY, sizeXInTile, sizeYInTile, swapOrder);
                    if (written != writeMap.numberOfSeparatedPlanes()) {
                        throw new AssertionError((Object)("Number of written tiles " + written + " != " + writeMap.numberOfSeparatedPlanes()));
                    }
                    ++repackCount;
                }
                this.progressInformation.tileIndex = tileCount++;
                if (this.shouldBreak()) break;
                ++xIndex;
                x += mapTileSizeX;
            }
            ++yIndex;
            y += mapTileSizeY;
        }
        long t3 = TiffIO.debugTime();
        int written = writeMap.completeWriting();
        if (written != 0) {
            throw new AssertionError((Object)("Number of tiles when completion " + written + " != 0"));
        }
        if (TiffIO.BUILT_IN_TIMING && LOGGABLE_DEBUG) {
            long sizeInBytes = writeMap.totalSizeInBytes();
            long t4 = TiffIO.debugTime();
            LOG.log(System.Logger.Level.DEBUG, String.format(Locale.US, "%s copied %s%s %dx%dx%d samples in %d tiles%s (%.3f MB) in %.3f ms = %.3f prepare + %.3f copy + %.3f complete, %.3f MB/s", this.getClass().getSimpleName(), directCopy ? "directly" : "with repacking" + (this.directCopy ? " (direct mode rejected)" : ""), swapOrder ? " and reordering bytes" : "", sizeX, sizeY, writeMap.numberOfChannels(), tileCount, repackCount < tileCount ? " (" + repackCount + " with repacking)" : "", (double)sizeInBytes / 1048576.0, (double)(t4 - t1) * 1.0E-6, (double)(t2 - t1) * 1.0E-6, (double)(t3 - t2) * 1.0E-6, (double)(t4 - t3) * 1.0E-6, (double)sizeInBytes / 1048576.0 / ((double)(t4 - t1) * 1.0E-9)));
        }
        return writeMap;
    }

    private static void checkImageCompatibility(TiffWriteMap writeMap, TiffReadMap readMap) {
        if (writeMap.sampleType() != readMap.sampleType()) {
            throw new IllegalArgumentException("Incompatible sample types: " + String.valueOf((Object)writeMap.sampleType()) + " in target TIFF, " + String.valueOf((Object)readMap.sampleType()) + " in source TIFF");
        }
        if (writeMap.numberOfChannels() != readMap.numberOfChannels()) {
            throw new IllegalArgumentException("Incompatible number of channels: " + writeMap.numberOfChannels() + " in target TIFF, " + readMap.numberOfChannels() + " in source TIFF");
        }
        if (writeMap.isPlanarSeparated() != readMap.isPlanarSeparated()) {
            throw new IllegalArgumentException("Incompatible \"planar separated\" mode: " + writeMap.isPlanarSeparated() + " in target TIFF, " + readMap.isPlanarSeparated() + " in source TIFF");
        }
    }

    private boolean canBeImageCopiedDirectly(TiffIFD writeIFD, ByteOrder writeByteOrder, TiffReadMap readMap) throws TiffException {
        if (!readMap.isByteOrderCompatible(writeByteOrder)) {
            return false;
        }
        int compressionCode = readMap.compressionCode();
        TagCompression compression = readMap.compression().orElse(null);
        boolean byteOrderIsNotUsedForImageData = TiffCopier.isByteOrBinary(readMap) && compression != null && !compression.canUseByteOrderForByteData();
        boolean bitDepthEqual = Arrays.equals(readMap.bitsPerSample(), writeIFD.getBitsPerSample());
        boolean photometricInterpretationEqual = readMap.photometricInterpretationCode() == writeIFD.getPhotometricInterpretationCode();
        return this.directCopy && (readMap.byteOrder() == writeByteOrder || byteOrderIsNotUsedForImageData) && compressionCode == writeIFD.getCompressionCode() && bitDepthEqual && photometricInterpretationEqual;
    }

    private static boolean isByteOrBinary(TiffReadMap readMap) {
        return readMap.isBinary() || readMap.sampleType().bitsPerSample() == 8;
    }

    private boolean canBeRectangleCopiedDirectly(TiffWriteMap writeMap, TiffReadMap readMap, int fromX, int fromY) throws TiffException {
        int mapTileSizeX = writeMap.tileSizeX();
        int mapTileSizeY = writeMap.tileSizeY();
        boolean equalTiles = readMap.tileSizeX() == mapTileSizeX && readMap.tileSizeY() == mapTileSizeY;
        return equalTiles && fromX % mapTileSizeX == 0 && fromY % mapTileSizeY == 0 && this.canBeImageCopiedDirectly(writeMap.ifd(), writeMap.byteOrder(), readMap);
    }

    private static int copyRectangle(TiffWriteMap writeMap, TiffReadMap readMap, int writeX, int writeY, int readX, int readY, int sizeX, int sizeY, boolean swapOrder) throws IOException {
        byte[] samples = readMap.loadSampleBytes(readX, readY, sizeX, sizeY, TiffReader.UnusualPrecisions.UNPACK);
        if (swapOrder) {
            samples = JArrays.copyAndSwapByteOrder((byte[])samples, readMap.elementType());
        }
        List<TiffTile> tiles = writeMap.updateSampleBytes(samples, writeX, writeY, sizeX, sizeY);
        return writeMap.writeCompletedTiles(tiles);
    }

    private static void copyEncodedTile(TiffWriteMap writeMap, TiffReadMap readMap, int writeXIndex, int writeYIndex, int readXIndex, int readYIndex) throws IOException {
        int numberOfSeparatedPlanes = writeMap.numberOfSeparatedPlanes();
        for (int p = 0; p < numberOfSeparatedPlanes; ++p) {
            TiffTile targetTile = writeMap.getOrNew(writeXIndex, writeYIndex, p);
            TiffTile sourceTile = readMap.readEncodedTile(readMap.index(readXIndex, readYIndex, p));
            targetTile.copyData(sourceTile, false);
            writeMap.put(targetTile);
            writeMap.writeTile(targetTile, true);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void copyTempFileBackToTiff(Path tiffFile, Object tempBuffer) throws IOException {
        this.progressInformation.copyingTemporaryFile = true;
        try {
            long sizeInBytes;
            long t1 = TiffIO.debugTime();
            this.shouldBreak();
            if (tempBuffer instanceof BytesHandle) {
                BytesHandle tempBytes = (BytesHandle)tempBuffer;
                try (DataHandle<?> tiffStream = TiffIO.getFileHandle(tiffFile);){
                    sizeInBytes = TiffIO.copyFile(tempBytes, tiffStream);
                }
            } else if (tempBuffer instanceof Path) {
                Path tempFile = (Path)tempBuffer;
                sizeInBytes = Files.size(tempFile);
                Files.copy(tempFile, tiffFile, StandardCopyOption.REPLACE_EXISTING);
            } else {
                throw new AssertionError((Object)("Unknown tempBuffer " + String.valueOf(tempBuffer)));
            }
            if (TiffIO.BUILT_IN_TIMING && LOGGABLE_DEBUG) {
                long t2 = TiffIO.debugTime();
                LOG.log(System.Logger.Level.DEBUG, String.format(Locale.US, "%s copied %d bytes%s (%.3f MB) in %.3f ms, %.3f MB/s", this.getClass().getSimpleName(), sizeInBytes, tempBuffer instanceof BytesHandle ? " from memory" : " from temporary file", (double)sizeInBytes / 1048576.0, (double)(t2 - t1) * 1.0E-6, (double)sizeInBytes / 1048576.0 / ((double)(t2 - t1) * 1.0E-9)));
            }
        }
        finally {
            this.progressInformation.copyingTemporaryFile = false;
        }
    }

    private void resetAllCounters() {
        this.progressInformation.imageIndex = 0;
        this.progressInformation.imageCount = 0;
        this.resetImageCounters();
    }

    private void resetImageCounters() {
        this.progressInformation.tileIndex = 0;
        this.progressInformation.tileCount = 0;
        this.progressInformation.copyingTemporaryFile = false;
    }

    private boolean shouldBreak() {
        if (this.progressUpdater != null) {
            this.progressUpdater.accept(this.progressInformation);
        }
        return this.interruptionChecker != null && this.interruptionChecker.getAsBoolean();
    }

    public final class ProgressInformation {
        private int imageIndex = 0;
        private int imageCount = 0;
        private int tileIndex = 0;
        private int tileCount = 0;
        private boolean copyingTemporaryFile = false;

        public TiffCopier copier() {
            return TiffCopier.this;
        }

        public int imageIndex() {
            return this.imageIndex;
        }

        public int imageCount() {
            return this.imageCount;
        }

        public boolean isLastImageCopied() {
            return this.imageIndex == this.imageCount - 1;
        }

        public int tileIndex() {
            return this.tileIndex;
        }

        public int tileCount() {
            return this.tileCount;
        }

        public boolean isLastTileCopied() {
            return this.tileIndex == this.tileCount - 1;
        }

        public boolean isCopyingTemporaryFile() {
            return this.copyingTemporaryFile;
        }
    }
}

