/*
 * Decompiled with CFR 0.152.
 */
package net.algart.matrices.tiff.data;

import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Objects;
import java.util.OptionalInt;
import java.util.concurrent.atomic.AtomicBoolean;
import net.algart.arrays.JArrays;
import net.algart.arrays.PackedBitArraysPer8;
import net.algart.matrices.tiff.TiffException;
import net.algart.matrices.tiff.TiffIFD;
import net.algart.matrices.tiff.UnsupportedTiffFormatException;
import net.algart.matrices.tiff.tags.TagCompression;
import net.algart.matrices.tiff.tags.TagPhotometricInterpretation;
import net.algart.matrices.tiff.tags.TagRational;
import net.algart.matrices.tiff.tiles.TiffMap;
import net.algart.matrices.tiff.tiles.TiffTile;

public class TiffUnpacking {
    private static final boolean OPTIMIZE_SEPARATING_WHOLE_BYTES = true;
    private static final boolean THOROUGHLY_TEST_Y_CB_CR_LOOP = false;

    private TiffUnpacking() {
    }

    public static boolean separateUnpackedSamples(TiffTile tile) throws TiffException {
        Objects.requireNonNull(tile);
        int decodedDataLength = tile.getDecodedDataLength();
        TiffIFD ifd = tile.ifd();
        AtomicBoolean simpleNonJpegFormat = new AtomicBoolean();
        if (!TiffUnpacking.isSimpleRearrangingBytesEnough(ifd, simpleNonJpegFormat)) {
            return false;
        }
        if (decodedDataLength > tile.map().tileSizeInBytes() && !simpleNonJpegFormat.get()) {
            throw new TiffException("Too large decoded TIFF data: " + decodedDataLength + " bytes, its is greater than one " + (tile.map().tilingMode().isTileGrid() ? "tile" : "strip") + " (" + tile.map().tileSizeInBytes() + " bytes); probably TIFF file is corrupted or format is not properly supported");
        }
        tile.adjustNumberOfPixels(true);
        tile.separateSamplesIfNecessary();
        return true;
    }

    public static boolean separateYCbCrToRGB(TiffTile tile) throws TiffException {
        Objects.requireNonNull(tile);
        TiffIFD ifd = tile.ifd();
        byte[] data = tile.getDecodedData();
        if (!ifd.isStandardYCbCrNonJpeg()) {
            return false;
        }
        TiffUnpacking.checkInterleaved(tile);
        TiffMap map = tile.map();
        if (map.isPlanarSeparated()) {
            throw new UnsupportedTiffFormatException("TIFF YCbCr photometric interpretation is not supported in planar format (separated component planes: TIFF tag PlanarConfiguration=2)");
        }
        int bits = ifd.tryEqualBitDepth().orElse(-1);
        if (bits != 8) {
            throw new UnsupportedTiffFormatException("Cannot unpack YCbCr TIFF image with " + Arrays.toString(ifd.getBitsPerSample()) + " bits per samples: only [8, 8, 8] variant is supported");
        }
        if (map.tileSamplesPerPixel() != 3) {
            throw new TiffException("TIFF YCbCr photometric interpretation requires 3 channels, but there are " + map.tileSamplesPerPixel() + " channels in SamplesPerPixel TIFF tag");
        }
        int sizeX = tile.getSizeX();
        int sizeY = tile.getSizeY();
        int numberOfPixels = tile.getSizeInPixels();
        byte[] unpacked = new byte[3 * numberOfPixels];
        double lumaRed = 0.299;
        double lumaGreen = 0.587;
        double lumaBlue = 0.114;
        int[] reference = ifd.getIntArray(532);
        if (reference == null) {
            reference = new int[]{0, 255, 128, 255, 128, 255};
        }
        int[] subsamplingLog = ifd.getYCbCrSubsamplingLogarithms();
        TagRational[] coefficients = ifd.getValue(529, TagRational[].class).orElse(new TagRational[0]);
        if (coefficients.length >= 3) {
            lumaRed = coefficients[0].doubleValue();
            lumaGreen = coefficients[1].doubleValue();
            lumaBlue = coefficients[2].doubleValue();
        }
        double crCoefficient = 2.0 - 2.0 * lumaRed;
        double cbCoefficient = 2.0 - 2.0 * lumaBlue;
        double lumaGreenInv = 1.0 / lumaGreen;
        int subXLog = subsamplingLog[0];
        int subYLog = subsamplingLog[1];
        int subXMinus1 = (1 << subXLog) - 1;
        int blockLog = subXLog + subYLog;
        int block = 1 << blockLog;
        int numberOfXBlocks = sizeX + subXMinus1 >>> subXLog;
        int i = 0;
        for (int yIndex = 0; yIndex < sizeY; ++yIndex) {
            int yBlockIndex = yIndex >>> subYLog;
            int yAligned = yBlockIndex << subYLog;
            int lineAligned = yBlockIndex * numberOfXBlocks;
            int xIndex = 0;
            while (xIndex < sizeX) {
                int blockIndex = i >>> blockLog;
                int aligned = blockIndex << blockLog;
                int indexInBlock = i - aligned;
                assert (indexInBlock < block);
                int blockIndexTwice = 2 * blockIndex;
                int blockStart = aligned + blockIndexTwice;
                int lumaIndex = blockStart + indexInBlock;
                int chromaIndex = blockStart + block;
                if (chromaIndex + 1 >= data.length) break;
                int yIndexInBlock = indexInBlock >>> subXLog;
                int xIndexInBlock = indexInBlock - (yIndexInBlock << subXLog);
                int resultYIndex = yAligned + yIndexInBlock;
                int resultXIndex = (blockIndex - lineAligned << subXLog) + xIndexInBlock;
                if (resultXIndex < sizeX && resultYIndex < sizeY) {
                    int resultIndex = resultYIndex * sizeX + resultXIndex;
                    int y = (data[lumaIndex] & 0xFF) - reference[0];
                    int cb = (data[chromaIndex] & 0xFF) - reference[2];
                    int cr = (data[chromaIndex + 1] & 0xFF) - reference[4];
                    double red = (double)cr * crCoefficient + (double)y;
                    double blue = (double)cb * cbCoefficient + (double)y;
                    double green = ((double)y - lumaBlue * blue - lumaRed * red) * lumaGreenInv;
                    unpacked[resultIndex] = (byte)TiffUnpacking.toUnsignedByte(red);
                    unpacked[numberOfPixels + resultIndex] = (byte)TiffUnpacking.toUnsignedByte(green);
                    unpacked[2 * numberOfPixels + resultIndex] = (byte)TiffUnpacking.toUnsignedByte(blue);
                }
                ++xIndex;
                ++i;
            }
            while ((i & subXMinus1) != 0) {
                ++i;
            }
        }
        tile.setDecodedData(unpacked);
        tile.setInterleaved(false);
        return true;
    }

    public static boolean unpackTiffBitsAndInvertValues(TiffTile tile, boolean scaleWhenIncreasingBitDepth, boolean correctInvertedBrightness) throws TiffException {
        Objects.requireNonNull(tile);
        tile.checkDecodedData();
        TiffIFD ifd = tile.ifd();
        if (TiffUnpacking.isSimpleRearrangingBytesEnough(ifd, null)) {
            return false;
        }
        if (ifd.isStandardYCbCrNonJpeg()) {
            return false;
        }
        TiffUnpacking.checkInterleaved(tile);
        if (!ifd.isStandardCompression() || ifd.isJpegOrOldJpeg()) {
            throw new IllegalStateException("Corrupted IFD, probably by direct modifications (non-standard/JPEG compression, though it was already checked)");
        }
        TagPhotometricInterpretation photometricInterpretation = ifd.getPhotometricInterpretation();
        if (photometricInterpretation.isIndexed() || photometricInterpretation == TagPhotometricInterpretation.TRANSPARENCY_MASK) {
            scaleWhenIncreasingBitDepth = false;
        }
        boolean invertedBrightness = photometricInterpretation.isInvertedBrightness();
        if (tile.sampleType().isFloatingPoint()) {
            throw new TiffException("Invalid TIFF image: floating-point values, compression \"" + ifd.compressionPrettyName() + "\", photometric interpretation \"" + photometricInterpretation.prettyName() + "\", " + Arrays.toString(ifd.getBitsPerSample()) + " bits per sample");
        }
        boolean invertValues = correctInvertedBrightness && invertedBrightness;
        int sizeX = tile.getSizeX();
        int sizeY = tile.getSizeY();
        assert (tile.getSizeInPixels() == sizeX * sizeY);
        byte[] source = tile.getDecodedData();
        byte[] result = new byte[tile.getSizeInBytes()];
        OptionalInt bytesPerSample = tile.bytesPerSample();
        if (tile.isWholeBytes()) {
            TiffUnpacking.unpackWholeBytesAndInvertValues(ifd, result, source, sizeX, sizeY, tile.samplesPerPixel(), bytesPerSample.orElseThrow(), scaleWhenIncreasingBitDepth, invertValues);
        } else {
            assert (tile.bitsPerSample() == 1) : ">1 bits per sample for non-whole bytes are not supported: " + String.valueOf(tile);
            assert (tile.samplesPerPixel() == 1) : ">1 samples per pixel for non-whole bytes are not supported: " + String.valueOf(tile);
            TiffUnpacking.extractSingleBitsAndInvertValues(result, source, sizeX, sizeY, invertValues);
        }
        tile.setDecodedData(result);
        tile.setInterleaved(false);
        return true;
    }

    private static boolean isSimpleRearrangingBytesEnough(TiffIFD ifd, AtomicBoolean simpleNonJpegFormat) throws TiffException {
        boolean advancedFormat;
        TagCompression compression = ifd.optCompression().orElse(null);
        boolean bl = advancedFormat = compression != null && (!compression.isStandard() || compression.isJpegOrOldJpeg());
        if (simpleNonJpegFormat != null) {
            simpleNonJpegFormat.set(!advancedFormat);
        }
        if (advancedFormat) {
            return true;
        }
        int bits = ifd.tryEqualBitDepthAlignedByBytes().orElse(-1);
        if (bits == -1) {
            return false;
        }
        if (bits != 8 && bits != 16 && bits != 24 && bits != 32 && bits != 64) {
            throw new UnsupportedTiffFormatException("Not supported TIFF format: compression \"" + ifd.compressionPrettyName() + "\", " + bits + " bits per every sample");
        }
        if (ifd.getPhotometricInterpretation() == TagPhotometricInterpretation.Y_CB_CR) {
            return false;
        }
        return !ifd.isStandardInvertedCompression();
    }

    private static void unpackWholeBytesAndInvertValues(TiffIFD ifd, byte[] unpacked, byte[] source, int sizeX, int sizeY, int samplesPerPixel, int bytesPerSample, boolean scaleWhenIncreasingBitDepth, boolean invertValues) throws TiffException {
        if (bytesPerSample > 4) {
            throw new IllegalStateException("Corrupted IFD, probably by direct modifications (" + bytesPerSample + " bytes/sample in tile, though this was already checked)");
        }
        int numberOfPixels = sizeX * sizeY;
        int[] bitsPerSample = ifd.getBitsPerSample();
        boolean byteAligned = Arrays.stream(bitsPerSample).noneMatch(bits -> (bits & 7) != 0);
        if (byteAligned && !ifd.getPhotometricInterpretation().isInvertedBrightness()) {
            throw new IllegalStateException("Corrupted IFD, probably from a parallel thread (BitsPerSample tag is byte-aligned and inversion is not necessary, though it was already checked)");
        }
        if (samplesPerPixel > bitsPerSample.length) {
            throw new IllegalStateException("Corrupted IFD, probably by direct modifications (" + samplesPerPixel + " samples/pixel is greater than the length of BitsPerSample tag; it is possible only for OLD_JPEG, that was already checked)");
        }
        ByteOrder byteOrder = ifd.getByteOrder();
        long[] multipliers = new long[bitsPerSample.length];
        for (int k = 0; k < multipliers.length; ++k) {
            multipliers[k] = ((1L << 8 * bytesPerSample) - 1L) / ((1L << bitsPerSample[k]) - 1L);
        }
        long pos = 0L;
        long length = PackedBitArraysPer8.unpackedLength((byte[])source);
        int i = 0;
        for (int yIndex = 0; yIndex < sizeY; ++yIndex) {
            int xIndex = 0;
            while (xIndex < sizeX) {
                for (int s = 0; s < samplesPerPixel; ++s) {
                    long value;
                    int bits2 = bitsPerSample[s];
                    assert (bits2 <= 32) : "the check \"bytesPerSample > 4\" was not performed!";
                    long maxValue = (1L << bits2) - 1L;
                    int outputIndex = (s * numberOfPixels + i) * bytesPerSample;
                    if (byteAligned) {
                        int index = (i * samplesPerPixel + s) * bytesPerSample;
                        value = JArrays.getBytes8((byte[])source, (int)index, (int)bytesPerSample, (ByteOrder)byteOrder);
                    } else {
                        if (pos >= length) {
                            return;
                        }
                        value = PackedBitArraysPer8.getBits64InReverseOrder((byte[])source, (long)pos, (int)bits2) & 0xFFFFFFFFL;
                        pos += (long)bits2;
                    }
                    if (invertValues) {
                        value = maxValue - value;
                    }
                    if (scaleWhenIncreasingBitDepth) {
                        value *= multipliers[s];
                    }
                    if (outputIndex + bytesPerSample > unpacked.length) {
                        throw new AssertionError((Object)("Out of range for unpacked data at (" + xIndex + ", " + yIndex + ") inside " + sizeX + "x" + sizeY + ", sample #" + s + ": " + outputIndex + "+" + bytesPerSample + ">" + unpacked.length));
                    }
                    JArrays.setBytes8((byte[])unpacked, (int)outputIndex, (long)value, (int)bytesPerSample, (ByteOrder)byteOrder);
                }
                ++xIndex;
                ++i;
            }
            pos = pos + 7L & 0xFFFFFFFFFFFFFFF8L;
        }
    }

    private static void extractSingleBitsAndInvertValues(byte[] unpacked, byte[] source, int sizeX, int sizeY, boolean invertValues) {
        long actual;
        long length = PackedBitArraysPer8.unpackedLength((byte[])source);
        long alignedLine = (long)sizeX + 7L & 0xFFFFFFFFFFFFFFF8L;
        long sOffset = 0L;
        long tOffset = 0L;
        int yIndex = 0;
        while (yIndex < sizeY && (actual = Math.min((long)sizeX, length - sOffset)) > 0L) {
            PackedBitArraysPer8.copyBitsFromReverseToNormalOrderNoSync((byte[])unpacked, (long)tOffset, (byte[])source, (long)sOffset, (long)actual);
            ++yIndex;
            sOffset += alignedLine;
            tOffset += (long)sizeX;
        }
        if (invertValues) {
            PackedBitArraysPer8.notBits((byte[])unpacked, (long)0L, (long)(8L * (long)unpacked.length));
        }
    }

    private static void checkInterleaved(TiffTile tile) {
        if (!tile.isInterleaved()) {
            throw new IllegalArgumentException("Tile data must be interleaved for correct completing to decode " + tile.ifd().compressionPrettyName() + " (separated data are allowed for codecs like JPEG, that must fully decode data themselves, but not for this compression): " + String.valueOf(tile));
        }
    }

    private static int toUnsignedByte(double v) {
        return v < 0.0 ? 0 : (v > 255.0 ? 255 : (int)Math.round(v));
    }

    private static void debugPrintBits(TiffTile tile) throws TiffException {
        if (tile.index().yIndex() != 0) {
            return;
        }
        byte[] data = tile.getDecodedData();
        int sizeX = tile.getSizeX();
        int[] bitsPerSample = tile.ifd().getBitsPerSample();
        int samplesPerPixel = tile.samplesPerPixel();
        System.out.printf("%nPacked bits %s:%n", Arrays.toString(bitsPerSample));
        int bit = 0;
        for (int i = 0; i < sizeX; ++i) {
            System.out.printf("Pixel #%d: ", i);
            for (int s = 0; s < samplesPerPixel; ++s) {
                int bits = bitsPerSample[s];
                int v = 0;
                int j = 0;
                while (j < bits) {
                    int bitIndex = 7 - bit % 8;
                    int b = data[bit / 8] >> bitIndex & 1;
                    System.out.print(b);
                    v |= b << bits - 1 - j;
                    ++j;
                    ++bit;
                }
                System.out.printf(" = %-6d ", v);
            }
            System.out.println();
        }
    }
}

