/*
 * Decompiled with CFR 0.152.
 */
package net.algart.contours;

import java.awt.geom.Point2D;
import java.util.Arrays;
import java.util.Objects;
import java.util.stream.IntStream;
import net.algart.arrays.ArraySorter;
import net.algart.arrays.DirectAccessible;
import net.algart.arrays.IntArray;
import net.algart.arrays.JArrays;
import net.algart.arrays.MutableIntArray;
import net.algart.arrays.SimpleMemoryModel;
import net.algart.arrays.TooLargeArrayException;
import net.algart.contours.ContourHeader;
import net.algart.contours.ContourLength;
import net.algart.contours.InsideContourStatus;
import net.algart.math.Point;
import net.algart.matrices.scanning.Boundary2DScanner;

public final class Contours {
    public static final int BLOCK_LENGTH = 2;
    public static final int MIN_ABSOLUTE_COORDINATE = -1073741824;
    public static final int MAX_ABSOLUTE_COORDINATE = 0x3FFFFFFF;
    private static final long MAX_ABSOLUTE_COORDINATE_HIGH_BITS = Integer.MIN_VALUE;
    public static final int MAX_NUMBER_OF_CONTOURS = 500000000;
    public static final int MAX_CONTOUR_NUMBER_OF_POINTS = 0x3FFFFEFF;
    private static final boolean DEBUG_MODE = false;
    int[] points = JArrays.EMPTY_INTS;
    int pointsLength = 0;
    private int[] contourHeaderOffsets = JArrays.EMPTY_INTS;
    private int numberOfContours = 0;
    private int openedContourHeaderMarker = -1;
    private int openedContourPointsMarker = -1;
    private int touchingMatrixBoundaryFlags = 0;
    private int contourMinX;
    private int contourMaxX;
    private int contourMinY;
    private int contourMaxY;
    private boolean optimizeCollinearSteps = false;

    private Contours() {
    }

    public static Contours newInstance() {
        return new Contours();
    }

    public boolean isOptimizeCollinearSteps() {
        return this.optimizeCollinearSteps;
    }

    public Contours setOptimizeCollinearSteps(boolean optimizeCollinearSteps) {
        this.optimizeCollinearSteps = optimizeCollinearSteps;
        return this;
    }

    public static boolean isSerializedContours(int[] serialized) {
        return Contours.isSerializedContour(serialized, 0);
    }

    public static boolean isSerializedContour(int[] serialized, int offset) {
        return offset < serialized.length && (serialized[offset] & 0xFFFFFF00) == 2135109888;
    }

    public static Contours deserialize(int[] serialized) {
        int p;
        int fullLength;
        Objects.requireNonNull(serialized, "Null serialized");
        Contours r = new Contours();
        r.points = (int[])serialized.clone();
        r.pointsLength = serialized.length;
        ContourHeader checkedHeader = new ContourHeader();
        for (p = 0; p < serialized.length; p += fullLength) {
            if (r.numberOfContours >= 500000000) {
                throw new TooLargeArrayException("Too large number of serialized contours: >500000000");
            }
            int value = r.points[p];
            if (!Contours.isSerializedContour(serialized, p)) {
                throw new IllegalArgumentException("Unsupported version of serialized contour array at position " + p + ", contour #" + r.numberOfContours + ": start contour signature high bits (in value " + value + "=0x" + Integer.toHexString(value) + ") are 0x" + Integer.toHexString(value & 0xFFFFFF00) + " instead of 0x" + Integer.toHexString(2135109888));
            }
            int headerLength = value & 0xFF;
            if (headerLength < 8) {
                throw new IllegalArgumentException("Too short header: " + headerLength + " elements < 8");
            }
            fullLength = r.points[p + 1];
            if (fullLength <= headerLength || (fullLength - headerLength) % 2 != 0) {
                throw new IllegalArgumentException("Serialized contour length must be even and positive, but it = " + fullLength + " - " + headerLength + " (full number of elements - number of header elements) at position " + p);
            }
            if ((long)p + (long)fullLength > (long)serialized.length) {
                throw new IllegalArgumentException("Serialized contour has not enough elements: current position " + p + " + number of contour elements " + fullLength + " > array length" + serialized.length);
            }
            checkedHeader.read(r, p);
            r.addContourHeaderOffset(p);
        }
        assert (p == r.pointsLength) : "the check \"...>serialized.length\" didn't work properly";
        return r;
    }

    public int[] serialize() {
        return Arrays.copyOf(this.points, this.pointsLength);
    }

    public boolean isEmpty() {
        return this.numberOfContours == 0;
    }

    public int numberOfContours() {
        return this.numberOfContours;
    }

    public void clear() {
        this.clear(false);
    }

    public void clear(boolean freeResources) {
        this.checkClosed();
        this.pointsLength = 0;
        this.numberOfContours = 0;
        if (freeResources) {
            this.points = JArrays.EMPTY_INTS;
            this.contourHeaderOffsets = JArrays.EMPTY_INTS;
        }
    }

    public void openContourForAdding(ContourHeader header) {
        if (this.numberOfContours >= 500000000) {
            throw new TooLargeArrayException("Cannot add contour: current number of contours is maximally possible 500000000");
        }
        if (this.pointsLength > 0x7FFFFDFF) {
            throw new TooLargeArrayException("Too large contour array: > 2147483135 elements");
        }
        this.openedContourHeaderMarker = this.pointsLength;
        this.touchingMatrixBoundaryFlags = 0;
        this.contourMinX = Integer.MAX_VALUE;
        this.contourMaxX = Integer.MIN_VALUE;
        this.contourMinY = Integer.MAX_VALUE;
        this.contourMaxY = Integer.MIN_VALUE;
        header.appendToEndForAllocatingSpace(this);
        this.openedContourPointsMarker = this.pointsLength;
    }

    public void addPoint(Boundary2DScanner scanner, long startX, long startY) {
        long x = scanner.x() + (long)scanner.lastStep().increasedPixelVertexX();
        long y = scanner.y() + (long)scanner.lastStep().increasedPixelVertexY();
        this.addPoint(startX + x, startY + y, x == 0L, x == scanner.dimX(), y == 0L, y == scanner.dimY());
    }

    public void addPoint(long x, long y) {
        this.addPoint(x, y, false, false, false, false);
    }

    public void addPoint(long x, long y, boolean xAtMinXMatrixBoundary, boolean xAtMaxXMatrixBoundary, boolean yAtMinYMatrixBoundary, boolean yAtMaxYMatrixBoundary) {
        int length;
        Contours.checkPoint(x, y);
        if (!this.isContourOpened()) {
            throw new IllegalStateException("Cannot add point to closed contour");
        }
        int intX = (int)x;
        int intY = (int)y;
        if (intX < this.contourMinX) {
            this.contourMinX = intX;
        }
        if (intX > this.contourMaxX) {
            this.contourMaxX = intX;
        }
        if (intY < this.contourMinY) {
            this.contourMinY = intY;
        }
        if (intY > this.contourMaxY) {
            this.contourMaxY = intY;
        }
        if (xAtMinXMatrixBoundary) {
            this.touchingMatrixBoundaryFlags |= 0x10;
        }
        if (xAtMaxXMatrixBoundary) {
            this.touchingMatrixBoundaryFlags |= 0x20;
        }
        if (yAtMinYMatrixBoundary) {
            this.touchingMatrixBoundaryFlags |= 0x40;
        }
        if (yAtMaxYMatrixBoundary) {
            this.touchingMatrixBoundaryFlags |= 0x80;
        }
        if ((length = this.pointsLength) < 8) {
            throw new IllegalStateException("Cannot add a point: header was not added yet");
        }
        if (this.optimizeCollinearSteps && (long)length >= (long)this.openedContourPointsMarker + 4L) {
            int previousX = this.points[length - 2];
            int previousY = this.points[length - 1];
            int previousPreviousX = this.points[length - 4];
            int previousPreviousY = this.points[length - 3];
            if (Contours.isReserved(previousX) || Contours.isReserved(previousPreviousX)) {
                throw new IllegalStateException("Damaged contour array: reserved value at position " + (Contours.isReserved(previousX) ? length - 2 : length - 4));
            }
            if (x == (long)previousX && y == (long)previousY) {
                return;
            }
            int previousDx = previousX - previousPreviousX;
            int previousDy = previousY - previousPreviousY;
            int dx = intX - previousX;
            int dy = intY - previousY;
            if (Contours.collinear32AndCodirectional(previousDx, previousDy, dx, dy) || previousDx == 0 && previousDy == 0) {
                this.points[length - 2] = intX;
                this.points[length - 1] = intY;
                return;
            }
        }
        if (this.pointsLength > this.points.length - 2) {
            this.ensureCapacityForPoints((long)this.pointsLength + 2L);
        }
        this.points[this.pointsLength++] = intX;
        this.points[this.pointsLength++] = intY;
    }

    public void closeContour(ContourHeader header) {
        Objects.requireNonNull(header, "Null header");
        if (!this.isContourOpened()) {
            throw new IllegalStateException("Cannot close already closed contour");
        }
        int length = this.pointsLength;
        int p = this.openedContourHeaderMarker;
        int q = this.openedContourPointsMarker;
        this.openedContourHeaderMarker = -1;
        this.openedContourPointsMarker = -1;
        if (q - p != header.headerLength()) {
            throw new IllegalStateException("Header length " + header.headerLength() + " does not match the length of header, previously written in the contour");
        }
        if (!Contours.isReserved(this.points[p])) {
            throw new IllegalStateException("Damaged contour array: no reserved value (signature) at position " + p);
        }
        if (length - q < 2) {
            throw new IllegalStateException("Cannot close empty contour: at least 1 point must be added (empty contours are not allowed");
        }
        int numberOfPoints = length - q >> 1;
        if (numberOfPoints > 0x3FFFFEFF) {
            throw new IllegalStateException("Too large number of points in a contour: it is > 1073741567");
        }
        this.points[p + 1] = length - p;
        this.points[p + 2] = this.contourMinX;
        this.points[p + 2 + 1] = this.contourMaxX;
        this.points[p + 2 + 2] = this.contourMinY;
        this.points[p + 2 + 3] = this.contourMaxY;
        this.points[p + 6] = 0x7FFF0000 | this.touchingMatrixBoundaryFlags | header.getFlagsWithoutContourTouching();
        this.points[p + 7] = header.getObjectLabel();
        this.addContourHeaderOffset(p);
    }

    public boolean isContourOpened() {
        return this.openedContourHeaderMarker >= 0;
    }

    public void addContour(ContourHeader header, IntArray contour) {
        this.addContour(header, contour, false);
    }

    public void addContour(ContourHeader header, IntArray contour, boolean packAddedContour) {
        DirectAccessible da;
        Objects.requireNonNull(header, "Null header");
        Objects.requireNonNull(contour, "Null contour");
        if (contour instanceof DirectAccessible && (da = (DirectAccessible)((Object)contour)).hasJavaArray()) {
            int[] points = (int[])da.javaArray();
            int offset = da.javaArrayOffset();
            int length = da.javaArrayLength();
            this.addContour(header, points, offset, length);
            return;
        }
        this.checkClosed();
        if (this.numberOfContours >= 500000000) {
            throw new TooLargeArrayException("Cannot add contour: current number of contours is maximally possible 500000000");
        }
        int oldLength = this.pointsLength;
        if (packAddedContour) {
            MutableIntArray resultContour = net.algart.arrays.Arrays.SMM.newEmptyIntArray();
            Contours.packContour(resultContour, contour);
            contour = resultContour;
        }
        long contourLength = contour.length();
        Contours.checkContourLength(contourLength);
        header.appendToEnd(this, (int)(contourLength >> 1));
        if (!header.hasContainingRectangle()) {
            Contours.findContainingRectangle(this.points, oldLength + 2, contour);
        }
        this.addContourHeaderOffset(oldLength);
        long newPointsLength = (long)this.pointsLength + contourLength;
        this.ensureCapacityForPoints(newPointsLength);
        contour.getData(0L, this.points, this.pointsLength, (int)contourLength);
        this.pointsLength = (int)newPointsLength;
    }

    public void addContour(ContourHeader header, int[] contour) {
        Objects.requireNonNull(contour, "Null contour");
        this.addContour(header, contour, 0, contour.length);
    }

    public void addContour(ContourHeader header, int[] contour, int contourOffset, int contourLength) {
        Objects.requireNonNull(contour, "Null contour");
        Contours.checkContourLengthAndOffset(contour, contourOffset, contourLength);
        this.checkClosed();
        if (this.numberOfContours >= 500000000) {
            throw new TooLargeArrayException("Cannot add contour: current number of contours is maximally possible 500000000");
        }
        int oldLength = this.pointsLength;
        int headerLength = header.headerLength();
        long occupiedContourLength = (long)headerLength + (long)contourLength;
        long newPointsLength = (long)this.pointsLength + occupiedContourLength;
        this.ensureCapacityForPoints(newPointsLength);
        this.pointsLength = (int)newPointsLength;
        int newPosition = header.write(this, oldLength, contourLength >> 1);
        if (!header.hasContainingRectangle()) {
            Contours.findContainingRectangle(this.points, oldLength + 2, contour, contourOffset, contourLength);
        }
        System.arraycopy(contour, contourOffset, this.points, newPosition, contourLength);
        this.addContourHeaderOffset(oldLength);
    }

    public void addContoursRange(Contours contours, int from, int to) {
        Objects.requireNonNull(contours, "Null contours");
        if (from > to) {
            throw new IllegalArgumentException("from = " + from + " > to = " + to);
        }
        if (from < 0 || to > contours.numberOfContours) {
            throw new IndexOutOfBoundsException("Required contours range " + from + ".." + (to - 1) + " is out of full range 0.." + (contours.numberOfContours - 1));
        }
        this.checkClosed();
        int count = to - from;
        if (count == 0) {
            return;
        }
        int oldNumberOfContours = this.numberOfContours;
        int fromOffset = contours.contourHeaderOffsets[from];
        int toOffset = to == contours.numberOfContours ? contours.pointsLength : contours.contourHeaderOffsets[to];
        int pointsCount = toOffset - fromOffset;
        int offsetIncrement = this.pointsLength - fromOffset;
        this.ensureCapacityForPoints((long)this.pointsLength + (long)pointsCount);
        this.ensureCapacityForHeaderOffsets((long)this.numberOfContours + (long)count);
        System.arraycopy(contours.points, fromOffset, this.points, this.pointsLength, pointsCount);
        this.pointsLength += pointsCount;
        Contours.copyAndIncreaseArray(contours.contourHeaderOffsets, from, this.contourHeaderOffsets, this.numberOfContours, count, offsetIncrement);
        this.numberOfContours += count;
        for (int k = oldNumberOfContours; k < this.numberOfContours; ++k) {
            this.testCorrectness(this.contourHeaderOffsets[k]);
        }
    }

    public void addContour(Contours contours, int contourIndex) {
        Objects.requireNonNull(contours, "Null contours");
        contours.checkContourIndex(contourIndex);
        this.checkClosed();
        int fromOffset = contours.contourHeaderOffsets[contourIndex];
        int nextIndex = contourIndex + 1;
        int toOffset = nextIndex == contours.numberOfContours ? contours.pointsLength : contours.contourHeaderOffsets[nextIndex];
        int pointsCount = toOffset - fromOffset;
        int oldLength = this.pointsLength;
        this.ensureCapacityForPoints((long)this.pointsLength + (long)pointsCount);
        this.addContourHeaderOffset(oldLength);
        System.arraycopy(contours.points, fromOffset, this.points, this.pointsLength, pointsCount);
        this.pointsLength += pointsCount;
    }

    public Contours contoursRange(int from, int to) {
        Contours result = Contours.newInstance();
        result.addContoursRange(this, from, to);
        return result;
    }

    public void addContours(Contours contours) {
        Objects.requireNonNull(contours, "Null contours");
        this.addContoursRange(contours, 0, contours.numberOfContours);
    }

    public void addTransformedContours(Contours contours, double scaleX, double scaleY, double shiftX, double shiftY, boolean removeDegeneratedContours) {
        Objects.requireNonNull(contours, "Null contours");
        ContourHeader header = new ContourHeader();
        ContourLength contourLength = new ContourLength();
        boolean invertOrientation = scaleX < 0.0 != scaleY < 0.0;
        int[] transformed = null;
        int[] packed = null;
        int n = contours.numberOfContours;
        for (int k = 0; k < n; ++k) {
            boolean degeneratedY;
            contours.getHeader(header, k);
            transformed = contours.getContourPointsAndReallocateResult(transformed, contourLength, k);
            int length = contourLength.getArrayLength();
            Contours.transformContour(transformed, 0, length, scaleX, scaleY, shiftX, shiftY);
            packed = Contours.packContourAndReallocateResultUnchecked(packed, contourLength, transformed, 0, length);
            int packedLength = contourLength.getArrayLength();
            header.transformContainingRectangle(scaleX, scaleY, shiftX, shiftY);
            if (invertOrientation) {
                header.setInternalContour(!header.isInternalContour());
            }
            int headerOffset = this.pointsLength;
            this.addContour(header, packed, 0, packedLength);
            if (!removeDegeneratedContours) continue;
            boolean degeneratedX = this.minX(headerOffset) == this.maxX(headerOffset);
            boolean bl = degeneratedY = this.minY(headerOffset) == this.maxY(headerOffset);
            if (degeneratedX && degeneratedY && packedLength != 2) {
                throw new AssertionError((Object)"packContour did not remove duplicated transformed");
            }
            if (!degeneratedX && !degeneratedY) continue;
            this.removeLastContour();
        }
    }

    private void addTransformedContoursSimple(Contours contours, double scaleX, double scaleY, double shiftX, double shiftY, boolean removeDegeneratedContours) {
        Objects.requireNonNull(contours, "Null contours");
        ContourHeader header = new ContourHeader();
        boolean invertOrientation = scaleX < 0.0 != scaleY < 0.0;
        MutableIntArray transformed = net.algart.arrays.Arrays.SMM.newEmptyIntArray();
        MutableIntArray packed = net.algart.arrays.Arrays.SMM.newEmptyIntArray();
        int n = contours.numberOfContours;
        for (int k = 0; k < n; ++k) {
            boolean degeneratedY;
            contours.getHeader(header, k);
            Contours.transformContour(transformed, contours.getContour(k), scaleX, scaleY, shiftX, shiftY);
            Contours.packContour(packed, transformed);
            header.removeContainingRectangle();
            if (invertOrientation) {
                header.setInternalContour(!header.isInternalContour());
            }
            int headerOffset = this.pointsLength;
            this.addContour(header, packed);
            if (!removeDegeneratedContours) continue;
            boolean degeneratedX = this.minX(headerOffset) == this.maxX(headerOffset);
            boolean bl = degeneratedY = this.minY(headerOffset) == this.maxY(headerOffset);
            if (degeneratedX && degeneratedY && packed.length() != 2L) {
                throw new AssertionError((Object)"packContour did not remove duplicated points");
            }
            if (!degeneratedX && !degeneratedY) continue;
            this.removeLastContour();
        }
    }

    public Contours transformContours(double scaleX, double scaleY, double shiftX, double shiftY, boolean removeDegeneratedContours) {
        Contours result = Contours.newInstance();
        result.addTransformedContours(this, scaleX, scaleY, shiftX, shiftY, removeDegeneratedContours);
        return result;
    }

    public void removeContoursRange(int from, int to) {
        int toOffset;
        if (from > to) {
            throw new IllegalArgumentException("from = " + from + " > to = " + to);
        }
        if (from < 0 || to > this.numberOfContours) {
            throw new IndexOutOfBoundsException("Required contours range " + from + ".." + (to - 1) + " is out of full range 0.." + (this.numberOfContours - 1));
        }
        this.checkClosed();
        if (from == to) {
            return;
        }
        int fromOffset = this.contourHeaderOffsets[from];
        int n = toOffset = to == this.numberOfContours ? this.pointsLength : this.contourHeaderOffsets[to];
        assert (fromOffset < toOffset) : from + " < " + to + ", but offsets " + fromOffset + " >= " + toOffset;
        int shiftedContoursCount = this.numberOfContours - to;
        int shiftedPointsCounts = this.pointsLength - toOffset;
        int offsetDecrement = toOffset - fromOffset;
        System.arraycopy(this.contourHeaderOffsets, to, this.contourHeaderOffsets, from, shiftedContoursCount);
        Contours.increaseArray(this.contourHeaderOffsets, from, shiftedContoursCount, -offsetDecrement);
        this.numberOfContours -= to - from;
        System.arraycopy(this.points, toOffset, this.points, fromOffset, shiftedPointsCounts);
        this.pointsLength -= offsetDecrement;
    }

    public void removeSingleContour(int contourIndex) {
        this.checkContourIndex(contourIndex);
        this.removeContoursRange(contourIndex, contourIndex + 1);
    }

    public void removeLastContour() {
        this.checkClosed();
        int lastContourIndex = this.numberOfContours - 1;
        if (lastContourIndex < 0) {
            throw new IllegalStateException("No contours");
        }
        this.pointsLength = this.contourHeaderOffsets[lastContourIndex];
        this.numberOfContours = lastContourIndex;
    }

    private void checkClosed() {
        if (this.isContourOpened()) {
            throw new IllegalStateException("Cannot modify array with opened contour");
        }
    }

    public int getContourNumberOfPoints(int contourIndex) {
        return this.getContourLength(contourIndex) >> 1;
    }

    public int getContourLength(int contourIndex) {
        this.checkContourIndex(contourIndex);
        int p = this.contourHeaderOffsets[contourIndex];
        int headerLength = this.points[p] & 0xFF;
        int fullLength = this.points[p + 1];
        return fullLength - headerLength;
    }

    public int getContourOffset(int contourIndex) {
        this.checkContourIndex(contourIndex);
        int p = this.contourHeaderOffsets[contourIndex];
        this.testCorrectness(p);
        int headerLength = this.points[p] & 0xFF;
        int offset = p + headerLength;
        assert (offset >= 0) : "Damaged contours" + String.valueOf(this);
        return offset;
    }

    public long getContourLengthAndOffset(int contourIndex) {
        this.checkContourIndex(contourIndex);
        int p = this.contourHeaderOffsets[contourIndex];
        int headerLength = this.points[p] & 0xFF;
        int fullLength = this.points[p + 1];
        int offset = p + headerLength;
        return Contours.packLowAndHigh(offset, fullLength - headerLength);
    }

    public static int extractLength(long lengthAndOffset) {
        return (int)(lengthAndOffset >>> 32);
    }

    public static int extractOffset(long lengthAndOffset) {
        return (int)lengthAndOffset;
    }

    public int getSerializedHeaderOffset(int contourIndex) {
        if (contourIndex == this.numberOfContours) {
            return this.pointsLength;
        }
        this.checkContourIndex(contourIndex);
        return this.contourHeaderOffsets[contourIndex];
    }

    public IntArray getContour(int contourIndex) {
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        return SimpleMemoryModel.asUpdatableIntArray(this.points).subArr(offset, length);
    }

    public int getContourPoints(int[] resultPoints, int contourIndex) {
        Objects.requireNonNull(resultPoints, "Null resultPoints");
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        System.arraycopy(this.points, offset, resultPoints, 0, length);
        return length;
    }

    public double getContourPointX(int contourIndex, int pointIndex) {
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        int m = length >> 1;
        if (pointIndex < 0 || pointIndex > m) {
            throw new IndexOutOfBoundsException("Index of contour point " + pointIndex + " is out of range 0.." + (m - 1));
        }
        return this.points[offset + 2 * pointIndex];
    }

    public double getContourPointY(int contourIndex, int pointIndex) {
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        int m = length >> 1;
        if (pointIndex < 0 || pointIndex > m) {
            throw new IndexOutOfBoundsException("Index of contour point " + pointIndex + " is out of range 0.." + (m - 1));
        }
        return this.points[offset + 2 * pointIndex + 1];
    }

    public int[] getContourPointsAndReallocateResult(int[] resultPoints, ContourLength resultLength, int contourIndex) {
        Objects.requireNonNull(resultLength, "Null result length");
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        if (resultPoints == null) {
            resultPoints = new int[Math.max(length, 16)];
        }
        resultPoints = Contours.ensureCapacityForContour(resultPoints, length);
        System.arraycopy(this.points, offset, resultPoints, 0, length);
        resultLength.setNumberOfPoints(length >> 1);
        return resultPoints;
    }

    public MutableIntArray unpackContour(int contourIndex) {
        return this.unpackContour(contourIndex, false);
    }

    public MutableIntArray unpackContour(int contourIndex, boolean processDiagonalSegments) {
        return Contours.unpackContour(this.getContour(contourIndex), processDiagonalSegments);
    }

    public int[] unpackContourAndReallocateResult(int[] resultPoints, ContourLength resultLength, int contourIndex) {
        return this.unpackContourAndReallocateResult(resultPoints, resultLength, contourIndex, false, false);
    }

    public int[] unpackContourAndReallocateResult(int[] resultPoints, ContourLength resultLength, int contourIndex, boolean processDiagonalSegments) {
        return this.unpackContourAndReallocateResult(resultPoints, resultLength, contourIndex, processDiagonalSegments, false);
    }

    public Contours packContours() {
        Contours packedContours = Contours.newInstance();
        ContourHeader header = new ContourHeader();
        int[] packed = null;
        ContourLength contourLength = new ContourLength();
        for (int k = 0; k < this.numberOfContours; ++k) {
            this.getHeader(header, k);
            packed = Contours.packContourAndReallocateResult(packed, contourLength, this.getContour(k));
            packedContours.addContour(header, packed, 0, contourLength.getArrayLength());
        }
        return packedContours;
    }

    public Contours unpackContours() {
        return this.unpackContours(false, null);
    }

    public Contours unpackContours(boolean processDiagonalSegments) {
        return this.unpackContours(processDiagonalSegments, null);
    }

    public Contours unpackContours(boolean processDiagonalSegments, long[] doubledAreas) {
        boolean withArea;
        boolean bl = withArea = doubledAreas != null;
        if (withArea && doubledAreas.length < this.numberOfContours) {
            throw new IllegalArgumentException("Too short doubledAreas array: its length " + doubledAreas.length + " < number of contours = " + this.numberOfContours);
        }
        Contours unpackedContours = Contours.newInstance();
        ContourHeader header = new ContourHeader();
        int[] unpacked = null;
        ContourLength contourLength = new ContourLength();
        for (int k = 0; k < this.numberOfContours; ++k) {
            this.getHeader(header, k);
            unpacked = this.unpackContourAndReallocateResult(unpacked, contourLength, k, processDiagonalSegments, withArea);
            unpackedContours.addContour(header, unpacked, 0, contourLength.getArrayLength());
            if (!withArea) continue;
            doubledAreas[k] = contourLength.doubledArea;
        }
        return unpackedContours;
    }

    public ContourHeader getHeader(int contourIndex) {
        return this.getHeader(null, contourIndex);
    }

    public ContourHeader getHeader(ContourHeader resultHeader, int contourIndex) {
        this.checkContourIndex(contourIndex);
        if (resultHeader == null) {
            resultHeader = new ContourHeader();
        }
        resultHeader.read(this, this.contourHeaderOffsets[contourIndex]);
        return resultHeader;
    }

    public int getObjectLabel(int contourIndex) {
        this.checkContourIndex(contourIndex);
        int p = this.contourHeaderOffsets[contourIndex];
        return this.points[p + 7];
    }

    public void setObjectLabel(int contourIndex, int label) {
        this.checkContourIndex(contourIndex);
        int p = this.contourHeaderOffsets[contourIndex];
        this.points[p + 7] = label;
    }

    public Integer getFrameIdOrNull(int contourIndex) {
        this.checkContourIndex(contourIndex);
        int p = this.contourHeaderOffsets[contourIndex];
        return (this.points[p + 6] & 0x100) != 0 ? Integer.valueOf(this.points[p + 9]) : null;
    }

    public boolean isInternalContour(int contourIndex) {
        this.checkContourIndex(contourIndex);
        int p = this.contourHeaderOffsets[contourIndex];
        return (this.points[p + 6] & 1) != 0;
    }

    public int[] getAllObjectLabels() {
        int[] result = new int[this.numberOfContours];
        for (int k = 0; k < result.length; ++k) {
            result[k] = this.getObjectLabel(k);
        }
        return result;
    }

    public boolean[] getAllInternalContour() {
        boolean[] result = new boolean[this.numberOfContours];
        for (int k = 0; k < result.length; ++k) {
            result[k] = this.isInternalContour(k);
        }
        return result;
    }

    public int[] getAllFrameId(int absentId) {
        int[] result = new int[this.numberOfContours];
        for (int k = 0; k < result.length; ++k) {
            Integer frameId = this.getFrameIdOrNull(k);
            result[k] = frameId == null ? absentId : frameId;
        }
        return result;
    }

    public boolean isPointStrictlyInside(int contourIndex, double x, double y) {
        return this.isPointStrictlyInside(contourIndex, x, y, false);
    }

    public boolean isPointStrictlyInside(int contourIndex, double x, double y, boolean surelyUnpacked) {
        return InsideContourStatus.INSIDE.matchesStatus(this.pointInsideContourInformation(contourIndex, x, y, surelyUnpacked));
    }

    public InsideContourStatus pointInsideContourStatus(int contourIndex, double x, double y, boolean surelyUnpacked) {
        return InsideContourStatus.ofInformation(this.pointInsideContourInformation(contourIndex, x, y, surelyUnpacked));
    }

    public double pointInsideContourInformation(int contourIndex, double x, double y, boolean surelyUnpacked) {
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        return Contours.pointInsideContourInformation(this.points, offset, length, x, y, surelyUnpacked, -1);
    }

    public Point findSomePointInside(int contourIndex) {
        return this.findSomePointInside(contourIndex, false);
    }

    public Point findSomePointInside(int contourIndex, boolean surelyUnpacked) {
        Point2D.Double result = new Point2D.Double();
        if (this.findSomePointInside(result, contourIndex, surelyUnpacked)) {
            return Point.of(((Point2D)result).getX(), ((Point2D)result).getY());
        }
        return null;
    }

    public boolean findSomePointInside(Point2D result, int contourIndex) {
        return this.findSomePointInside(result, contourIndex, false);
    }

    public boolean findSomePointInside(Point2D result, int contourIndex, boolean surelyUnpacked) {
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        return Contours.findSomePointInside(result, this.points, offset, length, surelyUnpacked);
    }

    public double perimeter(int contourIndex) {
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        return Contours.perimeter(this.points, offset, length);
    }

    public double segmentCentersPerimeter(int contourIndex) {
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        return Contours.segmentCentersPerimeter(this.points, offset, length);
    }

    public double area(int contourIndex) {
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        return Contours.area(this.points, offset, length);
    }

    public double segmentCentersArea(int contourIndex) {
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        return Contours.segmentCentersArea(this.points, offset, length);
    }

    public long preciseDoubledArea(int contourIndex) {
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int length = Contours.extractLength(lengthAndOffset);
        return Contours.preciseDoubledArea(this.points, offset, length);
    }

    public void sortIndexesByLabels(int[] indexes) {
        this.sortIndexesByLabels(indexes, false);
    }

    public void sortIndexesByLabels(int[] indexes, boolean internalContoursFirst) {
        Objects.requireNonNull(indexes, "Null indexes");
        ArraySorter.getQuickSorter().sortIndexes(indexes, 0, indexes.length, internalContoursFirst ? (firstIndex, secondIndex) -> {
            int label2;
            int label1 = this.getObjectLabel(firstIndex);
            return label1 < (label2 = this.getObjectLabel(secondIndex)) || label1 == label2 && this.isInternalContour(firstIndex) && !this.isInternalContour(secondIndex);
        } : (firstIndex, secondIndex) -> {
            int label2;
            int label1 = this.getObjectLabel(firstIndex);
            return label1 < (label2 = this.getObjectLabel(secondIndex)) || label1 == label2 && !this.isInternalContour(firstIndex) && this.isInternalContour(secondIndex);
        });
    }

    public void sortIndexesByPreciseArea(int[] indexes, boolean absoluteValueOfArea) {
        this.sortIndexesByPreciseArea(indexes, absoluteValueOfArea, false);
    }

    public void sortIndexesByPreciseArea(int[] indexes, boolean absoluteValueOfArea, boolean greaterAreasFirst) {
        Objects.requireNonNull(indexes, "Null indexes");
        int n = indexes.length;
        long[] areas = new long[n];
        IntStream.range(0, n + 15 >>> 4).parallel().forEach(block -> {
            int i;
            int to = (int)Math.min((long)i + 16L, (long)n);
            for (i = block << 4; i < to; ++i) {
                int index = indexes[i];
                if (index < 0 || index >= this.numberOfContours) {
                    throw new IndexOutOfBoundsException("Index of contour " + index + " (at ths position #" + i + " in the index array) is out of range 0.." + (this.numberOfContours - 1));
                }
                long area = this.preciseDoubledArea(index);
                if (absoluteValueOfArea) {
                    area = Math.abs(area);
                }
                areas[i] = greaterAreasFirst ? -area : area;
            }
        });
        ArraySorter.getQuickSorter().sort(0, n, (i, j) -> areas[i] < areas[j], (firstIndex, secondIndex) -> {
            int tempIndex = indexes[firstIndex];
            indexes[firstIndex] = indexes[secondIndex];
            indexes[secondIndex] = tempIndex;
            long tempArea = areas[firstIndex];
            areas[firstIndex] = areas[secondIndex];
            areas[secondIndex] = tempArea;
        });
    }

    public boolean equalsToSerialized(int[] serialized) {
        if (serialized == null || serialized.length != this.pointsLength) {
            return false;
        }
        return JArrays.arrayEquals(this.points, 0, serialized, 0, this.pointsLength);
    }

    public String toString() {
        return "array of " + this.numberOfContours + " contours (int[" + this.pointsLength + "])";
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Contours contours = (Contours)o;
        return this.pointsLength == contours.pointsLength && this.optimizeCollinearSteps == contours.optimizeCollinearSteps && JArrays.arrayEquals(this.points, 0, contours.points, 0, this.pointsLength);
    }

    public int hashCode() {
        int result = Objects.hash(this.pointsLength, this.optimizeCollinearSteps);
        for (int i = 0; i < this.pointsLength; ++i) {
            result = 31 * result + this.points[i];
        }
        return result;
    }

    public static int getContourNumberOfPoints(IntArray contour) {
        Contours.checkContourLength(contour);
        return (int)contour.length() >> 1;
    }

    public static int getContourPoints(int[] resultPoints, IntArray contour) {
        Objects.requireNonNull(resultPoints, "Null resultPoints");
        Contours.checkContourLength(contour);
        int length = (int)contour.length();
        contour.getData(0L, resultPoints, 0, length);
        return length >> 1;
    }

    public static int[] getContourPointsAndReallocateResult(int[] resultPoints, ContourLength resultLength, IntArray contour) {
        Objects.requireNonNull(resultLength, "Null result length");
        Contours.checkContourLength(contour);
        if (resultPoints == null || resultPoints.length < 16) {
            resultPoints = new int[16];
        }
        int length = (int)contour.length();
        resultPoints = Contours.ensureCapacityForContour(resultPoints, length);
        contour.getData(0L, resultPoints, 0, length);
        resultLength.setNumberOfPoints(length >> 1);
        return resultPoints;
    }

    public static int packContour(MutableIntArray resultContour, IntArray nonOptimizedContour) {
        Objects.requireNonNull(resultContour, "Null result contour");
        Objects.requireNonNull(nonOptimizedContour, "Null non-optimized contour");
        long n = nonOptimizedContour.length();
        Contours.checkContourLength(n);
        n = Contours.removeLastIdenticalPoints(nonOptimizedContour, n);
        resultContour.clear();
        int previousPreviousX = nonOptimizedContour.getInt(0L);
        int previousPreviousY = nonOptimizedContour.getInt(1L);
        Contours.checkPoint(previousPreviousX, previousPreviousY);
        int x0 = previousPreviousX;
        int y0 = previousPreviousY;
        resultContour.pushInt(x0);
        resultContour.pushInt(y0);
        if (n == 2L) {
            return 1;
        }
        assert (n >= 4L);
        long i = Contours.findFirstNotIdenticalPoint(nonOptimizedContour, n, x0, y0);
        int previousX = nonOptimizedContour.getInt(i++);
        int previousY = nonOptimizedContour.getInt(i++);
        assert (previousX != x0 || previousY != y0);
        Contours.checkPoint(previousX, previousY);
        int dx0 = previousX - x0;
        int dy0 = previousY - y0;
        resultContour.pushInt(previousX);
        resultContour.pushInt(previousY);
        assert (i <= n);
        assert (i >= 4L);
        while (i < n) {
            int x = nonOptimizedContour.getInt(i++);
            int y = nonOptimizedContour.getInt(i++);
            if (x == previousX && y == previousY) continue;
            Contours.checkPoint(x, y);
            int previousDx = previousX - previousPreviousX;
            int previousDy = previousY - previousPreviousY;
            int dx = x - previousX;
            int dy = y - previousY;
            if (Contours.collinear32AndCodirectional(previousDx, previousDy, dx, dy)) {
                long length = resultContour.length();
                resultContour.setInt(length - 2L, x);
                resultContour.setInt(length - 1L, y);
            } else {
                resultContour.pushInt(x);
                resultContour.pushInt(y);
                previousPreviousX = previousX;
                previousPreviousY = previousY;
            }
            previousX = x;
            previousY = y;
        }
        assert (resultContour.length() <= n);
        Contours.correctPackedContourWithCollinearFirstAndLastSegments(resultContour, x0, y0, dx0, dy0);
        return (int)(resultContour.length() >> 1);
    }

    public static int[] packContourAndReallocateResult(int[] resultPoints, ContourLength resultLength, IntArray nonOptimizedContour) {
        DirectAccessible da;
        Objects.requireNonNull(resultLength, "Null result length");
        Objects.requireNonNull(nonOptimizedContour, "Null non-optimized contour");
        long n = nonOptimizedContour.length();
        Contours.checkContourLength(n);
        if (nonOptimizedContour instanceof DirectAccessible && (da = (DirectAccessible)((Object)nonOptimizedContour)).hasJavaArray()) {
            int[] points = (int[])da.javaArray();
            int offset = da.javaArrayOffset();
            int length = da.javaArrayLength();
            return Contours.packContourAndReallocateResult(resultPoints, resultLength, points, offset, length);
        }
        n = Contours.removeLastIdenticalPoints(nonOptimizedContour, n);
        if (resultPoints == null || resultPoints.length < 16) {
            resultPoints = new int[16];
        }
        int previousPreviousX = nonOptimizedContour.getInt(0L);
        int previousPreviousY = nonOptimizedContour.getInt(1L);
        Contours.checkPoint(previousPreviousX, previousPreviousY);
        int x0 = previousPreviousX;
        int y0 = previousPreviousY;
        resultPoints[0] = x0;
        resultPoints[1] = y0;
        resultLength.setNumberOfPoints(1);
        if (n == 2L) {
            return resultPoints;
        }
        assert (n >= 4L);
        long i = Contours.findFirstNotIdenticalPoint(nonOptimizedContour, n, x0, y0);
        int previousX = nonOptimizedContour.getInt(i++);
        int previousY = nonOptimizedContour.getInt(i++);
        assert (previousX != x0 || previousY != y0);
        Contours.checkPoint(previousX, previousY);
        int dx0 = previousX - x0;
        int dy0 = previousY - y0;
        resultPoints[2] = previousX;
        resultPoints[3] = previousY;
        resultLength.setNumberOfPoints(2);
        assert (i <= n);
        assert (i >= 4L);
        int length = 4;
        while (i < n) {
            int x = nonOptimizedContour.getInt(i++);
            int y = nonOptimizedContour.getInt(i++);
            if (x == previousX && y == previousY) continue;
            Contours.checkPoint(x, y);
            int previousDx = previousX - previousPreviousX;
            int previousDy = previousY - previousPreviousY;
            int dx = x - previousX;
            int dy = y - previousY;
            if (Contours.collinear32AndCodirectional(previousDx, previousDy, dx, dy)) {
                resultPoints[length - 2] = x;
                resultPoints[length - 1] = y;
            } else {
                if (length > resultPoints.length - 2) {
                    resultPoints = Contours.ensureCapacityForContour(resultPoints, (long)length + 2L);
                }
                resultPoints[length++] = x;
                resultPoints[length++] = y;
                previousPreviousX = previousX;
                previousPreviousY = previousY;
            }
            previousX = x;
            previousY = y;
        }
        assert ((long)length <= n);
        length = Contours.correctPackedContourWithCollinearFirstAndLastSegments(resultPoints, x0, y0, dx0, dy0, length);
        resultLength.setNumberOfPoints(length >> 1);
        return resultPoints;
    }

    public static int[] packContourAndReallocateResult(int[] resultPoints, ContourLength resultLength, int[] nonOptimizedContour, int nonOptimizedOffset, int n) {
        Objects.requireNonNull(resultLength, "Null result length");
        Objects.requireNonNull(nonOptimizedContour, "Null non-optimized contour");
        Contours.checkContourLength(n);
        n = Contours.removeLastIdenticalPoints(nonOptimizedContour, nonOptimizedOffset, n);
        if (nonOptimizedOffset > nonOptimizedContour.length - n) {
            throw new IndexOutOfBoundsException("Offset in array = " + nonOptimizedOffset + ", length since this offset = " + n + ", but array length = " + nonOptimizedContour.length);
        }
        int to = nonOptimizedOffset + n;
        if (resultPoints == null || resultPoints.length < 16) {
            resultPoints = new int[16];
        }
        int x0 = nonOptimizedContour[nonOptimizedOffset];
        int y0 = nonOptimizedContour[nonOptimizedOffset + 1];
        Contours.checkPoint(x0, y0);
        resultPoints[0] = x0;
        resultPoints[1] = y0;
        resultLength.setNumberOfPoints(1);
        if (n == 2) {
            return resultPoints;
        }
        nonOptimizedOffset = Contours.findFirstNotIdenticalPoint(nonOptimizedContour, nonOptimizedOffset, n, x0, y0);
        int previousX = nonOptimizedContour[nonOptimizedOffset++];
        int previousY = nonOptimizedContour[nonOptimizedOffset++];
        assert (previousX != x0 || previousY != y0);
        assert (n >= 4);
        Contours.checkPoint(previousX, previousY);
        int dx0 = previousX - x0;
        int dy0 = previousY - y0;
        int previousDx = dx0;
        int previousDy = dy0;
        int length = 2;
        int resultPointsLengthMinus2 = resultPoints.length - 2;
        assert (nonOptimizedOffset <= to);
        while (nonOptimizedOffset < to) {
            int y;
            int dy;
            int x;
            int dx;
            if (((dx = (x = nonOptimizedContour[nonOptimizedOffset++]) - previousX) | (dy = (y = nonOptimizedContour[nonOptimizedOffset++]) - previousY)) == 0) continue;
            Contours.checkPoint(x, y);
            if (!Contours.collinear32AndCodirectional(previousDx, previousDy, dx, dy)) {
                if (length > resultPointsLengthMinus2) {
                    resultPoints = Contours.ensureCapacityForContour(resultPoints, (long)length + 2L);
                    resultPointsLengthMinus2 = resultPoints.length - 2;
                }
                resultPoints[length++] = previousX;
                resultPoints[length++] = previousY;
                previousDx = dx;
                previousDy = dy;
            }
            previousX = x;
            previousY = y;
        }
        if (length > resultPointsLengthMinus2) {
            resultPoints = Contours.ensureCapacityForContour(resultPoints, (long)length + 2L);
        }
        resultPoints[length++] = previousX;
        resultPoints[length++] = previousY;
        assert (length <= n);
        length = Contours.correctPackedContourWithCollinearFirstAndLastSegments(resultPoints, x0, y0, dx0, dy0, length);
        resultLength.setNumberOfPoints(length >> 1);
        return resultPoints;
    }

    static int[] packContourAndReallocateResultUnchecked(int[] resultPoints, ContourLength resultLength, int[] nonOptimizedContour, int nonOptimizedOffset, int n) {
        Objects.requireNonNull(resultLength, "Null result length");
        Objects.requireNonNull(nonOptimizedContour, "Null non-optimized contour");
        Contours.checkContourLength(n);
        n = Contours.removeLastIdenticalPoints(nonOptimizedContour, nonOptimizedOffset, n);
        if (nonOptimizedOffset > nonOptimizedContour.length - n) {
            throw new IndexOutOfBoundsException("Offset in array = " + nonOptimizedOffset + ", length since this offset = " + n + ", but array length = " + nonOptimizedContour.length);
        }
        int to = nonOptimizedOffset + n;
        if (resultPoints == null || resultPoints.length < 16) {
            resultPoints = new int[16];
        }
        int x0 = nonOptimizedContour[nonOptimizedOffset];
        int y0 = nonOptimizedContour[nonOptimizedOffset + 1];
        resultPoints[0] = x0;
        resultPoints[1] = y0;
        resultLength.setNumberOfPoints(1);
        if (n == 2) {
            return resultPoints;
        }
        nonOptimizedOffset = Contours.findFirstNotIdenticalPoint(nonOptimizedContour, nonOptimizedOffset, n, x0, y0);
        int previousX = nonOptimizedContour[nonOptimizedOffset++];
        int previousY = nonOptimizedContour[nonOptimizedOffset++];
        assert (previousX != x0 || previousY != y0);
        assert (n >= 4);
        int dx0 = previousX - x0;
        int dy0 = previousY - y0;
        int previousDx = dx0;
        int previousDy = dy0;
        int length = 2;
        int resultPointsLengthMinus2 = resultPoints.length - 2;
        assert (nonOptimizedOffset <= to);
        while (nonOptimizedOffset < to) {
            int y;
            int dy;
            int x;
            int dx;
            if (((dx = (x = nonOptimizedContour[nonOptimizedOffset++]) - previousX) | (dy = (y = nonOptimizedContour[nonOptimizedOffset++]) - previousY)) == 0) continue;
            if (!Contours.collinear32AndCodirectional(previousDx, previousDy, dx, dy)) {
                if (length > resultPointsLengthMinus2) {
                    resultPoints = Contours.ensureCapacityForContour(resultPoints, (long)length + 2L);
                    resultPointsLengthMinus2 = resultPoints.length - 2;
                }
                resultPoints[length++] = previousX;
                resultPoints[length++] = previousY;
                previousDx = dx;
                previousDy = dy;
            }
            previousX = x;
            previousY = y;
        }
        if (length > resultPointsLengthMinus2) {
            resultPoints = Contours.ensureCapacityForContour(resultPoints, (long)length + 2L);
        }
        resultPoints[length++] = previousX;
        resultPoints[length++] = previousY;
        assert (length <= n);
        length = Contours.correctPackedContourWithCollinearFirstAndLastSegments(resultPoints, x0, y0, dx0, dy0, length);
        resultLength.setNumberOfPoints(length >> 1);
        return resultPoints;
    }

    public static MutableIntArray unpackContour(IntArray optimizedContour, boolean processDiagonalSegments) {
        MutableIntArray resultContour = net.algart.arrays.Arrays.SMM.newEmptyIntArray();
        Contours.unpackContour(resultContour, optimizedContour, processDiagonalSegments);
        return resultContour;
    }

    public static int unpackContour(MutableIntArray resultContour, IntArray optimizedContour, boolean processDiagonalSegments) {
        Objects.requireNonNull(resultContour, "Null result contour");
        Objects.requireNonNull(optimizedContour, "Null optimized contour");
        long n = optimizedContour.length();
        Contours.checkContourLength(n);
        n = Contours.removeLastIdenticalPoints(optimizedContour, n);
        resultContour.clear();
        int lastX = optimizedContour.getInt(n - 2L);
        int lastY = optimizedContour.getInt(n - 1L);
        Contours.checkPoint(lastX, lastY);
        if (n == 2L) {
            resultContour.pushInt(lastX);
            resultContour.pushInt(lastY);
            return 1;
        }
        for (long i = 0L; i < n; i += 2L) {
            int x = optimizedContour.getInt(i);
            int y = optimizedContour.getInt(i + 1L);
            Contours.checkPoint(x, y);
            if (y == lastY) {
                if (lastX < x) {
                    while (lastX < x) {
                        Contours.checkAbilityToAddToContour(resultContour);
                        resultContour.pushInt(++lastX);
                        resultContour.pushInt(lastY);
                    }
                } else {
                    while (lastX > x) {
                        Contours.checkAbilityToAddToContour(resultContour);
                        resultContour.pushInt(--lastX);
                        resultContour.pushInt(lastY);
                    }
                }
            } else if (x == lastX) {
                if (lastY < y) {
                    while (lastY < y) {
                        Contours.checkAbilityToAddToContour(resultContour);
                        resultContour.pushInt(lastX);
                        resultContour.pushInt(++lastY);
                    }
                } else {
                    while (lastY > y) {
                        Contours.checkAbilityToAddToContour(resultContour);
                        resultContour.pushInt(lastX);
                        resultContour.pushInt(--lastY);
                    }
                }
            } else if (processDiagonalSegments) {
                Contours.addDiagonalExcludingFirst(resultContour, lastX, lastY, x, y);
                lastX = x;
                lastY = y;
            } else {
                throw new IllegalArgumentException("Cannot unpack contour containing non-horizontal and non-vertical segments (" + lastX + "," + lastY + " - " + x + "," + y + ") between points #" + ((i == 0L ? n : i) / 2L - 1L) + " and #" + i / 2L);
            }
            assert (lastX == x);
            assert (lastY == y);
        }
        long resultNumberOfPoints = resultContour.length() >> 1;
        assert (resultNumberOfPoints <= 0x3FFFFEFFL);
        return (int)resultNumberOfPoints;
    }

    public static int[] unpackContourAndReallocateResult(int[] resultPoints, ContourLength resultLength, IntArray optimizedContour, boolean processDiagonalSegments) {
        DirectAccessible da;
        Objects.requireNonNull(resultLength, "Null result length");
        Objects.requireNonNull(optimizedContour, "Null optimized contour");
        long n = optimizedContour.length();
        Contours.checkContourLength(n);
        n = Contours.removeLastIdenticalPoints(optimizedContour, n);
        if (resultPoints == null || resultPoints.length < 16) {
            resultPoints = new int[16];
        }
        int lastX = optimizedContour.getInt(n - 2L);
        int lastY = optimizedContour.getInt(n - 1L);
        Contours.checkPoint(lastX, lastY);
        resultLength.doubledArea = 0L;
        if (n == 2L) {
            resultPoints[0] = lastX;
            resultPoints[1] = lastY;
            resultLength.setNumberOfPoints(1);
            return resultPoints;
        }
        if (optimizedContour instanceof DirectAccessible && (da = (DirectAccessible)((Object)optimizedContour)).hasJavaArray()) {
            int[] points = (int[])da.javaArray();
            int offset = da.javaArrayOffset();
            return Contours.unpackContourAndReallocateResult(resultPoints, resultLength, points, offset, (int)n, lastX, lastY, processDiagonalSegments);
        }
        return Contours.unpackContourAndReallocateResult(resultPoints, resultLength, optimizedContour, n, lastX, lastY, processDiagonalSegments);
    }

    public static MutableIntArray reverseContour(MutableIntArray resultContour, IntArray contour) {
        Objects.requireNonNull(resultContour, "Null result contour");
        Contours.checkContourLength(contour);
        resultContour.clear();
        for (long i = contour.length() - 2L; i >= 0L; i -= 2L) {
            resultContour.pushInt(contour.getInt(i));
            resultContour.pushInt(contour.getInt(i + 1L));
        }
        return resultContour;
    }

    public static MutableIntArray transformContour(MutableIntArray resultContour, IntArray contour, double scaleX, double scaleY, double shiftX, double shiftY) {
        Objects.requireNonNull(resultContour, "Null result contour");
        Contours.checkContourLength(contour);
        resultContour.length(0L);
        long n = contour.length();
        for (long i = 0L; i < n; i += 2L) {
            long x = Math.round(scaleX * (double)contour.getInt(i) + shiftX);
            long y = Math.round(scaleY * (double)contour.getInt(i + 1L) + shiftY);
            Contours.checkPoint(x, y);
            resultContour.pushInt((int)x);
            resultContour.pushInt((int)y);
        }
        return resultContour;
    }

    public static void transformContour(int[] contour, int offset, int n, double scaleX, double scaleY, double shiftX, double shiftY) {
        Contours.checkContourLengthAndOffset(contour, offset, n);
        if (scaleX == 1.0 && scaleY == 1.0 && shiftX == 0.0 && shiftY == 0.0) {
            return;
        }
        IntStream.range(0, n + 255 >>> 8).parallel().forEach(block -> {
            int i;
            int to = (int)Math.min((long)i + 256L, (long)(offset + n));
            for (i = offset + (block << 8); i < to; i += 2) {
                long x = Math.round(scaleX * (double)contour[i] + shiftX);
                long y = Math.round(scaleY * (double)contour[i + 1] + shiftY);
                Contours.checkPoint(x, y);
                contour[i] = (int)x;
                contour[i + 1] = (int)y;
            }
        });
    }

    public static double pointInsideContourInformation(int[] contour, int offset, int n, double x, double y, boolean surelyUnpacked) {
        return Contours.pointInsideContourInformation(contour, offset, n, x, y, surelyUnpacked, -1);
    }

    public static boolean findSomePointInside(Point2D result, int[] contour, int offset, int n, boolean surelyUnpacked) {
        Contours.checkContourLengthAndOffset(contour, offset, n);
        Objects.requireNonNull(result, "Null result point");
        int lastX = contour[offset];
        int lastY = contour[offset + 1];
        result.setLocation(lastX, lastY);
        if (n == 2) {
            return false;
        }
        int offsetTo = offset + n;
        assert (offset < offsetTo);
        for (int p = offset + 2; p < offsetTo; p += 2) {
            int pointX = contour[p];
            int pointY = contour[p + 1];
            if (pointY != lastY) {
                double x = 0.5 * (double)((long)lastX + (long)pointX);
                double y = 0.5 * (double)((long)lastY + (long)pointY);
                double information = Contours.pointInsideContourInformation(contour, offset, n, x, y, surelyUnpacked, p);
                boolean ok = InsideContourStatus.isStrictlyInside(information);
                if (!ok && (ok = InsideContourStatus.isNormalLeftRightBoundary(information))) {
                    double x2 = InsideContourStatus.getBoundarySectionSecondEnd(information);
                    x = 0.5 * (x + x2);
                }
                if (ok) {
                    result.setLocation(x, y);
                    return true;
                }
            }
            lastX = pointX;
            lastY = pointY;
        }
        return false;
    }

    public static double perimeter(IntArray contour) {
        Contours.checkContourLength(contour);
        long n = contour.length();
        int lastX = contour.getInt(n - 2L);
        int lastY = contour.getInt(n - 1L);
        double perimeter = 0.0;
        for (long i = 0L; i < n; i += 2L) {
            int y;
            long dy;
            int x = contour.getInt(i);
            long dx = (long)x - (long)lastX;
            long dSqr = dx * dx + (dy = (long)(y = contour.getInt(i + 1L)) - (long)lastY) * dy;
            perimeter += dSqr <= 1L ? (double)dSqr : Math.sqrt(dSqr);
            lastX = x;
            lastY = y;
        }
        return perimeter;
    }

    public static double perimeter(int[] contour, int offset, int n) {
        int i;
        Contours.checkContourLengthAndOffset(contour, offset, n);
        int lastX = contour[offset + n - 2];
        int lastY = contour[offset + n - 1];
        double perimeter = 0.0;
        int to = i + n;
        for (i = offset; i < to; i += 2) {
            int x = contour[i];
            long dx = (long)x - (long)lastX;
            int y = contour[i + 1];
            long dy = (long)y - (long)lastY;
            long dSqr = dx * dx + dy * dy;
            perimeter += dSqr <= 1L ? (double)dSqr : Math.sqrt(dSqr);
            lastX = x;
            lastY = y;
        }
        return perimeter;
    }

    public static double segmentCentersPerimeter(IntArray contour) {
        Contours.checkContourLength(contour);
        long n = contour.length();
        long x1 = contour.getInt(n - 2L);
        long y1 = contour.getInt(n - 1L);
        long x2 = contour.getInt(0L);
        long y2 = contour.getInt(1L);
        double lastX = 0.5 * (double)(x1 + x2);
        double lastY = 0.5 * (double)(y1 + y2);
        double perimeter = 0.0;
        for (long i = 0L; i < n; i += 2L) {
            x1 = x2;
            y1 = y2;
            long j = i + 2L < n ? i + 2L : 0L;
            x2 = contour.getInt(j);
            y2 = contour.getInt(j + 1L);
            double x = 0.5 * (double)(x1 + x2);
            double y = 0.5 * (double)(y1 + y2);
            double dx = x - lastX;
            double dy = y - lastY;
            double dSqr = dx * dx + dy * dy;
            perimeter += Math.sqrt(dSqr);
            lastX = x;
            lastY = y;
        }
        return perimeter;
    }

    public static double segmentCentersPerimeter(int[] contour, int offset, int n) {
        int i;
        Contours.checkContourLengthAndOffset(contour, offset, n);
        long x1 = contour[offset + n - 2];
        long y1 = contour[offset + n - 1];
        long x2 = contour[offset];
        long y2 = contour[offset + 1];
        double lastX = 0.5 * (double)(x1 + x2);
        double lastY = 0.5 * (double)(y1 + y2);
        double perimeter = 0.0;
        int to = i + n;
        for (i = offset; i < to; i += 2) {
            x1 = x2;
            y1 = y2;
            int j = i + 2 < to ? i + 2 : offset;
            x2 = contour[j];
            y2 = contour[j + 1];
            double x = 0.5 * (double)(x1 + x2);
            double y = 0.5 * (double)(y1 + y2);
            double dx = x - lastX;
            double dy = y - lastY;
            double dSqr = dx * dx + dy * dy;
            perimeter += Math.sqrt(dSqr);
            lastX = x;
            lastY = y;
        }
        return perimeter;
    }

    public static double area(IntArray contour) {
        Contours.checkContourLength(contour);
        long n = contour.length();
        int lastX = contour.getInt(n - 2L);
        int lastY = contour.getInt(n - 1L);
        double area = 0.0;
        for (long i = 0L; i < n; i += 2L) {
            int x = contour.getInt(i);
            int y = contour.getInt(i + 1L);
            area += (double)(((long)x + (long)lastX) * ((long)y - (long)lastY));
            lastX = x;
            lastY = y;
        }
        return 0.5 * area;
    }

    public static double area(int[] contour, int offset, int n) {
        int i;
        Contours.checkContourLengthAndOffset(contour, offset, n);
        int lastX = contour[offset + n - 2];
        int lastY = contour[offset + n - 1];
        double area = 0.0;
        int to = i + n;
        for (i = offset; i < to; i += 2) {
            int x = contour[i];
            int y = contour[i + 1];
            area += (double)(((long)x + (long)lastX) * ((long)y - (long)lastY));
            lastX = x;
            lastY = y;
        }
        return 0.5 * area;
    }

    public static double segmentCentersArea(IntArray contour) {
        Contours.checkContourLength(contour);
        long n = contour.length();
        long x1 = contour.getInt(n - 2L);
        long y1 = contour.getInt(n - 1L);
        long x2 = contour.getInt(0L);
        long y2 = contour.getInt(1L);
        double lastX = 0.5 * (double)(x1 + x2);
        double lastY = 0.5 * (double)(y1 + y2);
        double area = 0.0;
        for (long i = 0L; i < n; i += 2L) {
            x1 = x2;
            y1 = y2;
            long j = i + 2L < n ? i + 2L : 0L;
            x2 = contour.getInt(j);
            y2 = contour.getInt(j + 1L);
            double x = 0.5 * (double)(x1 + x2);
            double y = 0.5 * (double)(y1 + y2);
            area += (x + lastX) * (y - lastY);
            lastX = x;
            lastY = y;
        }
        return 0.5 * area;
    }

    public static double segmentCentersArea(int[] contour, int offset, int n) {
        int i;
        Contours.checkContourLengthAndOffset(contour, offset, n);
        long x1 = contour[offset + n - 2];
        long y1 = contour[offset + n - 1];
        long x2 = contour[offset];
        long y2 = contour[offset + 1];
        double lastX = 0.5 * (double)(x1 + x2);
        double lastY = 0.5 * (double)(y1 + y2);
        double area = 0.0;
        int to = i + n;
        for (i = offset; i < to; i += 2) {
            x1 = x2;
            y1 = y2;
            int j = i + 2 < to ? i + 2 : offset;
            x2 = contour[j];
            y2 = contour[j + 1];
            double x = 0.5 * (double)(x1 + x2);
            double y = 0.5 * (double)(y1 + y2);
            area += (x + lastX) * (y - lastY);
            lastX = x;
            lastY = y;
        }
        return 0.5 * area;
    }

    public static long preciseDoubledArea(int[] contour, int offset, int n) {
        int i;
        Contours.checkContourLengthAndOffset(contour, offset, n);
        int lastX = contour[offset + n - 2];
        int lastY = contour[offset + n - 1];
        long area = 0L;
        int to = i + n;
        for (i = offset; i < to; i += 2) {
            int x = contour[i];
            int y = contour[i + 1];
            area += ((long)x + (long)lastX) * ((long)y - (long)lastY);
            lastX = x;
            lastY = y;
        }
        return area;
    }

    int minX(int headerOffset) {
        return this.points[headerOffset + 2];
    }

    int maxX(int headerOffset) {
        return this.points[headerOffset + 2 + 1];
    }

    int minY(int headerOffset) {
        return this.points[headerOffset + 2 + 2];
    }

    int maxY(int headerOffset) {
        return this.points[headerOffset + 2 + 3];
    }

    void ensureCapacityForPoints(long newPointsLength) {
        if (newPointsLength > (long)this.points.length) {
            if (newPointsLength > Integer.MAX_VALUE) {
                throw new TooLargeArrayException("Too large requested contour array: >=2^31 elements");
            }
            this.points = Arrays.copyOf(this.points, Math.max(16, Math.max((int)newPointsLength, (int)Math.min(Integer.MAX_VALUE, (long)(2.0 * (double)this.points.length)))));
        }
    }

    void checkContourIndex(int k) {
        if (k < 0 || k >= this.numberOfContours) {
            throw new IndexOutOfBoundsException("Index of contour " + k + " is out of range 0.." + (this.numberOfContours - 1));
        }
    }

    private void addContourHeaderOffset(int newHeaderOffset) {
        if (this.numberOfContours >= this.contourHeaderOffsets.length) {
            if (this.numberOfContours >= 500000000) {
                throw new TooLargeArrayException("Cannot add contour: current number of contours is maximally possible 500000000");
            }
            this.ensureCapacityForHeaderOffsets((long)this.numberOfContours + 1L);
        }
        this.contourHeaderOffsets[this.numberOfContours++] = newHeaderOffset;
    }

    private void ensureCapacityForHeaderOffsets(long newNumberOfContours) {
        if (newNumberOfContours > 500000000L) {
            throw new TooLargeArrayException("Too large requested number of contours: > 500000000");
        }
        if (newNumberOfContours > (long)this.contourHeaderOffsets.length) {
            this.contourHeaderOffsets = Arrays.copyOf(this.contourHeaderOffsets, Math.max(16, Math.max((int)newNumberOfContours, (int)Math.min(500000000L, (long)(2.0 * (double)this.contourHeaderOffsets.length)))));
        }
    }

    private void testCorrectness(int headerOffset) {
        assert ((this.points[headerOffset] & 0xFFFFFF00) == 2135109888) : "Contours array " + String.valueOf(this) + " damaged: invalid element 0x" + Integer.toHexString(this.points[headerOffset]) + " at offset " + headerOffset;
    }

    private static void checkContourLength(long length) {
        if (length < 0L) {
            throw new IllegalArgumentException("Negative contour length " + length);
        }
        if (length >> 1 > 0x3FFFFEFFL) {
            throw new IllegalArgumentException("Too large number of points in a contour: it is > 1073741567");
        }
        if (length % 2L != 0L) {
            throw new IllegalArgumentException("Contour length must be even, but it is " + length);
        }
        if (length == 0L) {
            throw new IllegalArgumentException("Empty contour is not allowed");
        }
    }

    private static void checkContourLengthAndOffset(int[] contour, int offset, int length) {
        Objects.requireNonNull(contour, "Null contour");
        if (offset < 0) {
            throw new IllegalArgumentException("Negative contour offset " + offset);
        }
        Contours.checkContourLength(length);
        if (offset > contour.length - length) {
            throw new IndexOutOfBoundsException("Contour offset + length >= contour length = " + contour.length);
        }
    }

    private static void findContainingRectangle(int[] minXMaxXMinYMaxY, int position, IntArray contour) {
        int minY;
        int minX;
        Objects.requireNonNull(minXMaxXMinYMaxY, "Null array for results");
        if (position < 0) {
            throw new IllegalArgumentException("Negative position = " + position);
        }
        if (minXMaxXMinYMaxY.length < position + 4) {
            throw new IllegalArgumentException("Too short result array for minX, maxX, minY, maxY: its length " + minXMaxXMinYMaxY.length + " < " + position + " + 4");
        }
        Contours.checkContourLength(contour);
        long n = contour.length();
        int maxX = minX = contour.getInt(0L);
        int maxY = minY = contour.getInt(1L);
        long disp = 2L;
        while (disp < n) {
            int x = contour.getInt(disp++);
            int y = contour.getInt(disp++);
            if (x < minX) {
                minX = x;
            }
            if (x > maxX) {
                maxX = x;
            }
            if (y < minY) {
                minY = y;
            }
            if (y <= maxY) continue;
            maxY = y;
        }
        minXMaxXMinYMaxY[position++] = minX;
        minXMaxXMinYMaxY[position++] = maxX;
        minXMaxXMinYMaxY[position++] = minY;
        minXMaxXMinYMaxY[position] = maxY;
    }

    private static void findContainingRectangle(int[] minXMaxXMinYMaxY, int position, int[] contour, int contourOffset, int n) {
        int minY;
        int minX;
        Objects.requireNonNull(minXMaxXMinYMaxY, "Null array for results");
        if (position < 0) {
            throw new IllegalArgumentException("Negative position = " + position);
        }
        if (minXMaxXMinYMaxY.length < position + 4) {
            throw new IllegalArgumentException("Too short result array for minX, maxX, minY, maxY: its length " + minXMaxXMinYMaxY.length + " < " + position + " + 4");
        }
        Contours.checkContourLength(n);
        int disp = contourOffset;
        int maxX = minX = contour[disp++];
        int maxY = minY = contour[disp++];
        int to = contourOffset + n;
        while (disp < to) {
            int x = contour[disp++];
            int y = contour[disp++];
            if (x < minX) {
                minX = x;
            }
            if (x > maxX) {
                maxX = x;
            }
            if (y < minY) {
                minY = y;
            }
            if (y <= maxY) continue;
            maxY = y;
        }
        minXMaxXMinYMaxY[position++] = minX;
        minXMaxXMinYMaxY[position++] = maxX;
        minXMaxXMinYMaxY[position++] = minY;
        minXMaxXMinYMaxY[position] = maxY;
    }

    int[] unpackContourAndReallocateResult(int[] resultPoints, ContourLength resultLength, int contourIndex, boolean processDiagonalSegments, boolean calculateArea) {
        Objects.requireNonNull(resultLength, "Null result length");
        if (resultPoints == null || resultPoints.length < 16) {
            resultPoints = new int[16];
        }
        long lengthAndOffset = this.getContourLengthAndOffset(contourIndex);
        int offset = Contours.extractOffset(lengthAndOffset);
        int n = Contours.extractLength(lengthAndOffset);
        n = Contours.removeLastIdenticalPoints(this.points, offset, n);
        int lastX = this.points[offset + n - 2];
        int lastY = this.points[offset + n - 1];
        resultPoints[0] = lastX;
        resultPoints[1] = lastY;
        resultLength.setNumberOfPoints(1);
        resultLength.doubledArea = 0L;
        if (n == 2) {
            return resultPoints;
        }
        if (calculateArea) {
            return Contours.unpackContourAndReallocateResultWithArea(resultPoints, resultLength, this.points, offset, n, lastX, lastY, processDiagonalSegments);
        }
        return Contours.unpackContourAndReallocateResult(resultPoints, resultLength, this.points, offset, n, lastX, lastY, processDiagonalSegments);
    }

    private static int[] unpackContourAndReallocateResultWithArea(int[] resultPoints, ContourLength resultLength, IntArray optimizedContour, long n, int lastX, int lastY, boolean processDiagonalSegments) {
        assert (n >= 2L);
        int length = 0;
        MutableInt mutableLength = new MutableInt();
        long doubledArea = 0L;
        for (long disp = 0L; disp < n; disp += 2L) {
            int x = optimizedContour.getInt(disp);
            int y = optimizedContour.getInt(disp + 1L);
            Contours.checkPoint(x, y);
            if (y == lastY) {
                if (lastX < x) {
                    newLength = ((long)(x - lastX) << 1) + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastX < x) {
                        resultPoints[length++] = ++lastX;
                        resultPoints[length++] = lastY;
                    }
                } else {
                    newLength = ((long)(lastX - x) << 1) + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastX > x) {
                        resultPoints[length++] = --lastX;
                        resultPoints[length++] = lastY;
                    }
                }
            } else if (x == lastX) {
                long doubledDifference = (long)(y - lastY) << 1;
                doubledArea += (long)x * doubledDifference;
                if (lastY < y) {
                    newLength = doubledDifference + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastY < y) {
                        resultPoints[length++] = lastX;
                        resultPoints[length++] = ++lastY;
                    }
                } else {
                    newLength = -doubledDifference + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastY > y) {
                        resultPoints[length++] = lastX;
                        resultPoints[length++] = --lastY;
                    }
                }
            } else if (processDiagonalSegments) {
                mutableLength.value = length;
                doubledArea += ((long)x + (long)lastX) * ((long)y - (long)lastY);
                resultPoints = Contours.addDiagonalExcludingFirst(resultPoints, mutableLength, lastX, lastY, x, y);
                length = mutableLength.value;
                lastX = x;
                lastY = y;
            } else {
                throw new IllegalArgumentException("Cannot unpack contour containing non-horizontal and non-vertical segments (" + lastX + "," + lastY + " - " + x + "," + y + ") between points #" + ((disp == 0L ? n : disp) / 2L - 1L) + " and #" + disp / 2L);
            }
            assert (lastX == x);
            assert (lastY == y);
        }
        resultLength.setNumberOfPoints(length >> 1);
        resultLength.doubledArea = doubledArea;
        return resultPoints;
    }

    private static int[] unpackContourAndReallocateResult(int[] resultPoints, ContourLength resultLength, IntArray optimizedContour, long n, int lastX, int lastY, boolean processDiagonalSegments) {
        assert (n >= 2L);
        int length = 0;
        MutableInt mutableLength = new MutableInt();
        for (long disp = 0L; disp < n; disp += 2L) {
            int x = optimizedContour.getInt(disp);
            int y = optimizedContour.getInt(disp + 1L);
            Contours.checkPoint(x, y);
            if (y == lastY) {
                if (lastX < x) {
                    newLength = ((long)(x - lastX) << 1) + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastX < x) {
                        resultPoints[length++] = ++lastX;
                        resultPoints[length++] = lastY;
                    }
                } else {
                    newLength = ((long)(lastX - x) << 1) + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastX > x) {
                        resultPoints[length++] = --lastX;
                        resultPoints[length++] = lastY;
                    }
                }
            } else if (x == lastX) {
                long doubledDifference = (long)(y - lastY) << 1;
                if (lastY < y) {
                    newLength = doubledDifference + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastY < y) {
                        resultPoints[length++] = lastX;
                        resultPoints[length++] = ++lastY;
                    }
                } else {
                    newLength = -doubledDifference + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastY > y) {
                        resultPoints[length++] = lastX;
                        resultPoints[length++] = --lastY;
                    }
                }
            } else if (processDiagonalSegments) {
                mutableLength.value = length;
                resultPoints = Contours.addDiagonalExcludingFirst(resultPoints, mutableLength, lastX, lastY, x, y);
                length = mutableLength.value;
                lastX = x;
                lastY = y;
            } else {
                throw new IllegalArgumentException("Cannot unpack contour containing non-horizontal and non-vertical segments (" + lastX + "," + lastY + " - " + x + "," + y + ") between points #" + ((disp == 0L ? n : disp) / 2L - 1L) + " and #" + disp / 2L);
            }
            assert (lastX == x);
            assert (lastY == y);
        }
        resultLength.setNumberOfPoints(length >> 1);
        return resultPoints;
    }

    private static int[] unpackContourAndReallocateResultWithArea(int[] resultPoints, ContourLength resultLength, int[] optimizedContour, int optimizedOffset, int n, int lastX, int lastY, boolean processDiagonalSegments) {
        int disp;
        assert (n >= 2);
        int length = 0;
        MutableInt mutableLength = new MutableInt();
        long doubledArea = 0L;
        int to = disp + n;
        for (disp = optimizedOffset; disp < to; disp += 2) {
            int x = optimizedContour[disp];
            int y = optimizedContour[disp + 1];
            Contours.checkPoint(x, y);
            if (y == lastY) {
                if (lastX < x) {
                    newLength = ((long)(x - lastX) << 1) + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastX < x) {
                        resultPoints[length++] = ++lastX;
                        resultPoints[length++] = lastY;
                    }
                } else {
                    newLength = ((long)(lastX - x) << 1) + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastX > x) {
                        resultPoints[length++] = --lastX;
                        resultPoints[length++] = lastY;
                    }
                }
            } else if (x == lastX) {
                long doubledDifference = (long)(y - lastY) << 1;
                doubledArea += (long)x * doubledDifference;
                if (lastY < y) {
                    newLength = doubledDifference + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastY < y) {
                        resultPoints[length++] = lastX;
                        resultPoints[length++] = ++lastY;
                    }
                } else {
                    newLength = -doubledDifference + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastY > y) {
                        resultPoints[length++] = lastX;
                        resultPoints[length++] = --lastY;
                    }
                }
            } else if (processDiagonalSegments) {
                mutableLength.value = length;
                doubledArea += ((long)x + (long)lastX) * ((long)y - (long)lastY);
                resultPoints = Contours.addDiagonalExcludingFirst(resultPoints, mutableLength, lastX, lastY, x, y);
                length = mutableLength.value;
                lastX = x;
                lastY = y;
            } else {
                throw new IllegalArgumentException("Cannot unpack contour containing non-horizontal and non-vertical segments (" + lastX + "," + lastY + " - " + x + "," + y + ") between points #" + ((disp == 0 ? n : disp) / 2 - 1) + " and #" + disp / 2);
            }
            assert (lastX == x);
            assert (lastY == y);
        }
        resultLength.setNumberOfPoints(length >> 1);
        resultLength.doubledArea = doubledArea;
        return resultPoints;
    }

    private static int[] unpackContourAndReallocateResult(int[] resultPoints, ContourLength resultLength, int[] optimizedContour, int optimizedOffset, int n, int lastX, int lastY, boolean processDiagonalSegments) {
        int disp;
        assert (n >= 2);
        int length = 0;
        MutableInt mutableLength = new MutableInt();
        int to = disp + n;
        for (disp = optimizedOffset; disp < to; disp += 2) {
            int x = optimizedContour[disp];
            int y = optimizedContour[disp + 1];
            Contours.checkPoint(x, y);
            if (y == lastY) {
                if (lastX < x) {
                    newLength = ((long)(x - lastX) << 1) + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastX < x) {
                        resultPoints[length++] = ++lastX;
                        resultPoints[length++] = lastY;
                    }
                } else {
                    newLength = ((long)(lastX - x) << 1) + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastX > x) {
                        resultPoints[length++] = --lastX;
                        resultPoints[length++] = lastY;
                    }
                }
            } else if (x == lastX) {
                long doubledDifference = (long)(y - lastY) << 1;
                if (lastY < y) {
                    newLength = doubledDifference + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastY < y) {
                        resultPoints[length++] = lastX;
                        resultPoints[length++] = ++lastY;
                    }
                } else {
                    newLength = -doubledDifference + (long)length;
                    if (newLength > (long)resultPoints.length) {
                        resultPoints = Contours.ensureCapacityForContour(resultPoints, newLength);
                    }
                    while (lastY > y) {
                        resultPoints[length++] = lastX;
                        resultPoints[length++] = --lastY;
                    }
                }
            } else if (processDiagonalSegments) {
                mutableLength.value = length;
                resultPoints = Contours.addDiagonalExcludingFirst(resultPoints, mutableLength, lastX, lastY, x, y);
                length = mutableLength.value;
                lastX = x;
                lastY = y;
            } else {
                throw new IllegalArgumentException("Cannot unpack contour containing non-horizontal and non-vertical segments (" + lastX + "," + lastY + " - " + x + "," + y + ") between points #" + ((disp == 0 ? n : disp) / 2 - 1) + " and #" + disp / 2);
            }
            assert (lastX == x);
            assert (lastY == y);
        }
        resultLength.setNumberOfPoints(length >> 1);
        return resultPoints;
    }

    private static void addDiagonalExcludingFirst(MutableIntArray result, int x1, int y1, int x2, int y2) {
        assert (x1 != x2 && y1 != y2) : "illegal usage for non-diagonal line ())" + x1 + "," + y1 + ") - (" + x2 + "," + y2 + ")";
        int xDifference = x2 - x1;
        int yDifference = y2 - y1;
        int xLengthMinus1 = Math.abs(xDifference);
        int yLengthMinus1 = Math.abs(yDifference);
        int numberOfSegmentsWithLength1 = xLengthMinus1 + yLengthMinus1;
        long savedLength = result.length();
        long newLength = savedLength + 2L * (long)numberOfSegmentsWithLength1;
        Contours.checkAbilityToAddToContour(result, numberOfSegmentsWithLength1);
        if (xLengthMinus1 < yLengthMinus1) {
            double tangent = (double)xDifference / (double)yDifference;
            if (y1 < y2) {
                double dy = 1.0;
                int lastX = x1;
                for (int y = y1 + 1; y < y2; ++y) {
                    int x = x1 + (int)Math.rint(dy * tangent);
                    dy += 1.0;
                    if (x != lastX) {
                        result.pushInt(lastX);
                        result.pushInt(y);
                        lastX = x;
                    }
                    result.pushInt(x);
                    result.pushInt(y);
                }
                if (x2 != lastX) {
                    result.pushInt(lastX);
                    result.pushInt(y2);
                }
            } else {
                double dy = yLengthMinus1 - 1;
                int nextX = x2 + (int)Math.rint(dy * tangent);
                if (x1 != nextX) {
                    result.pushInt(nextX);
                    result.pushInt(y1);
                }
                int y2p1 = y2 + 1;
                for (int y = y1 - 1; y > y2; --y) {
                    int x = nextX;
                    result.pushInt(x);
                    result.pushInt(y);
                    int n = nextX = y == y2p1 ? x2 : x2 + (int)Math.rint((dy -= 1.0) * tangent);
                    if (x == nextX) continue;
                    result.pushInt(nextX);
                    result.pushInt(y);
                }
            }
        } else {
            double tangent = (double)yDifference / (double)xDifference;
            if (x1 < x2) {
                double dx = 1.0;
                int lastY = y1;
                for (int x = x1 + 1; x < x2; ++x) {
                    int y = y1 + (int)Math.rint(dx * tangent);
                    dx += 1.0;
                    if (y != lastY) {
                        result.pushInt(x);
                        result.pushInt(lastY);
                        lastY = y;
                    }
                    result.pushInt(x);
                    result.pushInt(y);
                }
                if (y2 != lastY) {
                    result.pushInt(x2);
                    result.pushInt(lastY);
                }
            } else {
                double dx = xLengthMinus1 - 1;
                int nextY = y2 + (int)Math.rint(dx * tangent);
                if (y1 != nextY) {
                    result.pushInt(x1);
                    result.pushInt(nextY);
                }
                int x2p1 = x2 + 1;
                for (int x = x1 - 1; x > x2; --x) {
                    int y = nextY;
                    result.pushInt(x);
                    result.pushInt(y);
                    int n = nextY = x == x2p1 ? y2 : y2 + (int)Math.rint((dx -= 1.0) * tangent);
                    if (y == nextY) continue;
                    result.pushInt(x);
                    result.pushInt(nextY);
                }
            }
        }
        result.pushInt(x2);
        result.pushInt(y2);
        if (result.length() != newLength) {
            throw new AssertionError((Object)("Illegal result length: " + result.length() + " instead of " + newLength + " = " + savedLength + " + 2 * " + numberOfSegmentsWithLength1));
        }
    }

    private static int[] addDiagonalExcludingFirst(int[] result, MutableInt len, int x1, int y1, int x2, int y2) {
        int savedLength;
        assert (x1 != x2 && y1 != y2) : "illegal usage for non-diagonal line ())" + x1 + "," + y1 + ") - (" + x2 + "," + y2 + ")";
        int xDifference = x2 - x1;
        int yDifference = y2 - y1;
        int xLengthMinus1 = Math.abs(xDifference);
        int yLengthMinus1 = Math.abs(yDifference);
        int numberOfSegmentsWithLength1 = xLengthMinus1 + yLengthMinus1;
        int length = savedLength = len.value;
        long newLength = (long)savedLength + 2L * (long)numberOfSegmentsWithLength1;
        result = Contours.ensureCapacityForContour(result, newLength);
        len.value = (int)newLength;
        if (xLengthMinus1 < yLengthMinus1) {
            double tangent = (double)xDifference / (double)yDifference;
            if (y1 < y2) {
                double dy = 1.0;
                int lastX = x1;
                int y = y1 + 1;
                while (y < y2) {
                    int x = x1 + (int)Math.rint(dy * tangent);
                    dy += 1.0;
                    if (x != lastX) {
                        result[length++] = lastX;
                        result[length++] = y;
                        lastX = x;
                    }
                    result[length++] = x;
                    result[length++] = y++;
                }
                if (x2 != lastX) {
                    result[length++] = lastX;
                    result[length++] = y2;
                }
            } else {
                double dy = yLengthMinus1 - 1;
                int nextX = x2 + (int)Math.rint(dy * tangent);
                if (x1 != nextX) {
                    result[length++] = nextX;
                    result[length++] = y1;
                }
                int y2p1 = y2 + 1;
                for (int y = y1 - 1; y > y2; --y) {
                    int x = nextX;
                    result[length++] = x;
                    result[length++] = y;
                    int n = nextX = y == y2p1 ? x2 : x2 + (int)Math.rint((dy -= 1.0) * tangent);
                    if (x == nextX) continue;
                    result[length++] = nextX;
                    result[length++] = y;
                }
            }
        } else {
            double tangent = (double)yDifference / (double)xDifference;
            if (x1 < x2) {
                double dx = 1.0;
                int lastY = y1;
                int x = x1 + 1;
                while (x < x2) {
                    int y = y1 + (int)Math.rint(dx * tangent);
                    dx += 1.0;
                    if (y != lastY) {
                        result[length++] = x;
                        result[length++] = lastY;
                        lastY = y;
                    }
                    result[length++] = x++;
                    result[length++] = y;
                }
                if (y2 != lastY) {
                    result[length++] = x2;
                    result[length++] = lastY;
                }
            } else {
                double dx = xLengthMinus1 - 1;
                int nextY = y2 + (int)Math.rint(dx * tangent);
                if (y1 != nextY) {
                    result[length++] = x1;
                    result[length++] = nextY;
                }
                int x2p1 = x2 + 1;
                for (int x = x1 - 1; x > x2; --x) {
                    int y = nextY;
                    result[length++] = x;
                    result[length++] = y;
                    int n = nextY = x == x2p1 ? y2 : y2 + (int)Math.rint((dx -= 1.0) * tangent);
                    if (y == nextY) continue;
                    result[length++] = x;
                    result[length++] = nextY;
                }
            }
        }
        result[length++] = x2;
        result[length++] = y2;
        if ((long)length != newLength) {
            throw new AssertionError((Object)("Illegal result length: " + length + " instead of " + newLength + " = " + savedLength + " + 2 * " + numberOfSegmentsWithLength1));
        }
        return result;
    }

    private static long removeLastIdenticalPoints(IntArray nonOptimizedContour, long n) {
        assert (n >= 2L);
        int lastX = nonOptimizedContour.getInt(n - 2L);
        int lastY = nonOptimizedContour.getInt(n - 1L);
        while (n > 2L && nonOptimizedContour.getInt(n - 4L) == lastX && nonOptimizedContour.getInt(n - 3L) == lastY) {
            n -= 2L;
        }
        return n;
    }

    private static int removeLastIdenticalPoints(int[] nonOptimizedContour, int nonOptimizedOffset, int n) {
        assert (n >= 2);
        int lastX = nonOptimizedContour[nonOptimizedOffset + n - 2];
        int lastY = nonOptimizedContour[nonOptimizedOffset + n - 1];
        while (n > 2 && nonOptimizedContour[nonOptimizedOffset + n - 4] == lastX && nonOptimizedContour[nonOptimizedOffset + n - 3] == lastY) {
            n -= 2;
        }
        return n;
    }

    private static long findFirstNotIdenticalPoint(IntArray nonOptimizedContour, long n, int x0, int y0) {
        long i;
        assert (n >= 4L);
        for (i = 2L; i < n && nonOptimizedContour.getInt(i) == x0 && nonOptimizedContour.getInt(i + 1L) == y0; i += 2L) {
        }
        if (i == n) {
            throw new AssertionError((Object)"removeLastIdenticalPoints didn't work properly");
        }
        return i;
    }

    private static int findFirstNotIdenticalPoint(int[] nonOptimizedContour, int nonOptimizedOffset, int n, int x0, int y0) {
        int offset;
        assert (n >= 4);
        int to = nonOptimizedOffset + n;
        for (offset = nonOptimizedOffset + 2; offset < to && nonOptimizedContour[offset] == x0 && nonOptimizedContour[offset + 1] == y0; offset += 2) {
        }
        if (offset == to) {
            throw new AssertionError((Object)"removeLastIdenticalPoints didn't work properly");
        }
        return offset;
    }

    private static void correctPackedContourWithCollinearFirstAndLastSegments(MutableIntArray resultContour, int x0, int y0, int dx0, int dy0) {
        assert (dx0 != 0 || dy0 != 0) : "findFirstNotIdenticalPoint didn't work properly";
        long length = resultContour.length();
        if (length > 4L) {
            int dy;
            int dx;
            int y;
            int previousX = resultContour.getInt(length - 4L);
            int previousY = resultContour.getInt(length - 3L);
            int x = resultContour.getInt(length - 2L);
            if (Contours.collinear32AndCodirectional(x - previousX, (y = resultContour.getInt(length - 1L)) - previousY, dx = x0 - x, dy = y0 - y)) {
                resultContour.length(length - 2L);
                dx = x - previousX;
                dy = y - previousY;
                assert (dx != 0 || dy != 0) : "packContour main loop didn't work properly";
            }
            if (Contours.collinear32AndCodirectional(dx, dy, dx0, dy0)) {
                net.algart.arrays.Arrays.removeRange(resultContour, 0L, 2L);
            }
        }
    }

    private static int correctPackedContourWithCollinearFirstAndLastSegments(int[] resultContour, int x0, int y0, int dx0, int dy0, int length) {
        assert (dx0 != 0 || dy0 != 0) : "findFirstNotIdenticalPoint didn't work properly";
        if (length > 4) {
            int x = resultContour[length - 2];
            int previousX = resultContour[length - 4];
            int y = resultContour[length - 1];
            int previousY = resultContour[length - 3];
            int dx = x0 - x;
            int dy = y0 - y;
            if (Contours.collinear32AndCodirectional(x - previousX, y - previousY, dx, dy)) {
                length -= 2;
                dx = x - previousX;
                dy = y - previousY;
                assert (dx != 0 || dy != 0) : "packContour main loop didn't work properly";
            }
            if (Contours.collinear32AndCodirectional(dx, dy, dx0, dy0)) {
                System.arraycopy(resultContour, 2, resultContour, 0, length -= 2);
            }
        }
        return length;
    }

    private static void increaseArray(int[] array, int offset, int count, int increment) {
        assert (count >= 0);
        if (count == 0) {
            return;
        }
        assert (offset <= array.length - count);
        IntStream.range(0, count + 255 >>> 8).parallel().forEach(block -> {
            int i = offset + (block << 8);
            int to = (int)Math.min((long)i + 256L, (long)(offset + count));
            while (i < to) {
                int n = i++;
                array[n] = array[n] + increment;
            }
        });
    }

    private static void copyAndIncreaseArray(int[] array, int offset, int[] resultArray, int resultOffset, int count, int increment) {
        assert (count >= 0);
        if (count == 0) {
            return;
        }
        assert (offset <= array.length - count);
        assert (resultOffset <= resultArray.length - count);
        IntStream.range(0, count + 255 >>> 8).parallel().forEach(block -> {
            int blockOffset = block << 8;
            int j = resultOffset + blockOffset;
            int i = offset + blockOffset;
            int to = (int)Math.min((long)i + 256L, (long)offset + (long)count);
            while (i < to) {
                resultArray[j] = array[i] + increment;
                ++i;
                ++j;
            }
        });
    }

    static double pointInsideContourInformation(int[] contour, int offset, int n, double x, double y, boolean surelyUnpacked, int pOfSegmentContainingXY) {
        boolean leftEven;
        double degeneratedStatus = Contours.checkDegeneratedCases(contour, offset, n, x, y);
        if (!InsideContourStatus.isFurtherProcessingNecessaryStatus(degeneratedStatus)) {
            return degeneratedStatus;
        }
        assert (n >= 4) : "too little n=" + n + " was not checked";
        int offsetTo = offset + n;
        int intY = (int)StrictMath.ceil(y);
        boolean yIsInteger = (double)intY == y;
        int p = Contours.skipStartingHorizontal(contour, offset, offsetTo, x, yIsInteger, intY);
        if (p < 0) {
            return p == -1 ? InsideContourStatus.makeHorizontalBoundaryStatus() : InsideContourStatus.makeStrictlyOutsideStatus();
        }
        int pOrOffsetTo = p == offset ? offsetTo : p;
        int lastX = contour[pOrOffsetTo - 2];
        int lastY = contour[pOrOffsetTo - 1];
        int countLess = 0;
        int countGreater = 0;
        int countContaining = 0;
        double maxIntersectionLessX = Double.NEGATIVE_INFINITY;
        double minIntersectionGreaterX = Double.POSITIVE_INFINITY;
        int start = p;
        int count = n >> 1;
        do {
            int increment;
            assert (--count >= 0) : "infinite loop at position " + start;
            assert (p < offsetTo);
            if (surelyUnpacked && (increment = Contours.incrementForLongJump(Math.abs(lastY - intY), p, offsetTo, start)) > 0) {
                lastX = contour[(p += increment) - 2];
                lastY = contour[p - 1];
            }
            int pointY = contour[p + 1];
            int pointX = contour[p];
            if (yIsInteger && pointY == intY) {
                boolean contourTouchesHorizontal;
                assert (surelyUnpacked || lastY != intY) : "lastY=intY=" + intY + " is possible only in the beginning, but here cannot be pointY=intY";
                int minX = pointX;
                int maxX = pointX;
                int q = p + 2;
                if (q == offsetTo) {
                    q = offset;
                }
                while (contour[q + 1] == intY) {
                    assert (q != p) : "infinite loop at position " + p;
                    int newX = contour[q];
                    if (newX < minX) {
                        minX = newX;
                    }
                    if (newX > maxX) {
                        maxX = newX;
                    }
                    if ((q += 2) != offsetTo) continue;
                    q = offset;
                }
                boolean checkedXYIsLeftFromHorizontalSegment = x < (double)minX;
                boolean intersectionContainsX = !checkedXYIsLeftFromHorizontalSegment && x <= (double)maxX;
                p = (q == offset ? offsetTo : q) - 2;
                pointX = contour[p];
                assert (contour[p + 1] == pointY);
                int nextY = contour[p + 2 == offsetTo ? offset + 1 : p + 3];
                assert (nextY != intY) : "the loop above didn't skip position " + p;
                boolean bl = contourTouchesHorizontal = lastY < pointY != pointY < nextY;
                if (intersectionContainsX) {
                    if (minX != maxX) {
                        return InsideContourStatus.makeHorizontalBoundaryStatus();
                    }
                    ++countContaining;
                    if (contourTouchesHorizontal) {
                        ++countContaining;
                    }
                } else if (checkedXYIsLeftFromHorizontalSegment) {
                    if (!contourTouchesHorizontal) {
                        ++countGreater;
                    }
                    if ((double)minX < minIntersectionGreaterX) {
                        minIntersectionGreaterX = minX;
                    }
                } else {
                    if (!contourTouchesHorizontal) {
                        ++countLess;
                    }
                    if ((double)maxX > maxIntersectionLessX) {
                        maxIntersectionLessX = maxX;
                    }
                }
            } else if (pointY != lastY) {
                if (p == pOfSegmentContainingXY) {
                    ++countContaining;
                } else if (lastY < pointY ? (double)lastY < y && y < (double)pointY : (double)pointY < y && y < (double)lastY) {
                    double intersectionX = Contours.intersectionX(y, lastX, lastY, pointX, pointY);
                    assert (InsideContourStatus.permittedCoordinate(intersectionX));
                    if (intersectionX == x) {
                        ++countContaining;
                    } else if (x < intersectionX) {
                        ++countGreater;
                        if (intersectionX < minIntersectionGreaterX) {
                            minIntersectionGreaterX = intersectionX;
                        }
                    } else {
                        ++countLess;
                        if (intersectionX > maxIntersectionLessX) {
                            maxIntersectionLessX = intersectionX;
                        }
                    }
                }
            }
            lastX = pointX;
            lastY = pointY;
            if ((p += 2) != offsetTo) continue;
            p = offset;
        } while (p != start);
        if ((countLess + countContaining + countGreater & 1) != 0) {
            String message = "Imbalance of left/right/containing counters (" + countLess + ", " + countGreater + ", " + countContaining + ") at point (" + x + ", " + y + ")";
            if (surelyUnpacked) {
                throw new IllegalStateException(message + "; maybe, the contour is not actually unpacked, though surelyUnpacked argument is true");
            }
            throw new AssertionError((Object)message);
        }
        if (maxIntersectionLessX >= minIntersectionGreaterX) {
            throw new AssertionError((Object)("Invalid maxIntersectionLessX/minIntersectionGreaterX: " + maxIntersectionLessX + " >= " + minIntersectionGreaterX));
        }
        boolean centerEven = (countContaining & 1) == 0;
        boolean bl = leftEven = (countLess & 1) != 0;
        if (centerEven) {
            assert (leftEven == ((countGreater & 1) != 0));
            if (leftEven) {
                return InsideContourStatus.makeStrictlyInsideStatus(maxIntersectionLessX, minIntersectionGreaterX);
            }
            return countContaining == 0 ? InsideContourStatus.makeStrictlyOutsideStatus() : InsideContourStatus.makeDegeneratedLeftRightBoundaryStatus(x);
        }
        if (leftEven) {
            return InsideContourStatus.makeRightBoundaryStatus(maxIntersectionLessX);
        }
        return InsideContourStatus.makeLeftBoundaryStatus(minIntersectionGreaterX);
    }

    private static double checkDegeneratedCases(int[] contour, int offset, int n, double x, double y) {
        Contours.checkContourLengthAndOffset(contour, offset, n);
        assert ((n & 1) == 0);
        assert (n != 0) : "zero length cannot be accepted by checkContourLength()";
        if (x < -2.147483648E9 || x > 2.147483647E9 || y < -2.147483648E9 || y > 2.147483647E9) {
            return InsideContourStatus.makeStrictlyOutsideStatus();
        }
        if (n == 2) {
            return x == (double)contour[offset] && y == (double)contour[offset + 1] ? InsideContourStatus.makeHorizontalBoundaryStatus() : InsideContourStatus.makeStrictlyOutsideStatus();
        }
        return InsideContourStatus.makeFurtherProcessingNecessaryStatus();
    }

    private static int skipStartingHorizontal(int[] contour, int offset, int offsetTo, double x, boolean yIsInteger, int intY) {
        assert (offset >= 0 && offsetTo > offset + 2);
        if (yIsInteger && contour[offset + 1] == intY) {
            int minX;
            int maxX = minX = contour[offset];
            for (int p = offset + 2; p < offsetTo; p += 2) {
                int pointY = contour[p + 1];
                if (pointY != intY) {
                    return p;
                }
                int pointX = contour[p];
                if (pointX < minX) {
                    minX = pointX;
                }
                if (pointX <= maxX) continue;
                maxX = pointX;
            }
            boolean horizontalBoundary = (double)minX <= x && x <= (double)maxX;
            return horizontalBoundary ? -1 : -2;
        }
        return offset;
    }

    private static int incrementForLongJump(int distance, int p, int offsetTo, int start) {
        if (distance > 5) {
            --distance;
            int limit = (start > p ? start : offsetTo) - (p + 2);
            assert (limit >= 0);
            return Math.min(2 * distance, limit);
        }
        return 0;
    }

    private static double intersectionX(double y, int x1, int y1, int x2, int y2) {
        double segmentDx = (long)x2 - (long)x1;
        double segmentDy = (long)y2 - (long)y1;
        return (double)x1 + (y - (double)y1) / segmentDy * segmentDx;
    }

    private static boolean collinear32AndCodirectional(int dx1, int dy1, int dx2, int dy2) {
        return (long)dx1 * (long)dy2 == (long)dy1 * (long)dx2 && (dx1 ^ dx2) >= 0 && (dy1 ^ dy2) >= 0;
    }

    static void checkPoint(long x, long y) {
        if (((x - -1073741824L | y - -1073741824L) & Integer.MIN_VALUE) != 0L) {
            throw new IllegalArgumentException("Point coordinates (" + x + ", " + y + ") are out of allowed range -1073741824..1073741823");
        }
    }

    private static void checkPoint(int x, int y) throws IllegalArgumentException {
        if ((x - -1073741824 | y - -1073741824) < 0) {
            throw new IllegalArgumentException("Point coordinates (" + x + ", " + y + ") are out of allowed range -1073741824..1073741823");
        }
    }

    private static void checkContourLength(IntArray contour) {
        Objects.requireNonNull(contour, "Null contour");
        Contours.checkContourLength(contour.length());
    }

    private static void checkAbilityToAddToContour(MutableIntArray contour) {
        if (contour.length() >> 1 >= 0x3FFFFEFFL) {
            throw new IllegalArgumentException("Too large number of points in a contour: it is > 1073741567");
        }
    }

    private static void checkAbilityToAddToContour(MutableIntArray contour, int numberOfAddedPoints) {
        if (contour.length() >> 1 > (long)(0x3FFFFEFF - numberOfAddedPoints)) {
            throw new IllegalArgumentException("Too large number of points in a contour: it is > 1073741567");
        }
    }

    private static int[] ensureCapacityForContour(int[] points, long newLength) {
        if (newLength <= (long)points.length) {
            return points;
        }
        if (newLength >> 1 >= 0x3FFFFEFFL) {
            throw new IllegalArgumentException("Too large number of points in a contour: it is > 1073741567");
        }
        return Arrays.copyOf(points, Math.max((int)newLength, (int)Math.min(Integer.MAX_VALUE, (long)(2.0 * (double)points.length))));
    }

    private static boolean isReserved(int x) {
        return (x & 0xFF000000) == 0x7F000000;
    }

    private static long packLowAndHigh(int low, int high) {
        return (long)high << 32 | (long)low & 0xFFFFFFFFL;
    }

    private static class MutableInt {
        private int value;

        private MutableInt() {
        }
    }
}

