/*
 * Decompiled with CFR 0.152.
 */
package net.algart.math;

import java.util.Arrays;
import java.util.Collection;
import java.util.Objects;
import java.util.zip.CRC32;
import net.algart.math.IPoint;

public class Point
implements Comparable<Point> {
    private static final Point[] originsCache = new Point[16];
    final double[] coordinates;

    Point(double[] coordinates) {
        this.coordinates = coordinates;
    }

    public static Point of(double ... coordinates) {
        Objects.requireNonNull(coordinates, "Null coordinates argument");
        if (coordinates.length == 0) {
            throw new IllegalArgumentException("Empty coordinates array");
        }
        if (coordinates.length <= originsCache.length) {
            boolean origin = true;
            for (double coord : coordinates) {
                if (coord == 0.0) continue;
                origin = false;
                break;
            }
            if (origin) {
                return originsCache[coordinates.length - 1];
            }
        }
        return new Point((double[])coordinates.clone());
    }

    @Deprecated
    public static Point valueOf(double ... coordinates) {
        return Point.of(coordinates);
    }

    public static Point of(double x) {
        return x == 0.0 ? originsCache[0] : new Point(new double[]{x});
    }

    @Deprecated
    public static Point valueOf(double x) {
        return Point.of(x);
    }

    public static Point of(double x, double y) {
        return x == 0.0 && y == 0.0 ? originsCache[1] : new Point(new double[]{x, y});
    }

    @Deprecated
    public static Point valueOf(double x, double y) {
        return Point.of(x, y);
    }

    public static Point of(double x, double y, double z) {
        return x == 0.0 && y == 0.0 && z == 0.0 ? originsCache[2] : new Point(new double[]{x, y, z});
    }

    @Deprecated
    public static Point valueOf(double x, double y, double z) {
        return Point.of(x, y, z);
    }

    public static Point ofEqualCoordinates(int coordCount, double filler) {
        if (filler == 0.0) {
            return Point.origin(coordCount);
        }
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        double[] coordinates = new double[coordCount];
        Arrays.fill(coordinates, filler);
        return new Point(coordinates);
    }

    @Deprecated
    public static Point valueOfEqualCoordinates(int coordCount, double filler) {
        return Point.ofEqualCoordinates(coordCount, filler);
    }

    public static Point origin(int coordCount) {
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        if (coordCount <= originsCache.length) {
            return originsCache[coordCount - 1];
        }
        return new Point(new double[coordCount]);
    }

    public static Point minValue(int coordCount) {
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        double[] coordinates = new double[coordCount];
        Arrays.fill(coordinates, Double.NEGATIVE_INFINITY);
        return new Point(coordinates);
    }

    public static Point maxValue(int coordCount) {
        if (coordCount <= 0) {
            throw new IllegalArgumentException("Negative or zero number of coordinates: " + coordCount);
        }
        double[] coordinates = new double[coordCount];
        Arrays.fill(coordinates, Double.POSITIVE_INFINITY);
        return new Point(coordinates);
    }

    public static Point of(IPoint iPoint) {
        Objects.requireNonNull(iPoint, "Null iPoint argument");
        double[] coordinates = new double[iPoint.coordCount()];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = iPoint.coord(k);
        }
        return new Point(coordinates);
    }

    @Deprecated
    public static Point valueOf(IPoint iPoint) {
        return Point.of(iPoint);
    }

    public int coordCount() {
        return this.coordinates.length;
    }

    public double[] coordinates() {
        return (double[])this.coordinates.clone();
    }

    public double[] coordinates(double[] result) {
        if (result.length < this.coordinates.length) {
            throw new IllegalArgumentException("Too short result array: double[" + result.length + "]; " + this.coordinates.length + " elements required to store coordinates");
        }
        System.arraycopy(this.coordinates, 0, result, 0, this.coordinates.length);
        return result;
    }

    public double coord(int coordIndex) {
        return this.coordinates[coordIndex];
    }

    public double x() {
        return this.coordinates[0];
    }

    public double y() {
        if (this.coordinates.length < 2) {
            throw new IllegalStateException("Cannot get y-coordinate of " + this.coordinates.length + "-dimensional point");
        }
        return this.coordinates[1];
    }

    public double z() {
        if (this.coordinates.length < 3) {
            throw new IllegalStateException("Cannot get z-coordinate of " + this.coordinates.length + "-dimensional point");
        }
        return this.coordinates[2];
    }

    public boolean isOrigin() {
        for (double coord : this.coordinates) {
            if (coord == 0.0) continue;
            return false;
        }
        return true;
    }

    public double distanceFromOrigin() {
        if (this.coordinates.length == 1) {
            return StrictMath.abs(this.coordinates[0]);
        }
        double dSqr = 0.0;
        for (double coord : this.coordinates) {
            dSqr += coord * coord;
        }
        return StrictMath.sqrt(dSqr);
    }

    public double distanceFrom(Collection<Point> points) {
        Objects.requireNonNull(points, "Null points argument");
        double result = Double.POSITIVE_INFINITY;
        for (Point point : points) {
            double d;
            if (point.coordCount() != this.coordinates.length) {
                throw new IllegalArgumentException("Dimensions count mismatch: some of the passed points has " + point.coordCount() + " dimensions instead of " + this.coordinates.length);
            }
            if (this.coordinates.length == 1) {
                d = StrictMath.abs(this.coordinates[0] - point.coordinates[0]);
            } else if (this.coordinates.length == 2) {
                d = StrictMath.hypot(this.coordinates[0] - point.coordinates[0], this.coordinates[1] - point.coordinates[1]);
            } else {
                double dSqr = 0.0;
                for (int k = 0; k < this.coordinates.length; ++k) {
                    double diff = this.coordinates[k] - point.coordinates[k];
                    dSqr += diff * diff;
                }
                d = StrictMath.sqrt(dSqr);
            }
            if (!(d < result)) continue;
            result = d;
        }
        return result;
    }

    public Point add(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] + point.coordinates[k];
        }
        return new Point(coordinates);
    }

    public Point subtract(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] - point.coordinates[k];
        }
        return new Point(coordinates);
    }

    public Point min(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = Math.min(this.coordinates[k], point.coordinates[k]);
        }
        return new Point(coordinates);
    }

    public Point max(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = Math.max(this.coordinates[k], point.coordinates[k]);
        }
        return new Point(coordinates);
    }

    public Point addToAllCoordinates(double increment) {
        if (increment == 0.0) {
            return this;
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] + increment;
        }
        return new Point(coordinates);
    }

    public Point multiply(double multiplier) {
        if (multiplier == 0.0) {
            return Point.origin(this.coordinates.length);
        }
        if (multiplier == 1.0) {
            return this;
        }
        if (multiplier == -1.0) {
            return this.symmetric();
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = this.coordinates[k] * multiplier;
        }
        return new Point(coordinates);
    }

    public Point scale(double ... multipliers) {
        Objects.requireNonNull(multipliers, "Null multipliers argument");
        if (multipliers.length != this.coordinates.length) {
            throw new IllegalArgumentException("Illegal number of multipliers: " + multipliers.length + " instead of " + this.coordinates.length);
        }
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = multipliers[k] == 0.0 ? 0.0 : (multipliers[k] == 1.0 ? this.coordinates[k] : (multipliers[k] == -1.0 ? -this.coordinates[k] : this.coordinates[k] * multipliers[k]));
        }
        return new Point(coordinates);
    }

    public Point scaleAndShift(double[] multipliers, Point shift) {
        double[] coordinates = new double[this.coordinates.length];
        this.scaleAndShift(coordinates, multipliers, shift);
        return new Point(coordinates);
    }

    public void scaleAndShift(double[] resultCoordinates, double[] multipliers, Point shift) {
        if (resultCoordinates.length < this.coordinates.length) {
            throw new IllegalArgumentException("Too short result coordinates array: double[" + resultCoordinates.length + "]; " + this.coordinates.length + " elements required to store coordinates");
        }
        Objects.requireNonNull(multipliers, "Null multipliers argument");
        if (multipliers.length != this.coordinates.length) {
            throw new IllegalArgumentException("Illegal number of multipliers: " + multipliers.length + " instead of " + this.coordinates.length);
        }
        Objects.requireNonNull(shift, "Null shift argument");
        if (shift.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + shift.coordCount() + " instead of " + this.coordinates.length);
        }
        for (int k = 0; k < this.coordinates.length; ++k) {
            resultCoordinates[k] = multipliers[k] == 0.0 ? shift.coord(k) : (multipliers[k] == 1.0 ? shift.coord(k) + this.coordinates[k] : (multipliers[k] == -1.0 ? shift.coord(k) - this.coordinates[k] : shift.coord(k) + this.coordinates[k] * multipliers[k]));
        }
    }

    public Point shiftAlongAxis(int coordIndex, double shift) {
        this.coord(coordIndex);
        if (shift == 0.0) {
            return this;
        }
        double[] coordinates = (double[])this.coordinates.clone();
        int n = coordIndex;
        coordinates[n] = coordinates[n] + shift;
        return new Point(coordinates);
    }

    public double scalarProduct(Point point) {
        Objects.requireNonNull(point, "Null point argument");
        if (point.coordCount() != this.coordinates.length) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " instead of " + this.coordinates.length);
        }
        double result = 0.0;
        for (int k = 0; k < this.coordinates.length; ++k) {
            result += this.coordinates[k] * point.coordinates[k];
        }
        return result;
    }

    public Point symmetric() {
        double[] coordinates = new double[this.coordinates.length];
        for (int k = 0; k < coordinates.length; ++k) {
            coordinates[k] = -this.coordinates[k];
        }
        return new Point(coordinates);
    }

    public Point projectionAlongAxis(int coordIndex) {
        this.coord(coordIndex);
        if (this.coordinates.length == 1) {
            throw new IllegalStateException("Cannot perform projection of 1-dimensional figures");
        }
        double[] coordinates = new double[this.coordinates.length - 1];
        System.arraycopy(this.coordinates, 0, coordinates, 0, coordIndex);
        System.arraycopy(this.coordinates, coordIndex + 1, coordinates, coordIndex, coordinates.length - coordIndex);
        return new Point(coordinates);
    }

    boolean projectionAlongAxisEquals(int coordIndex, Point point) {
        int k;
        this.coord(coordIndex);
        if (this.coordinates.length == 1) {
            throw new IllegalStateException("Cannot perform projection of 1-dimensional figures");
        }
        if (this.coordinates.length != point.coordCount() + 1) {
            throw new IllegalArgumentException("Dimensions count mismatch: " + point.coordCount() + " is not equal to " + this.coordinates.length + "-1");
        }
        for (k = 0; k < coordIndex; ++k) {
            if (this.coordinates[k] == point.coordinates[k]) continue;
            return false;
        }
        for (k = coordIndex + 1; k < this.coordinates.length; ++k) {
            if (this.coordinates[k] == point.coordinates[k - 1]) continue;
            return false;
        }
        return true;
    }

    @Override
    public int compareTo(Point o) {
        return this.compareTo(o, 0);
    }

    public int compareTo(Point o, int firstCoordIndex) {
        int n = Math.max(this.coordinates.length, o.coordinates.length);
        if (firstCoordIndex < 0) {
            throw new IllegalArgumentException("Negative firstCoordIndex argument");
        }
        firstCoordIndex %= n;
        for (int k = n - 1; k >= 0; --k) {
            double otherCoord;
            int index = k + firstCoordIndex;
            if (index >= n) {
                index -= n;
            }
            double thisCoord = index >= this.coordinates.length ? 0.0 : this.coordinates[index];
            double d = otherCoord = index >= o.coordinates.length ? 0.0 : o.coordinates[index];
            if (thisCoord > otherCoord) {
                return 1;
            }
            if (!(thisCoord < otherCoord)) continue;
            return -1;
        }
        return Integer.compare(this.coordinates.length, o.coordinates.length);
    }

    public String toString() {
        StringBuilder result = new StringBuilder("(");
        for (int k = 0; k < this.coordinates.length; ++k) {
            if (k > 0) {
                result.append(", ");
            }
            result.append(this.coordinates[k]);
        }
        result.append(")");
        return result.toString();
    }

    public int hashCode() {
        CRC32 sum = new CRC32();
        byte[] bytes = new byte[this.coordinates.length * 8];
        Point.getBytes(this.coordinates, bytes);
        sum.update(bytes, 0, bytes.length);
        return (int)sum.getValue() ^ 0x1C11DD;
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof Point)) {
            return false;
        }
        Point p = (Point)obj;
        if (p.coordinates.length != this.coordinates.length) {
            return false;
        }
        for (int k = 0; k < this.coordinates.length; ++k) {
            if (Double.doubleToLongBits(p.coordinates[k]) == Double.doubleToLongBits(this.coordinates[k])) continue;
            return false;
        }
        return true;
    }

    public boolean isInteger() {
        for (double coordinate : this.coordinates) {
            if (coordinate == (double)((long)coordinate)) continue;
            return false;
        }
        return true;
    }

    public IPoint toIntegerPoint() {
        return IPoint.of(this);
    }

    public IPoint toRoundedPoint() {
        return IPoint.roundOf(this);
    }

    private static void getBytes(double[] array, byte[] result) {
        int disp = 0;
        for (int k = 0; k < array.length; ++k) {
            long l = Double.doubleToLongBits(array[k]);
            int value = (int)l ^ 0x2596115B;
            result[disp++] = (byte)value;
            result[disp++] = (byte)(value >>> 8);
            result[disp++] = (byte)(value >>> 16);
            result[disp++] = (byte)(value >>> 24);
            value = (int)(l >>> 32) ^ 0x375AA545;
            result[disp++] = (byte)value;
            result[disp++] = (byte)(value >>> 8);
            result[disp++] = (byte)(value >>> 16);
            result[disp++] = (byte)(value >>> 24);
        }
    }

    static {
        for (int i = 1; i <= originsCache.length; ++i) {
            Point.originsCache[i - 1] = new Point(new double[i]){

                @Override
                public boolean isOrigin() {
                    return true;
                }
            };
        }
    }
}

