/*
 * Decompiled with CFR 0.152.
 */
package net.algart.math.functions;

import java.util.Arrays;
import java.util.Locale;
import java.util.Objects;
import java.util.Random;
import net.algart.math.Point;
import net.algart.math.functions.AbstractCoordinateTransformationOperator;
import net.algart.math.functions.CoordinateTransformationOperator;
import net.algart.math.functions.LinearFunc;
import net.algart.math.functions.LinearOperator;

public class ProjectiveOperator
extends AbstractCoordinateTransformationOperator
implements CoordinateTransformationOperator {
    final double[] a;
    final double[] diagonal;
    final double[] b;
    final double[] c;
    final double d;
    final int n;
    final boolean zeroB;

    ProjectiveOperator(double[] a, double[] diagonal, double[] b, double[] c, double d) {
        assert (b != null);
        if (b.length == 0) {
            throw new IllegalArgumentException("Empty b vector (no coordinates)");
        }
        if (a != null && (long)a.length != (long)b.length * (long)b.length) {
            throw new IllegalArgumentException("Illegal size of A matrix: a.length=" + a.length + " must be equal to b.length^2=" + (long)b.length * (long)b.length);
        }
        if (diagonal != null && diagonal.length != b.length) {
            throw new IllegalArgumentException("b and diagonal vector lengths mismatch: diagonal.length=" + diagonal.length + ", b.length=" + b.length);
        }
        if (c != null && c.length != b.length) {
            throw new IllegalArgumentException("b and c vector lengths mismatch: b.length=" + b.length + ", c.length=" + c.length);
        }
        this.n = b.length;
        if (a != null) {
            int i;
            boolean isDiag = true;
            boolean unitDiag = true;
            int disp = 0;
            for (i = 0; i < this.n; ++i) {
                int j = 0;
                while (j < this.n) {
                    if (i == j) {
                        unitDiag &= a[disp] == 1.0;
                    } else {
                        isDiag &= a[disp] == 0.0;
                    }
                    ++j;
                    ++disp;
                }
            }
            if (isDiag) {
                if (!unitDiag) {
                    diagonal = new double[this.n];
                    i = 0;
                    disp = 0;
                    while (i < this.n) {
                        diagonal[i] = a[disp];
                        ++i;
                        disp += this.n + 1;
                    }
                }
                a = null;
            }
        } else if (diagonal != null) {
            boolean unitDiag = true;
            for (double v : diagonal) {
                unitDiag &= v == 1.0;
            }
            if (unitDiag) {
                diagonal = null;
            }
        }
        this.a = a;
        this.diagonal = diagonal;
        this.b = b;
        boolean zeroC = true;
        if (c != null) {
            for (double v : c) {
                zeroC &= v == 0.0;
            }
        }
        this.c = zeroC ? null : c;
        this.d = d;
        boolean zeroB = true;
        for (double v : b) {
            zeroB &= v == 0.0;
        }
        this.zeroB = zeroB;
    }

    public static ProjectiveOperator getInstance(double[] a, double[] b, double[] c, double d) {
        Objects.requireNonNull(a, "Null A matrix");
        Objects.requireNonNull(b, "Null b vector");
        Objects.requireNonNull(c, "Null c vector");
        if (c.length != b.length) {
            throw new IllegalArgumentException("b and c vector lengths mismatch: b.length=" + b.length + ", c.length=" + c.length);
        }
        if ((long)a.length != (long)b.length * (long)b.length) {
            throw new IllegalArgumentException("Illegal size of A matrix: a.length=" + a.length + " must be equal to b.length^2=" + b.length * b.length);
        }
        boolean zeroC = true;
        for (double v : c) {
            zeroC &= v == 0.0;
        }
        if (zeroC) {
            a = (double[])a.clone();
            b = (double[])b.clone();
            if (d != 1.0) {
                int k = 0;
                while (k < a.length) {
                    int n = k++;
                    a[n] = a[n] / d;
                }
                k = 0;
                while (k < b.length) {
                    int n = k++;
                    b[n] = b[n] / d;
                }
            }
            return new LinearOperator(a, null, b);
        }
        return new ProjectiveOperator((double[])a.clone(), null, (double[])b.clone(), (double[])c.clone(), d);
    }

    public static ProjectiveOperator getInstanceByPoints(Point[] q, Point[] p) {
        Objects.requireNonNull(p, "Null p argument");
        Objects.requireNonNull(q, "Null q argument");
        if (p.length != q.length) {
            throw new IllegalArgumentException("p and q point arrays lengths mismatch: p.length=" + p.length + ", q.length=" + q.length);
        }
        if (p.length == 0) {
            throw new IllegalArgumentException("Empty p and q arrays");
        }
        int n = p.length - 2;
        long numberOfUnknowns = (long)n * (long)(n + 2);
        if (numberOfUnknowns > Integer.MAX_VALUE || numberOfUnknowns * numberOfUnknowns > Integer.MAX_VALUE) {
            throw new OutOfMemoryError("Too large necessary matrix (more than Integer.MAX_VALUE elements)");
        }
        for (int k = 0; k < p.length; ++k) {
            if (p[k].coordCount() != n) {
                throw new IllegalArgumentException("n+2 n-dimensional points are necessary to find the projective operator, but we have " + (n + 2) + " points, and the source point #" + k + " is " + p[k].coordCount() + "-dimensional");
            }
            if (q[k].coordCount() == n) continue;
            throw new IllegalArgumentException("n+2 n-dimensional points are necessary to find the projective operator, but we have " + (n + 2) + " points, and the destination point #" + k + " is " + q[k].coordCount() + "-dimensional");
        }
        int m = (int)numberOfUnknowns;
        double[] s = new double[m * m];
        double[] t = new double[m];
        double[] v = new double[m];
        int k = 0;
        for (int i = 0; i < n; ++i) {
            int pointIndex = 0;
            while (pointIndex < p.length) {
                int j = 0;
                int sOfs = k * m + i * n;
                while (j < n) {
                    s[sOfs] = p[pointIndex].coord(j);
                    ++j;
                    ++sOfs;
                }
                s[k * m + n * n + i] = 1.0;
                j = 0;
                sOfs = k * m + n * n + n;
                while (j < n) {
                    s[sOfs] = -q[pointIndex].coord(i) * p[pointIndex].coord(j);
                    ++j;
                    ++sOfs;
                }
                t[k] = q[pointIndex].coord(i);
                ++pointIndex;
                ++k;
            }
        }
        LinearOperator.solveLinearEquationsSet(v, s, t);
        assert (v.length == m);
        double[] a = new double[n * n];
        double[] b = new double[n];
        double[] c = new double[n];
        System.arraycopy(v, 0, a, 0, n * n);
        System.arraycopy(v, n * n, b, 0, n);
        System.arraycopy(v, n * n + n, c, 0, n);
        return ProjectiveOperator.getInstance(a, b, c, 1.0);
    }

    public final double[] a() {
        if (this.a != null) {
            return (double[])this.a.clone();
        }
        if ((long)this.n * (long)this.n > Integer.MAX_VALUE) {
            throw new OutOfMemoryError("Too large matrix A (more than Integer.MAX_VALUE elements)");
        }
        double[] result = new double[this.n * this.n];
        int i = 0;
        int disp = 0;
        while (i < this.n) {
            result[disp] = this.diagonal == null ? 1.0 : this.diagonal[i];
            ++i;
            disp += this.n + 1;
        }
        return result;
    }

    public final double[] b() {
        return (double[])this.b.clone();
    }

    public final double[] c() {
        return this.c == null ? new double[this.n] : (double[])this.c.clone();
    }

    public final double d() {
        return this.d;
    }

    public final double[] diagonal() {
        if (this.diagonal != null) {
            return (double[])this.diagonal.clone();
        }
        double[] result = new double[this.n];
        if (this.a == null) {
            for (int i = 0; i < this.n; ++i) {
                result[i] = 1.0;
            }
        } else {
            int i = 0;
            int disp = 0;
            while (i < this.n) {
                result[i] = this.a[i];
                ++i;
                disp += this.n + 1;
            }
        }
        return result;
    }

    public final int n() {
        return this.n;
    }

    public final boolean isDiagonal() {
        return this.a == null;
    }

    public final boolean isShift() {
        boolean result;
        boolean bl = result = this.a == null && this.diagonal == null && this.c == null;
        if (result && !(this instanceof LinearOperator)) {
            throw new AssertionError((Object)("Shift operator must be an instance of " + String.valueOf(LinearOperator.class)));
        }
        return result;
    }

    public final boolean isZeroB() {
        return this.zeroB;
    }

    @Override
    public void map(double[] destPoint, double[] srcPoint) {
        this.calculateAxPlusB(destPoint, srcPoint);
        double divisor = this.d;
        if (this.c != null) {
            for (int i = 0; i < this.c.length; ++i) {
                divisor += this.c[i] * srcPoint[i];
            }
        }
        if (divisor != 1.0) {
            double multiplier = 1.0 / divisor;
            int i = 0;
            while (i < destPoint.length) {
                int n = i++;
                destPoint[n] = destPoint[n] * multiplier;
            }
        }
    }

    public String toString() {
        boolean shift = this.isShift();
        StringBuilder sA = new StringBuilder();
        if (this.isDiagonal()) {
            if (this.diagonal != null) {
                sA.append("diag[");
                for (int k = 0; k < this.diagonal.length; ++k) {
                    if (k > 0) {
                        sA.append(",");
                    }
                    sA.append(LinearFunc.goodFormat(this.diagonal[k]));
                }
                sA.append("]");
            }
        } else {
            sA.append("A");
        }
        StringBuilder sB = new StringBuilder();
        for (int k = 0; k < this.n; ++k) {
            if (k > 0) {
                sB.append(",");
            }
            sB.append(LinearFunc.goodFormat(this.b[k]));
        }
        StringBuilder sC = new StringBuilder();
        if (this.c != null) {
            for (int k = 0; k < this.c.length; ++k) {
                if (k > 0) {
                    sC.append(",");
                }
                sC.append(LinearFunc.goodFormat(this.c[k]));
            }
        }
        return "projective " + this.n + "-dimensional operator (" + String.valueOf(sA) + "x+b)/(cx+d), b=(" + String.valueOf(sB) + "), c=(" + String.valueOf(sC) + "), d=" + LinearFunc.goodFormat(this.d) + (shift ? " (shift)" : "");
    }

    @Override
    public int hashCode() {
        int result = this.a != null ? Arrays.hashCode(this.a) : 0;
        result = 37 * result + (this.diagonal != null ? Arrays.hashCode(this.diagonal) : 0);
        result = 37 * result + Arrays.hashCode(this.b);
        result = 37 * result + (this.c != null ? Arrays.hashCode(this.c) : 0);
        result = 37 * result + Double.hashCode(this.d);
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        int k;
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof ProjectiveOperator)) {
            return false;
        }
        ProjectiveOperator po = (ProjectiveOperator)obj;
        if (this.n != po.n) {
            return false;
        }
        if (this.d != po.d) {
            return false;
        }
        if (this.a == null == (po.a != null)) {
            return false;
        }
        if (this.a != null) {
            for (k = 0; k < this.a.length; ++k) {
                if (this.a[k] == po.a[k]) continue;
                return false;
            }
        }
        if (this.diagonal == null == (po.diagonal != null)) {
            return false;
        }
        if (this.diagonal != null) {
            for (k = 0; k < this.diagonal.length; ++k) {
                if (this.diagonal[k] == po.diagonal[k]) continue;
                return false;
            }
        }
        for (k = 0; k < this.b.length; ++k) {
            if (this.b[k] == po.b[k]) continue;
            return false;
        }
        if (this.c == null == (po.c != null)) {
            return false;
        }
        if (this.c != null) {
            for (k = 0; k < this.c.length; ++k) {
                if (this.c[k] == po.c[k]) continue;
                return false;
            }
        }
        return true;
    }

    void calculateAxPlusB(double[] destPoint, double[] srcPoint) {
        Objects.requireNonNull(destPoint, "Null destPoint");
        Objects.requireNonNull(srcPoint, "Null srcPoint");
        if (destPoint.length != this.n) {
            throw new IllegalArgumentException("Illegal length of destPoint array: " + destPoint.length + " for " + String.valueOf(this));
        }
        if (srcPoint.length != this.n) {
            throw new IllegalArgumentException("Illegal length of srcPoint array: " + srcPoint.length + " for " + String.valueOf(this));
        }
        if (this.a != null) {
            System.arraycopy(this.b, 0, destPoint, 0, destPoint.length);
            int disp = 0;
            for (int i = 0; i < this.n; ++i) {
                double sum = 0.0;
                int j = 0;
                while (j < this.n) {
                    sum += this.a[disp] * srcPoint[j];
                    ++j;
                    ++disp;
                }
                destPoint[i] = sum + this.b[i];
            }
        } else if (this.diagonal != null) {
            for (int i = 0; i < this.n; ++i) {
                destPoint[i] = this.diagonal[i] * srcPoint[i] + this.b[i];
            }
        } else {
            for (int i = 0; i < this.n; ++i) {
                destPoint[i] = srcPoint[i] + this.b[i];
            }
        }
    }

    static class Test {
        boolean verbose = false;
        int dimCount;
        int numberOfTests;
        long startSeed;
        Random rnd;
        Point[] p;
        Point[] q;
        Point[] r;

        Test() {
        }

        final void init(String[] args) {
            int startArgIndex = 0;
            if (startArgIndex < args.length && args[startArgIndex].equalsIgnoreCase("-v")) {
                this.verbose = true;
                ++startArgIndex;
            }
            if (args.length < startArgIndex + 2) {
                System.out.println("Usage: " + Test.class.getName() + " [-v] dimCount numberOfTests [randSeed]");
                System.exit(0);
            }
            this.dimCount = Integer.parseInt(args[startArgIndex]);
            this.numberOfTests = Integer.parseInt(args[startArgIndex]);
            this.startSeed = args.length < startArgIndex + 3 ? new Random().nextLong() : Long.parseLong(args[startArgIndex + 2]);
            this.rnd = new Random(this.startSeed);
            System.out.printf(Locale.US, "%d tests, randSeed = %d%n", this.numberOfTests, this.startSeed);
        }

        final void mainTest() {
            long t1 = System.nanoTime();
            double maxError = 0.0;
            for (int testCount = 1; testCount <= this.numberOfTests; ++testCount) {
                CoordinateTransformationOperator o2;
                this.newRndPoints(this.numberOfPoints(this.dimCount), this.dimCount);
                CoordinateTransformationOperator o = this.getOperator();
                this.mapPoints(o);
                double error = this.maxDiff();
                maxError = Math.max(maxError, error);
                if (this.verbose) {
                    System.out.println(testCount + ": difference " + error + "; operator hash code " + o.hashCode());
                }
                if (error > 0.001) {
                    System.err.println(testCount + ": difference " + error + " is BIG: " + String.valueOf(o) + " incorrectly maps " + String.valueOf(Arrays.asList(this.p)) + " to " + String.valueOf(Arrays.asList(this.r)) + " instead of " + String.valueOf(Arrays.asList(this.q)));
                }
                if (!o.equals(o2 = this.getOperator())) {
                    throw new AssertionError((Object)"Error in equals");
                }
                if (o2.hashCode() != o.hashCode()) {
                    throw new AssertionError((Object)"Error in hashCode");
                }
            }
            long t2 = System.nanoTime();
            System.out.printf(Locale.US, "All tests done in %.3f seconds (%.2f mcs/test), maximal error = %g, randSeed = %d%n", (double)(t2 - t1) * 1.0E-9, (double)(t2 - t1) * 0.001 / (double)this.numberOfTests, maxError, this.startSeed);
        }

        final void newRndPoints(int numberOfPoints, int dimCount) {
            this.p = new Point[numberOfPoints];
            this.q = new Point[numberOfPoints];
            this.r = new Point[numberOfPoints];
            double[] coordinates = new double[dimCount];
            for (int k = 0; k < this.p.length; ++k) {
                int j;
                for (j = 0; j < dimCount; ++j) {
                    coordinates[j] = this.rnd.nextDouble() - 0.5;
                }
                this.p[k] = Point.of(coordinates);
                for (j = 0; j < dimCount; ++j) {
                    coordinates[j] = this.rnd.nextDouble() - 0.5;
                }
                this.q[k] = Point.of(coordinates);
            }
        }

        final void mapPoints(CoordinateTransformationOperator o) {
            for (int k = 0; k < this.p.length; ++k) {
                double[] srcPoint = this.p[k].coordinates();
                double[] destPoint = new double[srcPoint.length];
                o.map(destPoint, srcPoint);
                this.r[k] = Point.of(destPoint);
            }
        }

        final double maxDiff() {
            double result = 0.0;
            for (int k = 0; k < this.p.length; ++k) {
                result = Math.max(result, this.q[k].subtract(this.r[k]).distanceFromOrigin());
            }
            return result;
        }

        int numberOfPoints(int dimCount) {
            return dimCount + 2;
        }

        CoordinateTransformationOperator getOperator() {
            return ProjectiveOperator.getInstanceByPoints(this.q, this.p);
        }

        public static void main(String[] args) {
            Test test = new Test();
            test.init(args);
            test.mainTest();
        }
    }
}

